<?php

namespace Drupal\purge_users\Plugin\Condition;

use Drupal\Core\Form\FormStateInterface;
use Drupal\purge_users\PurgeUsersConditionBase;

/**
 * Purge users that have not logged in for some amount of time.
 *
 * @Condition(
 *   id = "purge_users:not_logged_in",
 *   label = @Translation("Users not logged in for some time"),
 *   description = @Translation("Purge users that have not logged in for some amount of time."),
 *   status_related = TRUE,
 *   context_definitions = {
 *     "context" = @ContextDefinition("map", required = TRUE)
 *   }
 * )
 */
class NotLoggedInCondition extends PurgeUsersConditionBase {

  use PolicyConditionTrait;

  /**
   * {@inheritdoc}
   */
  public function evaluate() {
    $context = $this->getContextValue('context');
    if (!empty($context)) {
      $config = $this->getConfiguration();
      $login_date = strtotime('-' . $config['user_lastlogin_value'] . ' ' . $config['user_lastlogin_period']);
      $mode = $context['mode'];
      if ($mode === 'notify') {
        $policy = $context['policy'];
        $login_date = $this->addPolicyNotificationDelay($policy, $login_date);
      }
      $query = $context['query'];
      $query->condition('u.login', $login_date, '<');
      $query->condition('u.login', 0, '!=');
    }
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'user_lastlogin_value' => 0,
      'user_lastlogin_period' => 'days',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    // Get configuration.
    $config = $this->getConfiguration();

    $form['user_lastlogin_value'] = [
      '#type' => 'number',
      '#title' => $this->t('Value'),
      '#default_value' => $config['user_lastlogin_value'],
      '#prefix' => '<div class="purge-interval-selector clearfix">',
      '#attributes' => ['class' => ['purge-value']],
    ];

    $form['user_lastlogin_period'] = [
      '#title' => $this->t('Period'),
      '#type' => 'select',
      '#options' => [
        'days' => $this->t('Days'),
        'month' => $this->t('Months'),
        'year' => $this->t('Year'),
      ],
      '#default_value' => $config['user_lastlogin_period'],
      '#attributes' => ['class' => ['purge-period']],
      '#suffix' => '</div>',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    $login_value = (int) $form_state->getValue(['user_lastlogin_value']);
    $login_period = $form_state->getValue(['user_lastlogin_period']);
    // Validate to set the purge period to more than 1 day.
    if ($login_period === 'days' && $login_value < 1) {
      $form_state->setErrorByName('Period limit', $this->t('Purge period should be more than 1 day.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->configuration['user_lastlogin_value'] = $form_state->getValue('user_lastlogin_value');
    $this->configuration['user_lastlogin_period'] = $form_state->getValue('user_lastlogin_period');
  }

  /**
   * {@inheritdoc}
   */
  public function summary() {
    return $this->t(
      'Purge users that have not logged in for @value @period.',
      [
        '@value' => $this->configuration['user_lastlogin_value'],
        '@period' => $this->configuration['user_lastlogin_period'],
      ]
    );
  }

}
