<?php

namespace Drupal\purge_users\Plugin\Condition;

use Drupal\Core\Form\FormStateInterface;
use Drupal\purge_users\PurgeUsersConditionBase;

/**
 * Do not purge users who have not been properly notified beforehand.
 *
 * @Condition(
 *   id = "purge_users:notification_required",
 *   label = @Translation("Users pre-notified only"),
 *   description = @Translation("Do not purge users who have not been properly notified beforehand."),
 *   context_definitions = {
 *     "context" = @ContextDefinition("map", required = TRUE)
 *   }
 * )
 */
class NotificationRequiredCondition extends PurgeUsersConditionBase {

  use PolicyConditionTrait;

  /**
   * {@inheritdoc}
   */
  public function evaluate() {
    $context = $this->getContextValue('context');
    if (!empty($context) && ($context['mode'] === 'purge')) {
      $config = $this->getConfiguration();
      if ($config['notification_required']) {
        // Ensure that users have been properly notified.
        $policy = $context['policy'];
        $notification_delay = $this->subtractPolicyNotificationDelay($policy);
        /** @var \Drupal\Core\Database\Query\SelectInterface $query */
        $query = $context['query'];
        $query->innerJoin('purge_users_notifications', 'pun', 'pun.uid = u.uid AND pun.timestamp < :delay', [':delay' => $notification_delay]);
      }
    }
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'notification_required' => FALSE,
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    // Get configuration.
    $config = $this->getConfiguration();

    $form['notification_required'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Delete only users who have been properly notified'),
      '#default_value' => $config['notification_required'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->configuration['notification_required'] = $form_state->getValue('notification_required');
  }

  /**
   * {@inheritdoc}
   */
  public function summary() {
    if ($this->configuration['notification_required']) {
      return $this->t('Do not purge users who have not been properly notified beforehand.');
    }
    else {
      return $this->t('⚠ Purge users even if they have not been properly notified beforehand.');
    }
  }

}
