<?php

namespace Drupal\purge_users\Services;

use Drupal\Core\Condition\ConditionManager;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Plugin\Context\Context;
use Drupal\Core\Plugin\Context\ContextDefinition;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\purge_users\Entity\PurgeUsersPolicy;

/**
 * Implementation of the Purge Users Policy service.
 */
class PurgeUsersPolicyService implements PurgeUsersPolicyServiceInterface {

  use StringTranslationTrait;

  /**
   * Current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $config;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The messenger.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Logger channel service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $connection;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The condition manager.
   *
   * @var \Drupal\Core\Condition\ConditionManager
   */
  protected $conditionManager;

  /**
   * The queue factory service.
   *
   * @var \Drupal\Core\Queue\QueueFactory
   */
  protected $queueFactory;

  /**
   * UserManagementService constructor.
   *
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   Current user.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Database\Connection $connection
   *   The db connection.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\Core\Condition\ConditionManager $condition_manager
   *   The condition manager service.
   * @param \Drupal\Core\Queue\QueueFactory $queue_factory
   *   The queue factory service.
   */
  public function __construct(
    AccountProxyInterface $current_user,
    ConfigFactoryInterface $config_factory,
    ModuleHandlerInterface $module_handler,
    MessengerInterface $messenger,
    LoggerChannelFactoryInterface $logger_factory,
    Connection $connection,
    EntityTypeManagerInterface $entity_type_manager,
    ConditionManager $condition_manager,
    QueueFactory $queue_factory,
  ) {
    $this->currentUser = $current_user;
    $this->config = $config_factory;
    $this->moduleHandler = $module_handler;
    $this->messenger = $messenger;
    $this->loggerFactory = $logger_factory;
    $this->connection = $connection;
    $this->entityTypeManager = $entity_type_manager;
    $this->conditionManager = $condition_manager;
    $this->queueFactory = $queue_factory;
  }

  /**
   * {@inheritDoc}
   */
  public function clearQueues() {
    // Clear purge queue.
    $purge_queue = $this->queueFactory->get('purge_users');
    while ($item = $purge_queue->claimItem()) {
      $purge_queue->deleteItem($item);
    }
    // Clear notification queue.
    $notification_queue = $this->queueFactory->get('notification_users');
    while ($item = $notification_queue->claimItem()) {
      $notification_queue->deleteItem($item);
    }
  }

  /**
   * {@inheritDoc}
   */
  public function getAllPoliciesUserIdsToPurge(bool $cron): array {
    $policy_ids = PurgeUsersPolicy::loadMultiple();
    $uids = [];
    /** @var \Drupal\purge_users\Entity\PurgeUsersPolicy $policy */
    foreach ($policy_ids as $policy_id => $policy) {
      if (!$cron || $policy->isPurgeOnCron()) {
        $policy_uids = $this->getPolicyUserIdsByMode($policy, 'purge');
        foreach ($policy_uids as $uid) {
          $uids['user:' . $uid] = $policy_id;
        }
      }
    }
    return $uids;
  }

  /**
   * {@inheritDoc}
   */
  public function purgeAllUsers(bool $cron = FALSE, bool $dry_run = FALSE): int {
    $added_to_queue = 0;
    $user_ids = $this->getAllPoliciesUserIdsToPurge($cron);
    if (!empty($user_ids)) {
      // Add items to queue.
      $queue = $this->queueFactory->get('purge_users');
      // Don't add to the queue if it's not empty.
      if ($queue->numberOfItems() == 0) {
        foreach ($user_ids as $user_id_key => $policy_id) {
          $user_id = (int) substr($user_id_key, 5);
          $queue->createItem(['user_id' => $user_id, 'policy_id' => $policy_id, 'dry_run' => $dry_run]);
          $added_to_queue += 1;
        }
      }
    }
    return $added_to_queue;
  }

  /**
   * {@inheritDoc}
   */
  public function getAllPoliciesUserIdsToNotify(): array {
    $policy_ids = PurgeUsersPolicy::loadMultiple();
    $purge_uids = [];
    $notify_uids = [];
    foreach ($policy_ids as $policy_id => $policy) {
      $policy_purge_uids = $this->getPolicyUserIdsByMode($policy, 'purge');
      foreach ($policy_purge_uids as $uid) {
        $purge_uids['user:' . $uid] = $policy_id;
      }
      if ($this->isSendEmailUserBeforeNotification($policy)) {
        $policy_notify_uids = $this->getPolicyUserIdsByMode($policy, 'notify');
        foreach ($policy_notify_uids as $uid) {
          $notify_uids['user:' . $uid] = $policy_id;
        }
      }
    }
    return array_diff_key($notify_uids, $purge_uids);
  }

  /**
   * {@inheritDoc}
   */
  public function notifyAllUsers(): int {
    $added_to_queue = 0;
    $user_ids = $this->getAllPoliciesUserIdsToNotify();
    if (!empty($user_ids)) {
      // Add items to queue.
      $queue = $this->queueFactory->get('notification_users');
      // Don't add to the queue if it's not empty.
      if ($queue->numberOfItems() == 0) {
        foreach ($user_ids as $user_id_key => $policy_id) {
          $user_id = (int) substr($user_id_key, 5);
          $queue->createItem(['user_id' => $user_id, 'policy_id' => $policy_id]);
          $added_to_queue += 1;
        }
      }
    }
    return $added_to_queue;
  }

  /**
   * Get user ids to purge or notify for a policy.
   *
   * @param \Drupal\purge_users\Entity\PurgeUsersPolicy $policy
   *   The purge_users policy.
   * @param string $mode
   *   The query mode: "purge" or "notify".
   *
   * @return array
   *   The user ids.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  protected function getPolicyUserIdsByMode(PurgeUsersPolicy $policy, string $mode): array {
    if (empty($policy->getConditions())) {
      // Purging without any conditions is not allowed.
      return [];
    }

    // Query through db table to get user ids.
    $db = $this->connection;

    $query = $db->select('users_field_data', 'u')->distinct();
    $query->fields('u', ['uid']);
    $query->condition('u.uid', 1, '>');

    if ($policy->isDisregardBlockedUsers()) {
      $query->condition('u.status', 1);
    }

    $definition = new ContextDefinition('map', 'Policy context');
    $context = new Context($definition, [
      'db' => $db,
      'query' => $query,
      'policy' => $policy,
      'mode' => $mode,
    ]);

    $has_status_related_condition = FALSE;
    foreach ($policy->getConditions() as $condition) {
      /** @var \Drupal\purge_users\PurgeUsersConditionInterface $instance */
      $instance = $this->conditionManager->createInstance($condition['id'], $condition);
      $instance->setContext('context', $context);
      $instance->evaluate();
      if ($instance->isStatusRelated()) {
        $has_status_related_condition = TRUE;
      }
    }

    if (!$has_status_related_condition) {
      // Purging without any status related conditions is not allowed.
      return [];
    }

    $uids = $query->execute()->fetchCol();

    // Allow other modules to modify the result.
    $this->moduleHandler->alter('purge_policy_user_ids', $uids, $policy);

    return $uids;
  }

  /**
   * {@inheritDoc}
   */
  public function getPolicyUserIdsToPurge(PurgeUsersPolicy $policy): array {
    return $this->getPolicyUserIdsByMode($policy, 'purge');
  }

  /**
   * {@inheritDoc}
   */
  public function getPolicyUserIdsToNotify(PurgeUsersPolicy $policy): array {
    $purge_ids = $this->getPolicyUserIdsByMode($policy, 'purge');
    $notify_ids = $this->getPolicyUserIdsByMode($policy, 'notify');
    // We don't send pre-deletion notification to purged users.
    return array_diff($notify_ids, $purge_ids);
  }

  /**
   * Check if before deletion notification should be sent.
   *
   * @param \Drupal\purge_users\Entity\PurgeUsersPolicy $policy
   *   The current policy.
   *
   * @return bool
   *   TRUE if notification before deletion should be sent.
   */
  protected function isSendEmailUserBeforeNotification(PurgeUsersPolicy $policy) {
    $send_email = $policy->getSendEmailUserBeforeNotification();
    switch ($send_email) {
      case 'default':
        $config = $this->config->get('purge_users.settings');
        return $config->get('send_email_user_before_notification') ?? FALSE;

      case 'enabled':
        return TRUE;
    }
    return FALSE;
  }

}
