<?php

namespace Drupal\purge_users\Services;

use Drupal\purge_users\Entity\PurgeUsersPolicy;
use Drupal\user\UserInterface;

/**
 * Contract for the purge service.
 */
interface UserManagementServiceInterface {

  /**
   * Cancel a user and anonymize it.
   *
   * @param \Drupal\user\UserInterface $user
   *   The user to block.
   *   The cancel method to use.
   * @param \Drupal\purge_users\Entity\PurgeUsersPolicy|null $policy
   *   The policy to use.
   * @param bool $dry_run
   *   It TRUE, simulates the purge without canceling or deleting users.
   */
  public function purgeUser(UserInterface $user, ?PurgeUsersPolicy $policy = NULL, bool $dry_run = FALSE);

  /**
   * Sends a user notification before cancel and anonymize it.
   *
   * @param \Drupal\user\UserInterface $user
   *   The user to block.
   * @param \Drupal\purge_users\Entity\PurgeUsersPolicy|null $policy
   *   The policy to use.
   */
  public function notifyUser(UserInterface $user, ?PurgeUsersPolicy $policy = NULL): void;

  /**
   * Checks if a user is already notified.
   *
   * @param string $user_id
   *   The user id.
   * @param string $type
   *   The notification type.
   *
   * @return bool
   *   Return true if user id is found.
   */
  public function userIsNotified(string $user_id, string $type): bool;

  /**
   * Flags the user as notified.
   *
   * @param string $user_id
   *   The user id.
   * @param string $type
   *   The notification type.
   */
  public function flagUserAsNotified(string $user_id, string $type): void;

  /**
   * Removes the notification entry.
   *
   * @param string $user_id
   *   The user id.
   */
  public function removeNotificationFlags(string $user_id): void;

  /**
   * Build the email notification context.
   */
  public function getNotificationContext($langcode, $op = 'user_deletion', ?PurgeUsersPolicy $policy = NULL);

}
