<?php

declare(strict_types=1);

namespace Drupal\Tests\purge_users\Functional;

/**
 * Do not purge users who have contributed contents.
 *
 * - Purge method: Delete the account and its content.
 * - Disregard inactive/blocked users unselected.
 * - User Deletion Notification unselected.
 *
 * @group purge_users
 */
class ContentAuthorDeleteTest extends SettingsBase {

  /**
   * A node created by a blocked user.
   *
   * @var \Drupal\node\NodeInterface
   */
  protected $nodeBlocked;

  /**
   * A node created by an active user.
   *
   * @var \Drupal\node\NodeInterface
   */
  protected $nodeActive;

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    $this->nodeStorage = $this->container->get('entity_type.manager')->getStorage('node');
    $this->userStorage = $this->container->get('entity_type.manager')->getStorage('user');

    // Set the admin user for this scenario.
    $this->addAdminUser();

    // Set the user that will be deleted.
    $this->createTestUser();

    $this->setUpConfig();
  }

  /**
   * Setup configuration for the test.
   */
  protected function setUpConfig(): void {
    // Set the basic configuration and add the specific changes.
    $this->setBasicConfig();
    $this->config('purge_users.settings')
      ->set('user_blocked_value', '100')
      ->set('user_blocked_period', 'days')
      ->set('enabled_blocked_users', TRUE)
      ->set('user_lastlogin_value', '10')
      ->set('user_lastlogin_period', 'month')
      ->set('enabled_loggedin_users', TRUE)
      ->set('enabled_do_not_purge_authors', TRUE)
      ->save();
  }

  /**
   * {@inheritdoc}
   */
  protected function checkConfirmFormResults(): void {
    $this->checkTestResults();
  }

  /**
   * {@inheritdoc}
   */
  protected function checkCronResults(): void {
    $this->checkTestResults();
  }

  /**
   * Check the state of each user.
   */
  protected function checkTestResults(): void {

    // Blocked user is not deleted because it has content.
    $account = $this->userStorage->load($this->blockedUser->id());
    $this->assertNotNull($account);

    // Confirm user's content has not been deleted.
    $test_node = $this->nodeStorage->loadUnchanged($this->nodeBlocked->id());
    $this->assertNotNull($test_node);

    // Blocked user is deleted because it does not have authored content.
    $account = $this->userStorage->load($this->blockedUserToDelete->id());
    $this->assertNull($account);

    // Not logged-in user is not deleted because it has content.
    $account = $this->userStorage->load($this->activeUser->id());
    $this->assertNotNull($account);

    // Confirm user's content has not been deleted.
    $test_node = $this->nodeStorage->loadUnchanged($this->nodeActive->id());
    $this->assertNotNull($test_node);

    // Not logged-in user is deleted because it does not have authored content.
    $account = $this->userStorage->load($this->activeUserToDelete->id());
    $this->assertNull($account);

  }

  /**
   * Settings for a user to be deleted.
   *
   * This user and their content will be deleted.
   */
  protected function createTestUser(): void {

    // User created 150 days ago, user has logged in, status = 0, has content.
    $this->blockedUser = $this->createUser([], NULL, FALSE, [
      'created' => strtotime('-150 day'),
      'login' => strtotime('-150 day'),
    ]);

    // Node of this user.
    $this->nodeBlocked = $this->createNode([
      'uid' => $this->blockedUser->id(),
      'created' => strtotime('-150 day'),
      'changed' => strtotime('-150 day'),
    ]);

    // We now block the user.
    $this->blockedUser->status = 0;
    $this->blockedUser->save();

    // User created 150 days ago, user has logged in, status = 0, no content.
    $this->blockedUserToDelete = $this->createUser([], NULL, FALSE, [
      'created' => strtotime('-150 day'),
      'login' => strtotime('-150 day'),
      'status' => 0,
    ]);

    // User last logged in 13 months ago, has content.
    $this->activeUser = $this->createUser([], NULL, FALSE, [
      'created' => strtotime('-13 month'),
      'login' => strtotime('-13 month'),
    ]);

    // Node of this user.
    $this->nodeActive = $this->createNode([
      'uid' => $this->activeUser->id(),
      'created' => strtotime('-13 month'),
      'changed' => strtotime('-13 month'),
    ]);

    // User last logged in 13 months ago, no content.
    $this->activeUserToDelete = $this->createUser([], NULL, FALSE, [
      'created' => strtotime('-13 month'),
      'login' => strtotime('-13 month'),
    ]);

  }

}
