<?php

namespace Drupal\Tests\purge_users\Functional\policy;

use Drupal\Tests\purge_users\Functional\NotificationTest;

/**
 * Test all purge notifications.
 *
 * @group purge_users
 */
class PolicyNotificationTest extends NotificationTest {

  /**
   * {@inheritdoc}
   */
  public function setUpConfig(): void {
    $not_logged_in_policy = $this->setBasicPolicyConfig('not_logged_in_policy');
    $this->setUpNotLoggedInUsersPolicy($not_logged_in_policy);
    $never_logged_in_policy = $this->setBasicPolicyConfig('never_logged_in_policy');
    $this->setUpNeverLoggedInUsersPolicy($never_logged_in_policy);
    $blocked_policy = $this->setBasicPolicyConfig('blocked_policy');
    $this->setUpBlockedUsersPolicy($blocked_policy);
    $inactive_policy = $this->setBasicPolicyConfig('inactive_policy');
    $this->setUpInactiveUsersPolicy($inactive_policy);
  }

  /**
   * {@inheritdoc}
   */
  protected function setUpConfigNoEmails() {
    foreach ($this->policies as $policy) {
      $policy
        ->set('send_email_user_before_notification', 'disabled')
        ->set('send_email_notification', 'disabled')
        ->save();
    }
  }

  /**
   * Sends pre-notifications to all test users.
   */
  protected function preNotifyUsers(): void {
    $not_logged_in_policy = $this->policies['not_logged_in_policy'];
    $this->notifyManager->notifyUser($this->lastloginUser7Days, $not_logged_in_policy);
    $this->notifyManager->notifyUser($this->lastloginUser4Days, $not_logged_in_policy);
    $never_logged_in_policy = $this->policies['never_logged_in_policy'];
    $this->notifyManager->notifyUser($this->neverLoggedUser7Days, $never_logged_in_policy);
    $this->notifyManager->notifyUser($this->neverLoggedUser4Days, $never_logged_in_policy);
    $blocked_policy = $this->policies['blocked_policy'];
    $this->notifyManager->notifyUser($this->blockedUser7Days, $blocked_policy);
    $this->notifyManager->notifyUser($this->blockedUser4Days, $blocked_policy);
    $inactive_policy = $this->policies['inactive_policy'];
    $this->notifyManager->notifyUser($this->inactiveUser7Days, $inactive_policy);
    $this->notifyManager->notifyUser($this->inactiveUser4Days, $inactive_policy);
  }

  /**
   * Tests that a user who must be purged only receives the deletion email.
   */
  protected function setUpUserDeletionNotificationConfig() {
    // Disable all but one purge policy.
    $this->policies['not_logged_in_policy']->delete();
    $this->policies['never_logged_in_policy']->delete();
    $this->policies['blocked_policy']->delete();
    unset($this->policies['not_logged_in_policy']);
    unset($this->policies['never_logged_in_policy']);
    unset($this->policies['blocked_policy']);

    // Purge inactive users after 10 days.
    $inactive_policy = $this->policies['inactive_policy'];
    $conditions = $inactive_policy->getConditions();
    $conditions['purge_users:inactive'] = [
      'id' => 'purge_users:inactive',
      'user_inactive_value' => '10',
      'user_inactive_period' => 'days',
    ];
    $inactive_policy->set('policy_conditions', $conditions);

    $inactive_policy
      // Send pre-notification after 8 days.
      ->set('user_before_notification_value', '8')
      ->set('user_before_notification_period', 'days')
      ->set('send_email_user_before_notification', 'enabled')
      // Send purge email.
      ->set('send_email_notification', 'enabled')
      // Change the purge method, to be sure the user is being deleted.
      ->set('purge_user_cancel_method', 'user_cancel_delete');

    $inactive_policy->save();
  }

  /**
   * Test the notification required condition on an inactive user.
   */
  protected function setUpNotificationRequiredConfig() {
    // Disable all but one purge policy.
    $this->policies['not_logged_in_policy']->delete();
    $this->policies['never_logged_in_policy']->delete();
    $this->policies['blocked_policy']->delete();
    unset($this->policies['not_logged_in_policy']);
    unset($this->policies['never_logged_in_policy']);
    unset($this->policies['blocked_policy']);

    // Purge inactive users after 10 days.
    $inactive_policy = $this->policies['inactive_policy'];
    $conditions = $inactive_policy->getConditions();
    $conditions['purge_users:inactive'] = [
      'id' => 'purge_users:inactive',
      'user_inactive_value' => '10',
      'user_inactive_period' => 'days',
    ];
    $conditions['purge_users:notification_required'] = [
      'id' => 'purge_users:notification_required',
      'notification_required' => TRUE,
    ];
    $inactive_policy->set('policy_conditions', $conditions);

    $inactive_policy
      // Send pre-notification after 8 days.
      ->set('user_before_notification_value', '8')
      ->set('user_before_notification_period', 'days')
      ->set('send_email_user_before_notification', 'enabled')
      // Send purge email.
      ->set('send_email_notification', 'enabled')
      // Change the purge method, to be sure the user is being deleted.
      ->set('purge_user_cancel_method', 'user_cancel_delete');

    $inactive_policy->save();
  }

  /**
   * Test the notification required condition on an inactive user.
   */
  public function testNotificationRequiredCondition(): void {
    $user = $this->createUser([], NULL, FALSE, [
      'mail' => 'inactiveUser11Days@test.test',
      'created' => strtotime('-11 day'),
      'login' => 0,
      'status' => 0,
    ]);

    // Override the configuration to only target this user.
    $this->setUpNotificationRequiredConfig();

    $this->runPurgeOperation('form');

    // As notification is required, the user should have received
    // a pre-notification email.
    $this->assertPreNotificationEmail($user);
    // No further emails.
    // 3 pre-notification emails (4 days and 7 days and not notified).
    $this->assertCount(3, $this->getMails());

    $this->cronRun();

    // Check the user does not receive any other notification.
    $this->assertCount(3, $this->getMails());
  }

  /**
   * {@inheritdoc}
   */
  protected function setUpUserAccountSettingsNotificationConfig() {
    $this->config('user.settings')
      // Send core user module notifications when user is blocked.
      ->set('notify.status_blocked', TRUE)
      ->save();

    foreach ($this->policies as $policy) {
      $policy
        ->set('send_email_notification', 'disabled')
        ->save();
    }
  }

  /**
   * Set up the policy to purge not logged in users.
   *
   * @param \Drupal\purge_users\Entity\PurgeUsersPolicy $policy
   *   The policy entity.
   */
  protected function setUpNotLoggedInUsersPolicy($policy) {
    $this->addDefaultRoleConditions($policy);
    $conditions = $policy->getConditions();
    // Purge regular users after 5 days of not logging in.
    $conditions['purge_users:not_logged_in'] = [
      'id' => 'purge_users:not_logged_in',
      'user_lastlogin_value' => '5',
      'user_lastlogin_period' => 'days',
    ];
    $policy->set('policy_conditions', $conditions);
    $policy->save();
  }

  /**
   * Set up the policy to purge never-logged-in users.
   *
   * @param \Drupal\purge_users\Entity\PurgeUsersPolicy $policy
   *   The policy entity.
   */
  protected function setUpNeverLoggedInUsersPolicy($policy) {
    $this->addDefaultRoleConditions($policy);
    $conditions = $policy->getConditions();
    // Purge never-logged-in users after 5 days.
    $conditions['purge_users:never_logged_in'] = [
      'id' => 'purge_users:never_logged_in',
      'user_never_lastlogin_value' => '5',
      'user_never_lastlogin_period' => 'days',
    ];
    $policy->set('policy_conditions', $conditions);
    $policy->save();
  }

  /**
   * Set up the policy to purge blocked users.
   *
   * @param \Drupal\purge_users\Entity\PurgeUsersPolicy $policy
   *   The policy entity.
   */
  protected function setUpBlockedUsersPolicy($policy) {
    $this->addDefaultRoleConditions($policy);
    $conditions = $policy->getConditions();
    // Purge blocked users after 5 days.
    $conditions['purge_users:blocked'] = [
      'id' => 'purge_users:blocked',
      'user_blocked_value' => '5',
      'user_blocked_period' => 'days',
    ];
    $policy->set('policy_conditions', $conditions);
    $policy->save();
  }

  /**
   * Set up the policy to purge inactive users.
   *
   * @param \Drupal\purge_users\Entity\PurgeUsersPolicy $policy
   *   The policy entity.
   */
  protected function setUpInactiveUsersPolicy($policy) {
    $this->addDefaultRoleConditions($policy);
    $conditions = $policy->getConditions();
    // Purge inactive users after 5 days.
    $conditions['purge_users:inactive'] = [
      'id' => 'purge_users:inactive',
      'user_inactive_value' => '5',
      'user_inactive_period' => 'days',
    ];
    $policy->set('policy_conditions', $conditions);
    $policy->save();
  }

  /**
   * Add the default role conditions to the policy.
   *
   * @param \Drupal\purge_users\Entity\PurgeUsersPolicy $policy
   *   The policy entity.
   */
  private function addDefaultRoleConditions($policy) {
    $conditions = $policy->getConditions();
    $conditions['purge_users:excluded_roles'] = [
      'id' => 'purge_users:excluded_roles',
      'purge_excluded_users_roles' => ['administrator'],
    ];
    $conditions['purge_users:included_roles'] = [
      'id' => 'purge_users:included_roles',
      'purge_included_users_roles' => ['authenticated'],
    ];
    $policy->set('policy_conditions', $conditions);
  }

}
