<?php

declare(strict_types=1);

namespace Drupal\Tests\purge_users\Kernel;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormState;
use Drupal\KernelTests\KernelTestBase as CoreKernelTestBase;
use Drupal\purge_users\Entity\PurgeUsersPolicy;
use Drupal\purge_users\Form\PolicyConfirmationForm;
use Drupal\Tests\user\Traits\UserCreationTrait;
use Drupal\purge_users\Form\ConfirmationForm;

/**
 * Base class for kernel tests.
 *
 * @group purge_users
 */
abstract class KernelTestBase extends CoreKernelTestBase {

  use UserCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['system', 'user', 'purge_users'];

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The policy used for UI purge.
   *
   * @var \Drupal\purge_users\Services\PurgeUsersPolicyServiceInterface
   */
  protected $policyService;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * User storage.
   *
   * @var \Drupal\user\UserStorageInterface
   */
  protected $userStorage;

  /**
   * The cron service.
   *
   * @var \Drupal\Core\Cron
   */
  protected $cron;

  /**
   * Policy entities.
   *
   * @var array
   */
  protected $policies = [];

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();
    $this->initKernelTest();
    $this->setUpConfig();

    // Occupy user id 1 with an admin account.
    // Otherwise, we get confusing test results.
    $admin = $this->createUser([], 'admin', TRUE, [
      'created' => strtotime('-1 year'),
      'login' => 0,
      'status' => 0,
    ]);
    self::assertEquals(1, $admin->id());
  }

  /**
   * Initializes services needed for a kernel test.
   */
  protected function initKernelTest() {
    $this->installSchema('system', ['sequences']);
    $this->installSchema('user', ['users_data']);
    $this->installSchema('purge_users', ['purge_users_notifications']);
    $this->installEntitySchema('user');

    /** @var \Drupal\Core\Entity\EntityTypeManagerInterface $etm */
    $etm = $this->container->get('entity_type.manager');
    $this->userStorage = $etm->getStorage('user');
    $this->cron = $this->container->get('cron');

    $this->configFactory = $this->container->get('config.factory');
    $this->policyService = $this->container->get('purge_users.policy_service');
    $this->messenger = $this->container->get('messenger');

    $this->config('system.site')
      // Replicate settings from browser tests, to have same expected emails.
      ->set('mail', 'simpletest@example.com')
      ->set('name', 'Drupal')
      ->save();
  }

  /**
   * Sets a baseline configuration for all tests.
   */
  protected function setBasicConfig(): void {
    $this->config('purge_users.settings')
      // Disable all the purge conditions.
      ->set('enabled_inactive_users', FALSE)
      ->set('enabled_loggedin_users', FALSE)
      ->set('enabled_never_loggedin_users', FALSE)
      ->set('enabled_blocked_users', FALSE)
      // Exclude admin, include regular users.
      ->set('purge_excluded_users_roles', ['administrator'])
      ->set('purge_included_users_roles', ['authenticated'])
      // Disable notifications on purge, but set default text.
      ->set('send_email_notification', FALSE)
      ->set('inactive_user_notify_subject', 'Dear user')
      ->set('inactive_user_notify_text', 'Dear User, Your account has been deleted due the website’s policy to automatically remove users who match certain criteria. If you have concerns regarding the deletion, please talk to the administrator of the website. Thank you')
      // Disable notifications before purge, but set default text.
      ->set('user_before_deletion_subject', 'Dear user')
      ->set('send_email_user_before_notification', FALSE)
      ->set('user_before_deletion_text', 'Dear User, Your account will be deleted soon due the website’s policy to automatically remove users who match certain criteria. If you have concerns regarding the deletion, please talk to the administrator of the website. Thank you')
      // Purge when running cron.
      ->set('purge_on_cron', TRUE)
      ->save();
  }

  /**
   * Set the basic policy configuration for all tests.
   *
   * Each test will override accordingly.
   *
   * @param string $id
   *   The id of the policy.
   *
   * @return \Drupal\purge_users\Entity\PurgeUsersPolicy
   *   The policy entity.
   */
  protected function setBasicPolicyConfig(string $id = 'test_policy'): PurgeUsersPolicy {
    // Create a new PurgeUsersPolicy entity instance.
    /** @var \Drupal\purge_users\Entity\PurgeUsersPolicy $policy */
    $policy = PurgeUsersPolicy::create([
      'id' => $id,
      'label' => 'Test Policy',
      'description' => 'A policy for testing purposes.',
      // Disable notifications on purge, but set default text.
      'send_email_notification' => 'disabled',
      'inactive_user_notify_subject' => 'Dear user',
      'inactive_user_notify_text' => 'Dear User, Your account has been deleted due the website’s policy to automatically remove users who match certain criteria. If you have concerns regarding the deletion, please talk to the administrator of the website. Thank you',
      // Disable notifications before purge, but set default text.
      'send_email_user_before_notification' => 'disabled',
      'user_before_deletion_subject' => 'Dear user',
      'user_before_deletion_text' => 'Dear User, Your account will be deleted soon due the website’s policy to automatically remove users who match certain criteria. If you have concerns regarding the deletion, please talk to the administrator of the website. Thank you',
      // Purge when running cron.
      'purge_on_cron' => TRUE,
    ]);
    $conditions = $policy->getConditions();
    // Exclude admin, include regular users.
    $conditions['purge_users:excluded_roles'] = [
      'id' => 'purge_users:excluded_roles',
      'purge_excluded_users_roles' => ['administrator'],
    ];
    $conditions['purge_users:included_roles'] = [
      'id' => 'purge_users:included_roles',
      'purge_included_users_roles' => ['authenticated'],
    ];
    $policy->set('policy_conditions', $conditions);
    $policy->save();
    $this->policies[$id] = $policy;
    return $policy;
  }

  /**
   * Get the basic policy configuration for all tests.
   *
   * @param string $id
   *   The id of the policy.
   *
   * @return \Drupal\purge_users\Entity\PurgeUsersPolicy|null
   *   The policy entity.
   */
  protected function getBasicPolicyConfig(string $id = 'test_policy'): ?PurgeUsersPolicy {
    return $this->policies[$id] ?? NULL;
  }

  /**
   * Setup configuration for the test.
   */
  protected function setUpConfig(): void {
    // Set the basic configuration and add the specific changes.
    $this->setBasicConfig();
  }

  /**
   * Imports settings.
   *
   * @param array $settings
   *   Format: $[$name][$key] = $value.
   */
  protected function setSettings(array $settings): void {
    foreach ($settings as $name => $values) {
      $config = $this->config($name);
      foreach ($values as $k => $v) {
        $config->set($k, $v);
      }
      $config->save();
    }
  }

  /**
   * Runs the purge operation, either as cron or form submit.
   *
   * @param string $mode
   *   One of 'cron' or 'form'.
   * @param bool $dry_run
   *   Whether to run the operation in dry run mode.
   */
  protected function runPurgeOperation(string $mode, bool $dry_run = FALSE): void {
    if ($mode === 'form') {
      if (!empty($this->policies)) {
        $form_object = new PolicyConfirmationForm(
          $this->configFactory, $this->policyService, $this->messenger
        );
        $form_state = new FormState();
        $form_state->setValue('dry_run', $dry_run);
        foreach ($this->policies as $policy) {
          $form_object->setPolicy($policy);
          $this->submitConfirmationForm($form_object, $form_state);
        }
      }
      else {
        $form_object = new ConfirmationForm($this->configFactory);
        $this->submitConfirmationForm($form_object);
      }
    }
    else {
      $this->cron->run();
    }
  }

  /**
   * Submits the confirmation form.
   *
   * @param \Drupal\Core\Form\ConfirmFormBase $form_object
   *   The form object to submit.
   * @param \Drupal\Core\Form\FormState $form_state
   *   The form state object.
   */
  private function submitConfirmationForm(
    ConfirmFormBase $form_object,
    FormState $form_state = new FormState(),
  ): void {
    $form = [];
    $form_object->submitForm($form, $form_state);
    $batch = &batch_get();
    if ($batch) {
      $batch['progressive'] = FALSE;
      batch_process();
    }
  }

  /**
   * Duplicates existing datasets, appending a new parameter 'form'|'cron'.
   *
   * @param array[] $datasets
   *   Original datasets.
   *   Format: $[$dataset_name] = [...$args].
   *
   * @return array[]
   *   New datasets.
   *   Format: $[$dataset_name . '.form'|'.cron'] = [...$args, 'cron'|'form'].
   */
  protected static function duplicateAndAppendPurgeMode(array $datasets): array {
    $new_datasets = [];
    foreach ($datasets as $dataset_name => $dataset) {
      $new_datasets["$dataset_name.cron"] = array_merge($dataset, ['cron']);
      $new_datasets["$dataset_name.form"] = array_merge($dataset, ['form']);
    }
    return $new_datasets;
  }

}
