<?php

declare(strict_types=1);

namespace Drupal\Tests\purge_users\Kernel\policy;

use Drupal\Core\Test\AssertMailTrait;
use Drupal\Tests\purge_users\Kernel\KernelTestBase;
use Drupal\Tests\purge_users\Traits\ChangedUsersTrait;

/**
 * Covers dry run.
 *
 * @group purge_users
 */
class PolicyDryRunTest extends KernelTestBase {

  use AssertMailTrait;
  use ChangedUsersTrait;

  /**
   * Setup configuration for the test.
   */
  protected function setUpConfig(): void {
    // Set the basic policy configuration.
    $policy = $this->setBasicPolicyConfig();
    $conditions = $policy->getConditions();
    $conditions[] = [
      'id' => 'purge_users:never_logged_in',
      'user_never_lastlogin_value' => '2',
      'user_never_lastlogin_period' => 'year',
    ];
    $policy->set('policy_conditions', $conditions);
    $policy->set('purge_user_cancel_method', 'user_cancel_delete');
    $policy->set('send_email_notification', 'enabled');
    $policy->save();
  }

  /**
   * Tests different purge methods.
   *
   * @param string[] $expected_user_states
   *   Format: $[$key] = 'deleted'|'blocked'|'unblocked'|...
   *   Only contains entries that differ from the original state.
   *
   * @dataProvider providerPurgeMethod
   */
  public function testDryRun(array $expected_user_states): void {

    // Don't call strtotime repeatedly within the loop.
    $one_year_ago = strtotime('-1 year');
    $three_years_ago = strtotime('-3 year');

    // phpcs:disable
    // Keep these aligned for easy comparison.
    $user_values = [
      'created3y' =>         ['created' => $three_years_ago, 'status' => 1],
      'created3y_blocked' => ['created' => $three_years_ago, 'status' => 0],
      'created1y' =>         ['created' => $one_year_ago,    'status' => 1],
      'created1y_blocked' => ['created' => $one_year_ago,    'status' => 0],
    ];
    // phpcs:enable

    // Create users.
    $original_users = [];
    foreach ($user_values as $name => $values) {
      $values['login'] = 0;
      $user = $this->createUser(
        [],
        $name,
        FALSE,
        $values);
      $original_users[$name] = clone $user;
    }

    $this->runPurgeOperation('form', TRUE);

    self::assertSame(
      [
        // Dry run: no users should have been purged.
        'users' => [],
      ],
      [
        'users' => $this->getUserStateChanges($original_users),
      ],
    );

    // No notification email should have been sent in dry-run mode.
    $this->assertCount(0, $this->getMails());

    $this->runPurgeOperation('form', FALSE);

    self::assertSame(
      [
        'users' => $expected_user_states,
      ],
      [
        'users' => $this->getUserStateChanges($original_users),
      ],
    );

    // The deletion notification emails should have been sent.
    $this->assertCount(count($expected_user_states), $this->getMails());
  }

  /**
   * Data provider.
   *
   * @return array[]
   *   Format: $[$dataset_name] = [
   *     $expected_user_states,
   *   ].
   */
  public static function providerPurgeMethod(): array {
    return [
      'user_cancel_delete' => [
        [
          'created3y' => 'deleted',
          'created3y_blocked' => 'deleted',
        ],
      ],
    ];
  }

}
