<?php

declare(strict_types=1);

namespace Drupal\Tests\purge_users\Kernel\policy;

use Drupal\Tests\purge_users\Kernel\PurgeConditionTest;

/**
 * Covers settings that control which users get purged, and when.
 *
 * @group purge_users
 */
class PolicyPurgeConditionTest extends PurgeConditionTest {

  /**
   * Setup configuration for the test.
   */
  protected function setUpConfig(): void {
    // Set the basic policy configuration.
    $this->setBasicPolicyConfig();
  }

  /**
   * {@inheritdoc}
   */
  protected function setUpPurgeOnCron(bool $purge_on_cron) {
    $policy = $this->getBasicPolicyConfig();
    $conditions = $policy->getConditions();
    // Set a purge condition.
    $conditions[] = [
      'id' => 'purge_users:never_logged_in',
      'user_never_lastlogin_value' => '2',
      'user_never_lastlogin_period' => 'year',
    ];
    $policy->set('policy_conditions', $conditions);
    // Set purge method.
    $policy->set('purge_user_cancel_method', 'user_cancel_delete');
    // Set purge on cron.
    $policy->set('purge_on_cron', $purge_on_cron);
    $policy->save();
  }

  /**
   * Sets up the configuration for the role conditions test.
   */
  protected function setUpRoleConditions() {
    $policy = $this->getBasicPolicyConfig();
    $conditions = $policy->getConditions();
    // Set a purge condition.
    $conditions[] = [
      'id' => 'purge_users:never_logged_in',
      'user_never_lastlogin_value' => '2',
      'user_never_lastlogin_period' => 'year',
    ];
    $conditions['purge_users:excluded_roles'] = [
      'id' => 'purge_users:excluded_roles',
      'purge_excluded_users_roles' => ['administrator', 'moderator'],
    ];
    $conditions['purge_users:included_roles'] = [
      'id' => 'purge_users:included_roles',
      'purge_included_users_roles' => ['authenticated', 'contributor'],
    ];
    $policy->set('policy_conditions', $conditions);
    // Set purge method.
    $policy->set('purge_user_cancel_method', 'user_cancel_delete');
    $policy->save();
  }

  /**
   * Sets up the configuration for the user conditions test.
   *
   * @param array $settings
   *   Settings for the user conditions test.
   */
  protected function setUpUserConditions(array $settings) {
    $policy = $this->getBasicPolicyConfig();
    // Set default purge method.
    $policy->set('purge_user_cancel_method', 'user_cancel_delete');
    $conditions = $policy->getConditions();
    foreach ($settings as $k => $v) {
      if ($k === 'conditions') {
        // Set purge conditions.
        foreach ($v as $condition) {
          $conditions[] = $condition;
        }
      }
      else {
        $policy->set($k, $v);
      }
    }
    $policy->set('policy_conditions', $conditions);
    $policy->save();
  }

  /**
   * Data provider.
   *
   * @return array[]
   *   Format: $[$dataset_name] = [
   *     $expected_user_state_changes,
   *     $purge_users_config,
   *     'cron'|'form',
   *   ].
   */
  public static function userConditionsProvider(): array {
    $value = 2;
    $period = 'year';
    return static::duplicateAndAppendPurgeMode([
      'purge_none' => [
        [
          // No users are purged.
        ],
        [
          // No purge conditions are enabled.
        ],
      ],
      // All never logged-in users are purged.
      'never_logged_in_all' => [
        [
          'created3y_loginNever_blocked' => 'deleted',
          'created3y_loginNever' => 'deleted',
          'created1y_loginNever_blocked' => 'deleted',
          'created1y_loginNever' => 'deleted',
        ],
        [
          // Enable a never logged in purge condition with short timespan.
          'conditions' => [
            'purge_users:never_logged_in' => [
              'id' => 'purge_users:never_logged_in',
              'user_never_lastlogin_value' => 2,
              'user_never_lastlogin_period' => 'day',
            ],
          ],
        ],
      ],
      // All not logged-in users are purged.
      'not_logged_in_all' => [
        [
          'created3y_login3y_blocked' => 'deleted',
          'created3y_login3y' => 'deleted',
          'created3y_login1y_blocked' => 'deleted',
          'created3y_login1y' => 'deleted',
          'created1y_login1y_blocked' => 'deleted',
          'created1y_login1y' => 'deleted',
        ],
        [
          // Enable a not logged in purge conditions with short timespan.
          'conditions' => [
            'purge_users:not_logged_in' => [
              'id' => 'purge_users:not_logged_in',
              'user_lastlogin_value' => 2,
              'user_lastlogin_period' => 'day',
            ],
          ],
        ],
      ],
      // All blocked users are purged.
      'blocked_all' => [
        [
          'created3y_login3y_blocked' => 'deleted',
          'created3y_login1y_blocked' => 'deleted',
          'created1y_login1y_blocked' => 'deleted',
        ],
        [
          // Enable a blocked purge conditions with short timespan.
          'conditions' => [
            'purge_users:blocked' => [
              'id' => 'purge_users:blocked',
              'user_blocked_value' => 2,
              'user_blocked_period' => 'day',
            ],
          ],
        ],
      ],
      // All inactive users are purged.
      'inactive_all' => [
        [
          'created3y_loginNever_blocked' => 'deleted',
          'created1y_loginNever_blocked' => 'deleted',
        ],
        [
          // Enable an inactive purge conditions with short timespan.
          'conditions' => [
            'purge_users:inactive' => [
              'id' => 'purge_users:inactive',
              'user_inactive_value' => 2,
              'user_inactive_period' => 'day',
            ],
          ],
        ],
      ],
      'never_logged_in' => [
        [
          'created3y_loginNever_blocked' => 'deleted',
          'created3y_loginNever' => 'deleted',
        ],
        [
          'conditions' => [
            'purge_users:never_logged_in' => [
              'id' => 'purge_users:never_logged_in',
              'user_never_lastlogin_value' => $value,
              'user_never_lastlogin_period' => $period,
            ],
          ],
        ],
      ],
      'not_been_active_for' => [
        [
          'created3y_loginNever_blocked' => 'deleted',
        ],
        [
          'conditions' => [
            'purge_users:inactive' => [
              'id' => 'purge_users:inactive',
              'user_inactive_value' => $value,
              'user_inactive_period' => $period,
            ],
          ],
        ],
      ],
      'blocked_for' => [
        [
          'created3y_login3y_blocked' => 'deleted',
        ],
        [
          'conditions' => [
            'purge_users:blocked' => [
              'id' => 'purge_users:blocked',
              'user_blocked_value' => $value,
              'user_blocked_period' => $period,
            ],
          ],
        ],
      ],
      'not_logged_in_for' => [
        [
          'created3y_login3y_blocked' => 'deleted',
          'created3y_login3y' => 'deleted',
        ],
        [
          'conditions' => [
            'purge_users:not_logged_in' => [
              'id' => 'purge_users:not_logged_in',
              'user_lastlogin_value' => $value,
              'user_lastlogin_period' => $period,
            ],
          ],
        ],
      ],
      'not_logged_in_for.disregard_blocked' => [
        [
          'created3y_login3y' => 'deleted',
        ],
        [
          'conditions' => [
            'purge_users:not_logged_in' => [
              'id' => 'purge_users:not_logged_in',
              'user_lastlogin_value' => $value,
              'user_lastlogin_period' => $period,
            ],
          ],
          'disregard_blocked_users' => TRUE,
        ],
      ],
      'not_logged_in_for.block_on_purge' => [
        [
          'created3y_login3y' => 'blocked',
        ],
        [
          'conditions' => [
            'purge_users:not_logged_in' => [
              'id' => 'purge_users:not_logged_in',
              'user_lastlogin_value' => $value,
              'user_lastlogin_period' => $period,
            ],
          ],
          'purge_user_cancel_method' => 'user_cancel_block',
        ],
      ],
      'not_logged_in_for.disregard_blocked.block_on_purge' => [
        [
          'created3y_login3y' => 'blocked',
        ],
        [
          'conditions' => [
            'purge_users:not_logged_in' => [
              'id' => 'purge_users:not_logged_in',
              'user_lastlogin_value' => $value,
              'user_lastlogin_period' => $period,
            ],
          ],
          'disregard_blocked_users' => TRUE,
          'purge_user_cancel_method' => 'user_cancel_block',
        ],
      ],
      'not_logged_in_for.notification_required_false' => [
        [
          // Users are purged as pre-notification is not required (FALSE).
          'created3y_login3y_blocked' => 'deleted',
          'created3y_login3y' => 'deleted',
        ],
        [
          'conditions' => [
            'purge_users:not_logged_in' => [
              'id' => 'purge_users:not_logged_in',
              'user_lastlogin_value' => $value,
              'user_lastlogin_period' => $period,
            ],
            'purge_users:notification_required' => [
              'id' => 'purge_users:notification_required',
              'notification_required' => FALSE,
            ],
          ],
        ],
      ],
      'not_logged_in_for.notification_required_true' => [
        [
          // No users were purged because they have not yet been notified,
          // as required.
        ],
        [
          'conditions' => [
            'purge_users:not_logged_in' => [
              'id' => 'purge_users:not_logged_in',
              'user_lastlogin_value' => $value,
              'user_lastlogin_period' => $period,
            ],
            'purge_users:notification_required' => [
              'id' => 'purge_users:notification_required',
              'notification_required' => TRUE,
            ],
          ],
        ],
      ],
    ]);
  }

}
