<?php
namespace Bridge\Weblibs;

use Bridge\Weblibs\BridgeClientParamsInterface;
use Bridge\Weblibs\BridgeRequestServiceInterface;
use Bridge\Weblibs\BridgeApiService;
use Bridge\Weblibs\BridgeClientContentInterface;
use Bridge\Weblibs\BridgeClientRendererInterface;
use Bridge\Weblibs\BridgeBlockService;

class BridgeDataGetter
{
    private BridgeClientParamsInterface $clientParams;

    private BridgeRequestServiceInterface $requestService;
    private BridgeApiService $bridgeApiService;

    private BridgeClientContentInterface $clientContentService;

    private BridgeClientRendererInterface $renderer;
    private BridgeBlockService $bridgeBlockService;

    public function __construct(BridgeClientParamsInterface $clientParams, BridgeRequestServiceInterface $requestService, BridgeClientContentInterface $clientContentService, BridgeClientRendererInterface $renderer, BridgeApiService $bridgeApiService, BridgeBlockService $bridgeBlockService) {
        $this->clientParams = $clientParams;
        $this->requestService = $requestService;
        $this->clientContentService = $clientContentService;
        $this->renderer = $renderer;
        $this->bridgeApiService = $bridgeApiService;
		$this->bridgeBlockService = $bridgeBlockService;
    }

    public function getListData($attributes) {

        $res = array(
            'success' => false,
            'message' => '',
            'total' => null,
            'data' => null,
            'infos' => array(),
            'parameters' => array()
        );

        $bridgeCredentials = $this->clientParams->getBridgeCredentials();
        $urlBridge = $bridgeCredentials->urlBridge;

        if(empty($urlBridge)) {
            $res['message'] = "L'URL de Bridge n'est pas définie - contrôlez les paramètres du plugin";
            return $res;
        }

        if(!is_array($attributes) || (!isset($attributes['id']) && !isset($attributes['relatedList']))) {
            $res['message'] = 'Le paramètre id est obligatoire dans le shortcode';
            return $res;
        }

        // La liste est filtrée par défaut en lisant les paramètres GET POST du contexte
        // l'attribut useRequestFilters permet de désactiver ce comportement
        if(!isset($attributes['useRequestFilters']))
            $attributes['useRequestFilters'] = true;

        // Variable utilisée à différents endroits du code
        $webListId = $attributes['id'];
        $res['infos']['attributes'] = $attributes;
        $res['infos']['webListId'] = $webListId;
        $res['infos']['useRequestFilters'] = $attributes['useRequestFilters'];
        $res['infos']['debug'] = array() ;
        // Classe CSS
        $containerCssClass = (isset($attributes['container_css_class']) && $attributes['container_css_class'] != '') ? $attributes['container_css_class'] : '';
        $res['infos']['containerCssClass'] = $containerCssClass;

        // Pagination des résultats
        // Première fiche
        $first = (isset($attributes['first']) && $attributes['first'] != '') ? $attributes['first'] : '1';
        $res['infos']['first'] = $first;

        // Si pagination : page à afficher
        $currentPage = (isset($attributes['brpa']) && $attributes['brpa'] != '') ? $attributes['brpa'] : '';
        $res['infos']['currentPage'] = $currentPage;
        $res['infos']['lastPage'] = $currentPage;

        $bridgeAjaxUrl = $this->clientParams->getAjaxURL();
        $res['infos']['bridgeAjaxUrl'] = $bridgeAjaxUrl;

        // Nb de fiches à afficher
        $max = (isset($attributes['max']) && $attributes['max'] != '') ? $attributes['max'] : '';
        $limitPerPage = (isset($attributes['max']) && $attributes['max'] != '') ? $attributes['max'] : '12';
        $res['infos']['max'] = $max;
        $res['infos']['limitPerPage'] = $limitPerPage;

        $lang = $this->clientParams->getLanguage();
        $res['infos']['lang'] = $lang;

        // On ajoute les paramètres de filtre du moteur et les paramètres de tri
        $brParamsUrlFiltre = '';
        $brParamsUrlSort = '';

        // On démarre un session pour pouvoir enregistrer des paramètres de filtres de localisation
        if (session_status() === PHP_SESSION_NONE && !headers_sent()) {
            session_start();
        }
        $geoLat = '';
        $geoLon = '';
        $geoCity = '';

        if (count($_REQUEST) > 0 && $attributes['useRequestFilters']) {

            foreach ($_REQUEST as $key=>$value) {
                // Patch 06.04.2023 : l'infinite scroll génère un paramètre d'URL aléatoire sans valeur : on ne doit pas le prendre en compte dans l'appel Bridge sinon les identifiers de liste vont bouger
                if(!isset($value) || empty($value))
                    continue;

                // braf: Bridge Active Filter
                // brai: Bridge Active Item
                if ($key == 'braf' || $key == 'brai' || (strlen($key) > 6 && substr($key,0,6) == 'brflt_')) {
                    // On doit encoder en URI les paramètres de type valeur saisie par l'internaute
                    if (strpos($key, '_value') !== false || strpos($key, '_city') !== false) {
                        $brParamsUrlFiltre .= "&$key=" . urlencode($value);
                    } else {
                        $brParamsUrlFiltre .= "&$key=$value";
                    }
                    // On mémorise les paramètres de recherche géographique en session
                    if (strpos($key, '_city') !== false) {
                        $geoCity = $value;
                    }
                    if (strpos($key, '_lat') !== false) {
                        $geoLat = $value;
                    }
                    if (strpos($key, '_lon') !== false) {
                        $geoLon = $value;
                    }
                }

                // bras: Bridge Active Sort
                // brsd: Bridge Sort Direction
                if ($key == 'bras' || $key == 'brsd') {
                    $brParamsUrlSort .= "&$key=$value";
                }

                if ($key == 'max' && !empty($value)) {
                    $limitPerPage = (int) $value;
                    $res['infos']['limitPerPage'] = $limitPerPage;
                }

				if ($key == 'selection_id' && !empty($value)) {
					$attributes['selection_id'] = $value;
                }
	            if ($key == 'scoring' && !empty($value)) {
		            $attributes['scoring'] = $value;
	            }
	            if ($key == 'minscore' && $value !== '') {
		            $attributes['minscore'] = $value;
	            }


                // Pagination : paramètre brpa = numéro de page
                if ($key == 'brpa' && !empty($value)) {
                    $res['infos']['debug'][] = "ON A UN BRPA DANS LURL";
                    $currentPage = (int) $value;
                    $res['infos']['currentPage'] = $currentPage;
                    // La paramètre de pagination a priorité sur le first
                    $first= '';
                }
            }
        }

        // On enregistre les paramètres de recherche géolocalisée en session pour pouvoir réafficher la distance de chaque fiche
        if(!empty($geoLat) && !empty($geoLon) && !empty($geoCity) ) {
            $_SESSION['bridge_geolat'] = $geoLat;
            $_SESSION['bridge_geolon'] = $geoLon;
            $_SESSION['bridge_geocity'] = $geoCity;

        }

        // on regarde si on change le contenu de la liste ou si on met juste à jour l'affichage (scroll, tri)
        $change = (isset($attributes['change']) && $attributes['change'] != '') ? $attributes['change'] : '0';

        // JM 22/03/2022 : on change le nom du paramètre (update -> v2), moins risqué
        $urlPaginationParams = "&brpa=" . $currentPage . "&first=" . $first . "&max=" . $max ;
        $url = $urlBridge . "/weblist/getProductsForList/" . $webListId . "?v2=1&lang=" . $lang . "&change=" . $change;

        // Filtres et tris forcés
        if(isset($attributes['braf']) && !empty($attributes['braf'])) {
            $brParamsUrlFiltre .= "&braf=" . $attributes['braf'];
        }
        if(isset($attributes['brai']) && !empty($attributes['brai'])) {
            $brParamsUrlFiltre .= "&brai=" . $attributes['brai'];
        }
        if(isset($attributes['bras']) && !empty($attributes['bras'])) {
            $brParamsUrlSort .= "&bras=" . $attributes['bras'];
        }
        if(isset($attributes['brsd']) && !empty($attributes['brsd'])) {
            $brParamsUrlSort .= "&brsd=" . $attributes['brsd'];
        }

        if ($brParamsUrlFiltre != '') {
            $url .= $brParamsUrlFiltre;
        }

        if ($brParamsUrlSort != '') {
            $url .= $brParamsUrlSort;
        }

        // Feat : 23/10/2022 : possibilité de passer un nom de commune pour cumulr un filtre par commune à la séelction de la liste
        $res['infos']['filter_city'] = '';
        if(isset($attributes['filter_city']) && !empty($attributes['filter_city'])) {
            $url .= "&filter_city=" . $attributes['filter_city'];
            $res['infos']['filter_city'] = $attributes['filter_city'];
        }

        // Feat : 25/05/2022 : possibilité de passer une liste de productCodes pour forcer les fiches à afficher dans le shortcode
        $productCodes = '';
        if(isset($attributes['product_codes']) && !empty($attributes['product_codes'])) {
            $productCodes = $attributes['product_codes'];
            if(is_array($productCodes))
                $productCodes = implode(',', $productCodes);
            $url .= "&productCodes=" . $productCodes;
        }
        $res['infos']['productCodes'] = $productCodes;

        // 23/01/2023 : filtre par lat lon rayon
        $res['infos']['filter_lat'] = '';
        $res['infos']['filter_lon'] = '';
        $res['infos']['filter_rayon'] = '';
        if(isset($attributes['filter_lat']) && isset($attributes['filter_lon']) && isset($attributes['filter_rayon'])) {
            $url .= "&lat=" . $attributes['filter_lat'] . "&lon=" . $attributes['filter_lon'] . "&rayon=" . $attributes['filter_rayon'];
            $res['infos']['filter_lat'] = $attributes['filter_lat'];
            $res['infos']['filter_lon'] = $attributes['filter_lon'];
            $res['infos']['filter_rayon'] = $attributes['filter_rayon'];
        }

        // 23/01/2023 : filtre par id de sélection
        $res['infos']['selection_id'] = '';
        if(isset($attributes['selection_id']) && !empty($attributes['selection_id'])) {
            $url .= "&selection_id=" . (int) $attributes['selection_id'];
            $res['infos']['selection_id'] = (int) $attributes['selection_id'];
        }
        // 13/02/2023 : filtre par critères / mdalités
        $res['infos']['filter_criterions'] = '';
        if(isset($attributes['filter_criterions']) && !empty($attributes['filter_criterions'])) {
            // Syntaxes pour l'attribut criterion (multicritères avec ou sans valeur) - on parle bien de criterions_codes et de modality_codes
            // CRIT1:MODA1|MODA2|MODA3,CRIT2:MODA4=VALEUR,CRIT3:MODA5=MIN-MAX,CRIT4,CRIT5:MODA6|MODA7=MIN-MAX
            $url .= "&filter_criterions=" . urlencode($attributes['filter_criterions']);
            $res['infos']['filter_criterions'] = $attributes['filter_criterions'];
        }
        // 13/02/2023 : application d'un scoring
        $res['infos']['scoring'] = '';
        if(isset($attributes['scoring']) && !empty($attributes['scoring'])) {
            $url .= "&scoring=" . (int) $attributes['scoring'];
            $res['infos']['scoring'] = $attributes['scoring'];
        }
	    // 07/06/2025 : on gère le minscore
	    $res['infos']['minscore'] = '';
	    if(isset($attributes['minscore']) && $attributes['minscore'] !== '') {
		    $url .= "&minscore=" . (int) $attributes['minscore'];
		    $res['infos']['minscore'] = $attributes['minscore'];
	    }
        // 13/02/2023 : application d'un tri dynamique
	    // Patch 07/06/2025 : cde n'était pas géré dans Bridge
        $res['infos']['sort_one'] = '';
        if(isset($attributes['sort_one']) && !empty($attributes['sort_one'])) {
	            $url .= "&brtri1=" . (int) $attributes['sort_one'];
            $res['infos']['sort_one'] = $attributes['sort_one'];
        }
        // 13/02/2023 : application d'un tri dynamique
        $res['infos']['sort_two'] = '';
        if(isset($attributes['sort_two']) && !empty($attributes['sort_two'])) {
            $url .= "&brtri2=" . (int) $attributes['sort_two'];
            $res['infos']['sort_two'] = $attributes['sort_two'];
        }
        // 13/02/2023 : application d'un tri dynamique
        $res['infos']['sort_three'] = '';
        if(isset($attributes['sort_three']) && !empty($attributes['sort_three'])) {
            $url .= "&brtri3=" . (int) $attributes['sort_three'];
            $res['infos']['sort_three'] = $attributes['sort_three'];
        }
        // 13/02/2023 : application d'un tri dynamique
        $res['infos']['duplicate_by'] = '';
        if(isset($attributes['duplicate_by']) && !empty($attributes['duplicate_by'])) {
            $url .= "&duplicate_by=" . $attributes['duplicate_by'];
            $res['infos']['duplicate_by'] = $attributes['duplicate_by'];
        }

        // Support des requetes minimalistes pour les points sur la carte
        if(isset($attributes['minimal_select']) && !empty($attributes['minimal_select'])) {
            $url .= "&minimalSelect=1" ;
        }

        // 23/01/2023 : Nouveau : possibilité de passer es attributes pour les listes reliées à des composants(fiches à proximité et fiches associées)
        if(isset($attributes['relatedList']) && $attributes['relatedList'] == true && isset($attributes['relatedListBlocksEntity']) && !empty($attributes['relatedListBlocksEntity']) && isset($attributes['relatedListBlocksId']) && !empty($attributes['relatedListBlocksId']) ) {
            $url .= "&rl=1&rl_ent=" .  $attributes['relatedListBlocksEntity'] . "&rl_id=" . $attributes['relatedListBlocksId'];
        }
        // 23/01/2023 : Nouveau : on exclut le code de produit d'origine dont on veut extraire les liés
        $res['infos']['relatedProductCode'] = '';
        $res['infos']['excluded_codes'] = '';
        if(isset($attributes['relatedProductCode']) && !empty($attributes['relatedProductCode'])) {
            if(is_array($attributes['relatedProductCode']))
                $attributes['relatedProductCode'] = implode('|', $attributes['relatedProductCode']);
            $url .= "&excluded_codes=" . $attributes['relatedProductCode'];
            $res['infos']['relatedProductCode'] = $attributes['relatedProductCode'];
            $res['infos']['excluded_codes'] = $attributes['relatedProductCode'];
        }

        if(isset($attributes['excluded_codes']) && !empty($attributes['excluded_codes']) && strpos($url, 'excluded_codes') === false) {
            if(is_array($attributes['excluded_codes']))
                $attributes['excluded_codes'] = implode('|', $attributes['excluded_codes']);
            $url .= "&excluded_codes=" . $attributes['excluded_codes'];
            $res['infos']['excluded_codes'] = $attributes['excluded_codes'];
        }

        // AJout 30/04/2024 : on passe l'id du site pour les statistiques
        $siteId = $this->clientParams->getSiteBridge();
        $url .= "&siteId=" . $siteId;


        // MàJ 23/03/2023
        // On n'ajoute les paramèters de paginaiton qu'à la toute fin de manière à obtenir une URL qui identifie le contenu d'une liste de manière unique - quelle que soit la pagination (pour l'unique identifier)
        $urlWithoutPaginationParams = $url;
        $url .= $urlPaginationParams;

        if(isset($_REQUEST['dncdebug']) && $_REQUEST['dncdebug'] == 'PAGINATION')
            echo "urlWithoutPaginationParams : $urlWithoutPaginationParams<br\>\n";

        if(isset($_REQUEST['dncdebug']) && $_REQUEST['dncdebug'] == 'Y')
            echo "<a href='".$url."' target='_blank' class='uk-button uk-button-primary'>Détails de liste</a>";


        try {
            // 11/04/2024 : implémentation d'un système de cache volatile fichier
            $tmpData = $this->requestService->cachedFileGetContent($url);
        } catch (\Exception $e) {
            $res['message'] = $e->getMessage();
            return $res;
        }
        $data = null;

        if (!empty($tmpData)) {
            $tmpDataDecoded = json_decode($tmpData);
            if ($tmpDataDecoded->success && $tmpDataDecoded->data) {
                $data = $tmpDataDecoded->data;
            } else {
                $res['message'] = $tmpDataDecoded->message;
                return $res;
            }
        } else {
            $res['message'] = 'Échec de la récupération de la liste - 1 ';
            if(isset($_REQUEST['dncdebug']) && $_REQUEST['dncdebug'] == 'CACHE') {
                print_r($tmpData);
                die('aiaiaia');
            }

            return $res;
        }


        if (!is_object($data)) {
            $res['message'] = "ERREUR d'appel à Bridge : aucune donnée valide renvoyée";
            return $res;
        }

        if (!isset($data->selection) || !isset($data->selection->results) || !isset($data->selection->results->products ) || !is_array($data->selection->results->products)) {
            $res['message'] = "ERREUR d'appel à Bridge : aucune donnée valide renvoyée - CODE 2";
            return $res;
        }

        // Inutile => devoiement Wordpress
        // Mis en commentaire le 07/04/2022 par ND après avoir vérifié que ce n'était plus utilisé
        // $tabProductsPosts = BridgeUtils::getPostIdsFromProductList($data->selection->results->products, $lang );

        // Nombre total de fiches
        $totalProducts = $data->selection->results->total;
        $totalproducts = $totalProducts; // Compatibilite
        $res['infos']['totalProducts'] = $totalProducts;
        $res['total'] = $totalProducts;
        $res['infos']['productCodes'] = $productCodes;

        // On corrige la variable limitPerPage par rapport à ce qu'il y avait dans Bridge
        if(isset($data->parameters) && isset($data->parameters->WEBLIST_MAX_ITEMS) && isset($data->parameters->WEBLIST_MAX_ITEMS->value))
            $limitPerPage = (int) $data->parameters->WEBLIST_MAX_ITEMS->value;

        $res['infos']['limitPerPage'] = $limitPerPage;

        // Si on a pas passé de paramètre de page dans l'URL, on va quand même recalculer sur quelle page on est à partir du paramètre first
        if(empty($res['infos']['currentPage'] )) {
            $currentPage = (round(((int) $first - 1) / (int) $limitPerPage)) + 1 ;
            $res['infos']['currentPage'] = $currentPage;
            $res['infos']['debug'][] = "ON RECALCULE currentPage à partir de first $first et limitperpage : $limitPerPage ";
        }

        // Maintenant on va préparer une variable stockant le dernier numéro de page pour la pagination
        $res['infos']['lastPage'] = ceil( $totalProducts / $limitPerPage );

        // Préparation du permalink
	    $permalink = $this->clientContentService->getListPermalink($data);

        if ($permalink !== '' && substr($permalink, 0 , 1) === '/') $permalink = substr($permalink, 1);
        $res['infos']['permalink'] = $permalink;
        $res['infos']['linkToMainSection'] = false;

        // Feat 24.10.2022 : ici on gère les liens dynamiquesvers les rubriques principales
        // TODO : optimiser les performances en précalculant le permalien de la rubrique principale principal dans les post meta des fiches lors de l'import
        if(strlen($permalink) >= 19 && mb_strpos( $permalink, 'RUBRIQUE_PRINCIPALE') !== false) {
            $res['infos']['linkToMainSection'] = true;
        }

        // Patch 26/05/2023 : s'il n'existe pas de post correspondant, on exclut la fiche de la liste
        if(count($data->selection->results->products) > 0 ) {
            for ($i = count($data->selection->results->products) - 1; $i >= 0 ; $i--) {
                $tmpPostId = $this->clientContentService->getPostIdFromProductCode($data->selection->results->products[$i]->productCode);
                if ($tmpPostId === false) {
                    array_splice($data->selection->results->products, $i, 1) ;
                }
            }
        }
        // Fin patch 26.05.2023

        // On injecte les liens vers les fiches de details et on met à jour la prorpiété isInSession pour dire si la fiche est dans le carnet de voyages courant
        if(count($data->selection->results->products) > 0 ) {
            for($i = 0 ; $i < count($data->selection->results->products) ; $i++) {

                // Evolution 22.10.2022 : lien vers la Rubrique principale
                $link = '';
                if($res['infos']['linkToMainSection'] == true) {
                    $link = $this->clientContentService->getMainTermLinkFromProductCode($data->selection->results->products[$i]->productCode, $data->selection->results->products[$i]->slug);
                } else {
                    $link = $this->clientContentService->getProductStandardLink($data->selection->results->products[$i]->productCode, $data->selection->results->products[$i]->slug, $permalink);
                }
                $data->selection->results->products[$i]->link = $link;

                // On insère l'information que la fiche se trouve dans le carnet de voyages ou non
                $isInSession = false;
                if(!empty($_SESSION) && !empty($_SESSION["cv"])) {
                    foreach ($_SESSION["cv"] as $f) {
                        if ($f["id"] == $data->selection->results->products[$i]->productCode) {
                            $isInSession = true;
                            break;
                        }
                    }
                }
                $geoSearch = false;
                $geoDistance = 0;
                $geoCity = '';
                // calcul de la distance au point de recherche
                if(!empty($_SESSION) && !empty($_SESSION["bridge_geolat"]) && !empty($data->selection->results->products[$i]->latitude) && !empty($data->selection->results->products[$i]->longitude)) {
                    $geoLat = (float) $_SESSION['bridge_geolat'];
                    $geoLon = (float) $_SESSION['bridge_geolon'];
                    $geoSearch = true;
                    $geoCity = $_SESSION['bridge_geocity'] ;
                    $geoDistance = $this->bridgeBlockService->distance($data->selection->results->products[$i]->latitude, $data->selection->results->products[$i]->longitude, $geoLat, $geoLon );
                }

                $data->selection->results->products[$i]->isInSession = $isInSession;
                $data->selection->results->products[$i]->geoSearch = $geoSearch;
                $data->selection->results->products[$i]->geoDistance = $geoDistance;
                $data->selection->results->products[$i]->geoCity = $geoCity;
            }
        }

        // ND 07.11.23 patch sessions bloquantes
        // session_write_close();

        // proprété inutile laissée pour des raisons historiques
        $res['infos']['cityComboDynList'] = null;

        // On récupère les paramètres spécifiques
        $params = $this->formatParametersListe($data->parameters);

        // Paramètres de modèles
        $bridgeModeleListe = '';
        if(isset($data->listTemplate))
            $bridgeModeleListe = $data->listTemplate;
        $bridgeModeleListeitem = '';
        if(isset($data->listItemTemplate))
            $bridgeModeleListeitem = $data->listItemTemplate;
        if ($bridgeModeleListe == '') $bridgeModeleListe = 'liste1';
        if ($bridgeModeleListeitem == '') $bridgeModeleListeitem = 'carte1';

        // recalage erreur d'orthographe
        $bridgeModeleListe = str_replace('caroussel', 'carousel', $bridgeModeleListe);

        // Feat 25/05/2022 : gestion des attributs bridgeModeleListe et bridgeModeleListeitem dans le shortcode pour overrider ce qui est défini dans la liste
        if(isset($attributes['template']) && !empty($attributes['template'])) {
            $res['infos']['bridgeModeleListe'] = $attributes['template'];
        } else {
            $res['infos']['bridgeModeleListe'] = $bridgeModeleListe;
        }

        // ND : NE PAS CORRIGER LE CAMELCASE SVP
        if(isset($attributes['item_template']) && !empty($attributes['item_template'])) {
            $bridgeModeleListeitem = $attributes['item_template'];
        }
        $res['infos']['bridgeModeleListeitem'] = $bridgeModeleListeitem;

        // Paramètres Carousel
        if ($bridgeModeleListe == "carousel" || $bridgeModeleListe == "carrousel") {
            $res['parameters']['numberItems'] = (isset($params->WEBLIST_CAROUSEL_MAX_ITEMS) && $params->WEBLIST_CAROUSEL_MAX_ITEMS != '') ? $params->WEBLIST_CAROUSEL_MAX_ITEMS : '';
            $res['parameters']['showImage'] = (isset($params->WEBLIST_CAROUSEL_SHOW_IMAGE) && $params->WEBLIST_CAROUSEL_SHOW_IMAGE == 1) ? true : false;
            $res['parameters']['defaultImage'] = (isset($params->WEBLIST_CAROUSEL_DEFAULT_IMAGE) && $params->WEBLIST_CAROUSEL_DEFAULT_IMAGE != '') ? $params->WEBLIST_CAROUSEL_DEFAULT_IMAGE : '';
            $res['parameters']['paginationType'] = (isset($params->WEBLIST_PAGINATION_TYPE) && $params->WEBLIST_PAGINATION_TYPE != '') ? $params->WEBLIST_PAGINATION_TYPE : '';
            $res['parameters']['numberColumn'] = (is_object($params->WEBLIST_CAROUSEL_NUMBER_DISPLAYED_PHOTOS) && is_object($params->WEBLIST_CAROUSEL_NUMBER_DISPLAYED_PHOTOS)) ? $params->WEBLIST_CAROUSEL_NUMBER_DISPLAYED_PHOTOS : '';
            $res['parameters']['spacingColumn'] = (isset($params->WEBLIST_CAROUSEL_PHOTO_SPACING) && $params->WEBLIST_CAROUSEL_PHOTO_SPACING != '') ? $params->WEBLIST_CAROUSEL_PHOTO_SPACING : '';
            $res['parameters']['displayCvButton'] = (isset($params->WEBLIST_CAROUSEL_DISPLAY_BUTTON) && $params->WEBLIST_CAROUSEL_DISPLAY_BUTTON == 1);
            $res['parameters']['displayListTitle'] = (isset($params->WEBLIST_CAROUSEL_AFFICHER_TITRE) && $params->WEBLIST_CAROUSEL_AFFICHER_TITRE == 1);
            $res['parameters']['titleTag'] =  (isset($params->WEBLIST_CAROUSEL_TITLE_TAG) && $params->WEBLIST_CAROUSEL_TITLE_TAG != '') ? $params->WEBLIST_CAROUSEL_TITLE_TAG : 'h1';
            $res['parameters']['displayDescription'] = (isset($params->WEBLIST_CAROUSEL_AFFICHER_DESCRIPTION) && $params->WEBLIST_CAROUSEL_AFFICHER_DESCRIPTION == 1);
            $res['parameters']['displayInContainer'] = (isset($params->WEBLIST_CAROUSEL_AFFICHER_CONTAINER) && $params->WEBLIST_CAROUSEL_AFFICHER_CONTAINER == 1);

            // Navigation
            if (isset($params->WEBLIST_CAROUSEL_NAVIGATION) && $params->WEBLIST_CAROUSEL_NAVIGATION != '') :
                switch ($params->WEBLIST_CAROUSEL_NAVIGATION):
                    case 'AUCUN':
                        $res['parameters']['arrow'] = false;
                        $res['parameters']['dots'] = false;
                        break;
                    case 'FLECHE':
                        $res['parameters']['arrow'] = true;
                        $res['parameters']['dots'] = false;
                        break;
                    case 'PUCE':
                        $res['parameters']['arrow'] = false;
                        $res['parameters']['dots'] = true;
                        break;
                    case 'FLECHE_ET_PUCE':
                        $res['parameters']['arrow'] = true;
                        $res['parameters']['dots'] = true;
                        break;
                    default:
                        $res['parameters']['arrow'] = true;
                        $res['parameters']['dots'] = true;
                        break;
                endswitch;
            endif;

            // Paramètres du carousel
            $paramsCarousel = '';
            $paramsCarousel .= (isset($params->WEBLIST_CAROUSEL_CENTERED) && $params->WEBLIST_CAROUSEL_CENTERED == 1) ? 'center:true;' : '';
            $paramsCarousel .= (isset($params->WEBLIST_CAROUSEL_AUTOPLAY) && $params->WEBLIST_CAROUSEL_AUTOPLAY == 1) ? 'autoplay:true;' : '';
            $paramsCarousel .= (isset($params->WEBLIST_CAROUSEL_SETS) && $params->WEBLIST_CAROUSEL_SETS == 1) ? 'sets:true;' : '';
            $paramsCarousel .= (isset($params->WEBLIST_CAROUSEL_FINITE) && $params->WEBLIST_CAROUSEL_FINITE == 1) ? 'finite: true;' : '';
            $paramsCarousel .= (isset($params->WEBLIST_CAROUSEL_AUTOPLAY_INTERVAL) && $params->WEBLIST_CAROUSEL_AUTOPLAY_INTERVAL != '') ? 'autoplay-interval:' . ($params->WEBLIST_CAROUSEL_AUTOPLAY_INTERVAL * 1000) . ';' : '';
            $res['parameters']['paramsCarousel'] = $paramsCarousel;

        } else {
            // Paramètres des modèles de liste
            $res['parameters']['numberItems'] = (isset($params->WEBLIST_MAX_ITEMS) && $params->WEBLIST_MAX_ITEMS != '') ? $params->WEBLIST_MAX_ITEMS : '';
            $res['parameters']['showImage'] = (isset($params->WEBLIST_SHOW_IMAGE) && $params->WEBLIST_SHOW_IMAGE != 1) ? false : true;
            $res['parameters']['defaultImage'] = (isset($params->WEBLIST_DEFAULT_IMAGE) && $params->WEBLIST_DEFAULT_IMAGE != '') ? $params->WEBLIST_DEFAULT_IMAGE : '';
            $res['parameters']['paginationType'] = (isset($params->WEBLIST_PAGINATION_TYPE) && $params->WEBLIST_PAGINATION_TYPE != '') ? $params->WEBLIST_PAGINATION_TYPE : '';
            $res['parameters']['numberColumn'] = (isset($params->WEBLIST_NUMBER_COLUMNS) && $params->WEBLIST_NUMBER_COLUMNS != '') ? $params->WEBLIST_NUMBER_COLUMNS : '';
            $res['parameters']['spacingColumn'] = (isset($params->WEBLIST_COLUMN_SPACING) && $params->WEBLIST_COLUMN_SPACING != '') ? $params->WEBLIST_COLUMN_SPACING : '';
            $res['parameters']['displayCvButton'] = (isset($params->WEBLIST_DISPLAY_BUTTON) && $params->WEBLIST_DISPLAY_BUTTON == 1);
            $res['parameters']['displayRelNextPrev'] = (isset($params->WEBLIST_REL_NEXT_PREV) && $params->WEBLIST_REL_NEXT_PREV == 1);
            $res['parameters']['displayListTitle'] = (isset($params->WEBLIST_AFFICHER_TITRE) && $params->WEBLIST_AFFICHER_TITRE == 1);
            $res['parameters']['titleTag'] =  (isset($params->WEBLIST_TITLE_TAG) && $params->WEBLIST_TITLE_TAG != '') ? $params->WEBLIST_TITLE_TAG : 'h1';
            $res['parameters']['displayDescription'] = (isset($params->WEBLIST_AFFICHER_DESCRIPTION) && $params->WEBLIST_AFFICHER_DESCRIPTION == 1);
            $res['parameters']['displayImage'] = (isset($params->WEBLIST_AFFICHER_IMAGE) && $params->WEBLIST_AFFICHER_IMAGE == 1);
            $res['parameters']['displayInContainer'] = (isset($params->WEBLIST_AFFICHER_CONTAINER) && $params->WEBLIST_AFFICHER_CONTAINER == 1);
            $res['parameters']['modeMultiMoteurs'] = (isset($params->WEBLIST_MOTEUR_MODE_MULTI_MOTEURS) && $params->WEBLIST_MOTEUR_MODE_MULTI_MOTEURS == 'OUI');
            $res['parameters']['multiMoteursIndexfiltreSwitch'] = (isset($params->WEBLIST_MULTIMOTEURS_INDEX_FILTRE_SWITCH)) ? $params->WEBLIST_MULTIMOTEURS_INDEX_FILTRE_SWITCH: '1';
            $res['parameters']['multiMoteursAfficherSections'] = (isset($params->WEBLIST_MULTIMOTEURS_AFFICHER_SECTIONS) && $params->WEBLIST_MULTIMOTEURS_AFFICHER_SECTIONS == 'OUI');

            // Normalisation du parametre  numberColumn
            if(!isset( $res['parameters']['numberColumn']) || !is_object( $res['parameters']['numberColumn']))
                $res['parameters']['numberColumn'] = new \stdClass();

            if(!isset( $res['parameters']['numberColumn']->default ))
                $res['parameters']['numberColumn']->default = '2';
            if(!isset( $res['parameters']['numberColumn']->xl ))
                $res['parameters']['numberColumn']->xl = $res['parameters']['numberColumn']->default;
            if(!isset( $res['parameters']['numberColumn']->l ))
                $res['parameters']['numberColumn']->l = $res['parameters']['numberColumn']->default;
            if(!isset( $res['parameters']['numberColumn']->m ))
                $res['parameters']['numberColumn']->m = $res['parameters']['numberColumn']->default;
            if(!isset( $res['parameters']['numberColumn']->s ))
                $res['parameters']['numberColumn']->s = $res['parameters']['numberColumn']->default;
            if(!isset( $res['parameters']['numberColumn']->xs ))
                $res['parameters']['numberColumn']->xs = $res['parameters']['numberColumn']->default;

        }

        // Paramètres des modèles et on fixe des paramètre au cas ou
        if($res['parameters']['numberItems'] != '') $res['parameters']['limitPerPage'] = $res['parameters']['numberItems'];
        if($res['parameters']['paginationType'] == '') $res['parameters']['paginationType'] = 'INFINITE_SCROLL'; // 'Pagination' 'Aucune pagination'

        // On génère un identifiant aléatoire pour cette instance de liste (utilisé pour isoler chaque liste en JS)
        // MàJ  03/07/2022 : non, on a besoin que cet identifiant soit prédictible : on met l'ID de liste
        // MàJ 08/03/2023 : si, dans le sshortcodes, si plusieurs pagination bouton dans la même page, l'identifiant doit être unique et je ne retrouve pas l'endroit où ça devait être prédictible
        // MàJ 23/03/2023 : certes mais on besoin d'un id prédictible car il est utilisé dans l'infinite scroll et la page suivante chargée en ajax place l'identifiant unique dans la pagination et l'infinite s'arête tout de suite si l'ID change
        $res['parameters']['uniqueIdentifier'] = md5($urlWithoutPaginationParams) ; // $this->bridgeBlockService->generateRandomString(); // $data->id ; // $this->bridgeBlockService->generateRandomString();


        // S'il n'y a aucun produit avec les critères sélectionnés, on affiche la liste avec un message (AJAX)
        if (empty($data->selection->results->products)) {
            $res['message'] = 'Aucun résultat';
        }
        $res['data'] = $data;
        $res['success'] = true;

        // On transforme les propriété infos et parameters en objets
        $res['parameters'] = json_decode(json_encode($res['parameters']));
        $res['infos'] = json_decode(json_encode($res['infos']));

        return $res;
    }


    /**
     * Appelle le contrôleur Bridge et récupère les données d'une carte dans la langue souhaitée, avec tous ses paramètres
     * et l'arborescence de ses items et subitems, avec les paramètres de ceux-ci et les produits qui leur sont associés
     *
     * @param $attributes : tablea associatif de paramètres d'affichage
     *
     * @return array
     */
    public function getMapData($attributes) {
        // prepareMapDataForRender

        $res = array(
            'success' => false,
            'message' => '',
            'total' => null,
            'data' => null,
            'infos' => array(),
            'parameters' => array()
        );

        $bridgeCredentials = $this->clientParams->getBridgeCredentials();
        $urlBridge = $bridgeCredentials->urlBridge;

        if(empty($urlBridge)) {
            $res['message'] = "L'URL de Bridge n'est pas définie - contrôlez les paramètres du plugin";
            return $res;
        }

        if(!is_array($attributes) || empty($attributes['id'])){
            $res['message'] = 'Le paramètre id est obligatoire dans le shortcode';
            return $res;
        }

        // Variable utilisée à différents endroits du code
        $webMapId = $attributes['id'];
        $res['infos']['mapId'] = $webMapId;

        $lang = $this->clientParams->getLanguage();
        $res['infos']['lang'] = $lang;
        $langUpper = strtoupper($lang);

        $url = $urlBridge . "/webmap/getWebMapData/" . $webMapId . "?lang=" . $lang;

        if (isset($_REQUEST['dncdebug']) && $_REQUEST['dncdebug'] == 'Y') {
            echo "<a href='".$url."' target='_blank' class='uk-button uk-button-primary'>Détails de liste</a>";
        }

        // 11/04/2024 : implémentation d'un système de cache volatile fichier
        $tmpData = $this->requestService->cachedFileGetContent($url);

        if(isset($attributes['maptmpl']) && !empty($attributes['maptmpl']))
            $mapTemplate = $attributes['maptmpl'];
        else
            $mapTemplate = '';

        if(isset($attributes['popup']) && !empty($attributes['popup']))
            $popupTemplate = $attributes['popup'];
        else
            $popupTemplate = '';

        if (!empty($tmpData)) {
            $tmpDataDecoded = json_decode($tmpData);
            if ($tmpDataDecoded->success && !empty($tmpDataDecoded->data)) {
                $mapData = $tmpDataDecoded->data;
                $res['data'] = $mapData;
                if (is_object($mapData)) {

                    if($mapTemplate=='')
                        $mapTemplate = $mapData->mapTemplate;

                    $res['infos']['mapTemplate'] = $mapTemplate;

                    if($popupTemplate=='')
                        $popupTemplate = $mapData->popupTemplate;

                    $res['infos']['popupTemplate'] = $popupTemplate;
                    $mapParameters = $mapData->parameters;

                    $res['parameters'] = $mapParameters;


                    $mapTest = $langUpper;
                    if ($langUpper === 'FR' && !empty($mapParameters->legendTitle)) {
                        $legendFrontTitle = $mapParameters->legendTitle;
                    } elseif ($langUpper !== 'FR' && isset($mapParameters->legendTitleTranslations) && !empty($mapParameters->legendTitleTranslations->$langUpper)) {
                        $legendFrontTitle = $mapParameters->legendTitleTranslations->$langUpper;
                    }else {$legendFrontTitle = 'Légende';}
                    $res['infos']['legendFrontTitle'] = $legendFrontTitle;
                } else {
                    $res['message'] = 'Erreur : données invalides 1 mapData n\'est pas un objet';
                    return $res;
                }
            } else {
                if (isset($tmpDataDecoded->message)) {
                    $res['message'] = 'Erreur 3 : ' . $tmpDataDecoded->message;
                    return $res;
                } else {
                    $res['message'] = 'Erreur : données invalides 2';
                    return $res;
                }
            }
        } else {
            $res['message'] = 'Erreur à la récupération des données';
            return $res;
        }

        if (empty($mapTemplate)) {
            $res['message'] = 'Vous devez choisir un template de carte';
            return $res;
        }

        if (isset($mapParameters->isDisplayingPopup) && $mapParameters->isDisplayingPopup == true && empty($popupTemplate)) {
            $res['message'] = 'Vous devez choisir un template de popup';
            return $res;
        }

        if (!isset($mapParameters->mapType)) {
            $res['message'] = 'Vous devez choisir un fond de carte';
            return $res;
        }

        if ($mapParameters->mapType === 'MAPTYPE_IGN' && empty($mapParameters->keyIgn)) {
            $res['message'] = 'Pour utiliser un fond de carte IGN, vous devez enregistrer une clef';
            return $res;
        }

        $res['success'] = true;
        return $res;
    }

    /**
     * Appelle le contrôleur Bridge et récupère un moteur dans la langue souhaitée,
     * avec le permalien de la liste indiquée en paramètre et l'id du moteur de cette liste
     *
     * @param array $attributes : paramètres d'affichage (idmoteur et idliste)
     *
     * @return array
     */
    public function getMoteurData($attributes) {
        $onlyMoteur = true;

        $res = array('success' => false);

        $bridgeCredentials = $this->clientParams->getBridgeCredentials();
        $urlBridge = $bridgeCredentials->urlBridge;

        // $customCss = $this->clientParams->getCustomCssURl();

        if (empty($urlBridge)) {
            $res['message'] = "L'URL de Bridge n'est pas définie - contrôlez les paramètres du plugin";
            return $res;
        }

        if (!is_array($attributes) || !isset($attributes['idmoteur'])){
            $res['message'] = 'Le paramètre idmoteur est obligatoire dans le shortcode' ;
            return $res;
        } else {
            $moteurId = $attributes['idmoteur'];
        }
        if (!is_array($attributes) || !isset($attributes['idliste'])){
            $res['message'] = 'Le paramètre idliste est obligatoire dans le shortcode' ;
            return $res;
        } else {
            $webListId = $attributes['idliste'];
        }
        $listInfos = array('webListId' => $webListId, 'productCodes' => '');

        // URL de destination du moteur
        $targetUrl = '';
        if (isset($attributes['target_url']) && !empty($attributes['target_url'])) {
            $targetUrl = $attributes['target_url'];
        }

        // Auto submit : false par défaut
        $autoSubmit = false;
        if (isset($attributes['auto_submit']) && !empty($attributes['auto_submit'])) {
            $autoSubmit = ($attributes['auto_submit'] == '1');
        }

        $lang = $this->clientParams->getLanguage();

        // Dans cette configuration, on sera toujours en train de récupérer la liste pour la première fois, on veut toutes les données
        $change = true;

        $url = $urlBridge . '/weblist/getProductsForList/' . $webListId . '?v2=1&engine=' . $moteurId . '&lang=' . $lang . '&change=' . $change;

        if (isset($_REQUEST['dncdebug']) && $_REQUEST['dncdebug'] == 'Y') {
            echo "<a href='".$url."' target='_blank' class='uk-button uk-button-primary'>Détails de liste</a>";
        }

        // 11/04/2024 : implémentation d'un système de cache volatile fichier
        $tmpData = $this->requestService->cachedFileGetContent($url);

        $data = null;
        $moteur = null;
        $webListEngineId = null;

        if (!empty($tmpData)) {
            $tmpDataDecoded = json_decode($tmpData);
            if ($tmpDataDecoded->success && $tmpDataDecoded->data) {
                $data = $tmpDataDecoded->data;
            } else {
                $res['message'] = 'Erreur lors de la récupération des données Bridge : ' .$tmpDataDecoded->message;
                return $res;
            }
        } else {
            $res['message'] = 'Échec lors de la récupération des données Bridge';
            return $res;
        }

        if (!empty($data)) {
            // On a reçu la liste entière, mais on veut juste l'id de la liste, son permalien et le moteur souhaité :
            // On prend le moteur passé en paramètre
            $moteur = $data->moteur;
            // et on récupère la liste des communes pour le filtre citycombodyn
            $cityComboDynList = null;

            if (!empty($moteur) && !empty($moteur->sections) && is_array($moteur->sections)) {
                foreach ($moteur->sections as $section) {
                    if (!empty($section->subSections)) {
                        foreach ($section->subSections as $subSection) {
                            if (!empty($subSection->filters) && is_array($subSection->filters)) {
                                foreach ($subSection->filters as $filter) {
                                    if ($filter->displayType === 'citycombodyn' && !empty($filter->cityComboDynList) && is_array($filter->cityComboDynList)) {
                                        foreach ($filter->cityComboDynList as $city) {
                                            $cityComboDynList .= (array_search($city, $filter->cityComboDynList) == 0 ? '' : ',') . $city;
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            $webListEngineId = null;

            if (isset($data->moteurOriginal)) {
                // quand le moteur passé en paramètre est le même que celui associé à la liste, on n'a noté que son id pour éviter de doublonner les données
                if (gettype($data->moteurOriginal) === 'integer') {
                    $webListEngineId = $data->moteurOriginal;
                } else {
                    $webListEngineId = $data->moteurOriginal->id;
                }
            }
            // On récupère le permalien de la liste
            $webListPermalink = $this->clientContentService->getListPermalink($data);

            // Quelques variables nécessaires à l'affichage des templates et au fonctionnement des scripts
            // nombre total de fiches
            $totalProducts = $data->selection->results->total;
            // nombre de fiches max par affichage
            $limitPerPage = 12;
            $bridgeAjaxUrl = $this->clientParams->getAjaxURL();

            $res['success'] = true;
            $res['listInfos'] = $listInfos;
            $res['moteur'] = $moteur;
            $res['infos'] = array(
                'webListEngineId' => $webListEngineId,
                'bridgeAjaxUrl' => $bridgeAjaxUrl,
                'totalProducts' => $totalProducts,
                'webListPermalink' => $webListPermalink,
                'cityComboDynList' => $cityComboDynList,
                'targetUrl' => $targetUrl,
                'autoSubmit' => $autoSubmit,
            );

            return $res;
        }
        return $res;
    }

    public function prepareMoteurDataForRender($data, $attributes = array()) {
        $lang = $this->clientParams->getLanguage();
        $langUpper = strtoupper($lang);

        // Pour que tout soit objet : plus simple
        $data = json_decode(json_encode($data));

        if(is_object($data) && (!isset($data->infos) || !is_object($data->infos))) {
            $data->infos = new \stdClass();
        }

        // On stocke les infos relatives à l'URL et aux filtres et tris passés en paramètre
        $uri_parts = explode('?', $_SERVER['REQUEST_URI'], 2);
        if(is_object($data) && isset($data->infos) && is_object($data->infos)) {
            $data->infos->currentUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://$_SERVER[HTTP_HOST]$uri_parts[0]" ;
            $data->infos->activeFilters = (isset($_REQUEST["braf"])) ? $_REQUEST["braf"] : "" ;
            $data->infos->activeItems = (isset($_REQUEST["brai"])) ? $_REQUEST["brai"] : "" ;
            $data->infos->httpRequest = $_REQUEST;
        }

        $bridgeParameters = json_decode(json_encode($this->clientParams->getBridgeParameters()));
        // On override les paramètres de balises HTML avec ce qu'on pourrait trouver dans $data->infos->attributes
        if(isset($attributes) && !empty($attributes)) {
            $attributes = json_decode(json_encode($attributes));
            $markupOptionGroups = $this->clientParams->getMarkupOptionGroups();
            foreach($markupOptionGroups as $groupKey => $group) {
                foreach($group as $attributeKey => $attribute) {
                    if(isset($attributes->$attributeKey) && !empty($attributes->$attributeKey)) {
                        $bridgeParameters->$attributeKey = $attributes->$attributeKey;
                    }
                }
            }
        }
        return array (
            'moteur' => $data->moteur,
            'moteurInfos' => $data->infos,
            'bridgeParameters' => $bridgeParameters,
            'listInfos' => $data->listInfos,
        );
    }

    /**
     * Appelée par le contrôleur AJAX par les vues de cartes interactives
     * Construit la liste des produits associés aux WebMapItems sélectionnés
     * @param array $webMapItems : tableau des id (string) des items sélectionnés
     * @param string $lang : langue dans laquelle la carte est affichée
     *
     * @return array $res avec bool success, string message, array data (objets produits avec les infos de base)
     */
    public function getWebMapProductsToShow($webMapItems, $lang) {
        $bridgeCredentials = $this->clientParams->getBridgeCredentials();
        $urlBridge = $bridgeCredentials->urlBridge;

        $res = array(
            'success' => false,
            'message' => null,
            'data' => null
        );

        if (empty($urlBridge)) {
            $res['message'] = "L'URL de Bridge n'est pas définie - contrôlez les paramètres du plugin";
            return $res;
        }

        if (isset($_REQUEST['dncdebug']) && $_REQUEST['dncdebug'] == 'Y') {
            echo "<a href='".$urlBridge."' target='_blank' class='uk-button uk-button-primary'>Détails de liste</a>";
        }

        // On a un tableau de string
        $webMapItemsInt = array();

        foreach ($webMapItems as $webMapItem) {
            $webMapItemsInt[] = (int) $webMapItem;
        }

        $items = implode(',', $webMapItemsInt);

        $productsToShow = array();

        $url = $urlBridge . '/webmap/getProductListByItems?items=' . $items . '&lang=' . $lang;

        // 11/04/2024 : implémentation d'un système de cache volatile fichier
        $tmpData = $this->requestService->cachedFileGetContent($url);

        if (!empty($tmpData)) {
            $tmpDataDecoded = json_decode($tmpData);

            if (isset($tmpDataDecoded->success) && $tmpDataDecoded->success && isset($tmpDataDecoded->data)) {
                $productsToShow = $tmpDataDecoded->data;

                if (isset($tmpDataDecoded->message)) {
                    $res['message'] = $tmpDataDecoded->message;
                }

            } elseif (isset($tmpDataDecoded->success) && !$tmpDataDecoded->success && isset($tmpDataDecoded->message)) {
                $res['message'] = $tmpDataDecoded->message;
                return $res;

            } else {
                $res['message'] = 'Erreur : données invalides';
                return $res;
            }

        } else {
            $res['message'] = "Erreur à la récupération des données";
            return $res;
        }

        $res['success'] = true;
        $res['data'] = $productsToShow;

        return $res;
    }

    /**
     * Appelée par le contrôleur AJAX par les vues de cartes interactives au clic sur un marqueur
     * Récupère les données du produit à afficher dans le pop-up
     * Et les renvoie directement via BridgeUtils::sendJson
     *
     * @param int $webMapId : l'id de la carte affichée
     * @param string $lang : la langue dans laquelle la carte est affichée (fr, de, en, etc.)
     * @param int $productCode : le code du produit à afficher
     *
     * @return array
     */
    public function getWebMapProductData($webMapId = '', $lang = 'fr', $productCode = '', $popupTemplate = '') {

        $res = array(
            'success' => false,
            'message' => '',
            'total' => null,
            'data' => null,
            'infos' => array(),
            'parameters' => array()
        );

        $bridgeCredentials = $this->clientParams->getBridgeCredentials();
        $urlBridge = $bridgeCredentials->urlBridge;

        if(empty($urlBridge)) {
            $res['message'] = "L'URL de Bridge n'est pas définie - contrôlez les paramètres du plugin";
            return $res;
        }
        if(empty($webMapId)) {
            $res['message'] = "Pas d\'id de map fourni";
            return $res;
        }
        if(empty($productCode)) {
            $res['message'] = "Pas de product code fourni";
            return $res;
        }
        if(empty($popupTemplate)) {
            $res['message'] = "Veuillez choisir un modèle de popup";
            return $res;
        }

        if (isset($_REQUEST['dncdebug']) && $_REQUEST['dncdebug'] == 'Y') {
            echo "<a href='".$urlBridge."' target='_blank' class='uk-button uk-button-primary'>Détails de liste</a>";
        }
        $res['infos']['webMapId'] = $webMapId;
        $res['infos']['popupTemplate'] = $popupTemplate;
        $res['infos']['lang'] = $lang;

        $url = $urlBridge . '/webmap/getProductData/' . $webMapId . '/' . $productCode . '?lang=' . $lang;

        // 11/04/2024 : implémentation d'un système de cache volatile fichier
        $tmpData = $this->requestService->cachedFileGetContent($url);

        if (!empty($tmpData)) {
            $tmpDataDecoded = json_decode($tmpData);
            if (isset($tmpDataDecoded->success) && $tmpDataDecoded->success == true && isset($tmpDataDecoded->data)) {
                // On insère la présence en CV et le lien
                $tmpDataDecoded->data->product->link = $this->clientContentService->getPostLinkFromProductCode($tmpDataDecoded->data->product->productCode);
                // On insère l'information que la fiche se trouve dans le carnet de voyages ou non
                $isInSession = false;
                // ND 07.11.23 patch sessions bloquantes
                // On démarre un session pour pouvoir enregistrer des paramètres de filtres de localisation
                if (session_status() === PHP_SESSION_NONE && !headers_sent()) {
                    session_start();
                }
                if(!empty($_SESSION) && !empty($_SESSION["cv"])) {
                    foreach ($_SESSION["cv"] as $f) {
                        if ($f["id"] == $tmpDataDecoded->data->product->productCode) {
                            $isInSession = true;
                            break;
                        }
                    }
                }
                // ND 07.11.23 patch sessions bloquantes
                // session_write_close();
                $tmpDataDecoded->data->product->isInSession = $isInSession;
                $popupData = $tmpDataDecoded->data->product;
                $res['data'] = $popupData;
            } elseif (isset($tmpDataDecoded->success) && !$tmpDataDecoded->success && isset($tmpDataDecoded->message)) {
                $res['message'] = "Erreur 06 : " . $tmpDataDecoded->message;
                return $res;
            } else {
                $res['message'] = "Erreur : données invalides";
                return $res;
            }
        } else {
            $res['message'] = "Erreur à la récupération des données";
            return $res;
        }
        $res['success'] = true;
        return $res;
    }


    /**
     * Appelle le contrôleur Bridge et récupère les données d'un emplacement publicitaire AnnonceSpace, avec les annonces associées actives
     *
     * @param $attributes : array, paramètres d'affichage
     *
     * @return array
     */
    public function getAnnonceSpaceData(array $attributes): array {
        $res = array(
            'success' => false,
            'message' => '',
            'data' => null,
            'infos' => array()
        );

        $bridgeCredentials = $this->clientParams->getBridgeCredentials();
        $urlBridge = $bridgeCredentials->urlBridge;

        if(empty($urlBridge)) {
            $res['message'] = "L'URL de Bridge n'est pas définie - contrôlez les paramètres du plugin";
            return $res;
        }

        if(!is_array($attributes) || empty($attributes['space'])){
            $res['message'] = 'Le paramètre space est obligatoire dans le shortcode';
            return $res;
        }
        $annonceSpaceId = $attributes['space'];
        $res['infos']['id'] = $annonceSpaceId;

        $lang = $this->clientParams->getLanguage();
        $res['infos']['lang'] = $lang;

        $url = $urlBridge . "/annonces/getAnnonceSpaceData/" . $annonceSpaceId . '?lang=' . $lang;

        if (isset($_REQUEST['dncdebug']) && $_REQUEST['dncdebug'] == 'Y') {
            echo "<a href='".$url."' target='_blank' class='uk-button uk-button-primary'>Détails de l'emplacement pub</a>";
        }

        try {
            // 11/04/2024 : implémentation d'un système de cache volatile fichier
            $tmpData = @file_get_contents($url);

            if (empty($tmpData)) {
                $res['message'] = 'Erreur à la récupération des données';
                return $res;
            }
            $tmpDataDecoded = json_decode($tmpData);
            if (!$tmpDataDecoded->success) {
                $res['message'] = $tmpDataDecoded->message ?? 'Erreur 2325 : données invalides';
                return $res;
            }
            $annonceSpaceData = $tmpDataDecoded->data;
            if (empty($annonceSpaceData)) {
                $res['success'] = true;
                $res['infos'] = json_decode((json_encode($res['infos'])));
                return $res;
            }
            if (!is_object($annonceSpaceData)) {
                $res['message'] = 'Erreur 3684 : données invalides';
                return $res;
            }

            // On transforme infos en objet
            $res['infos'] = json_decode((json_encode($res['infos'])));

            $res['data'] = $annonceSpaceData;
            $res['success'] = true;
            return $res;
        } catch (\Exception $e) {
            $res['message'] = $e->getMessage();
            return $res;
        }
    }


    /**
     * Retourne des données complètes de fiches associées (tableau d'objets products) à partir d'un objet product
     *
     * @param $fiche
     * @return array|false|mixed objet de réponse de l'API Bridge
     */
    public function getCoupledProducts($fiche) {

        $res = array();
        if(!is_object($fiche) || !isset($fiche->coupled) || empty($fiche->coupled))
            return $res;

        $codes = array();
        foreach($fiche->coupled as $tmpObj) {
            // Patch 15.05.2023 : on exclut les fiches qui n'existent pas dans le CMS
            $tmpProductCode = $tmpObj->coupledProductCode;
            $tmpPostId = $this->clientContentService->getPostIdFromProductCode($tmpProductCode);
            if ($tmpPostId !== false) {
                $codes[] = $tmpObj->coupledProductCode;
            }
            // Fin patch 15.05.2023
        }


        $lang = $this->clientParams->getLanguage();
        // Appel à Bridge
        $params = 'language=' . $lang .  '&completeResponse=1&first=1&max=5';
        $items = array('tag' => 'pro.productCode', 'value' => array('value' => implode('|',$codes)));
        $params .= '&items[]=' . urlencode(json_encode($items));

        // 10/11/2025 : optimisations d'appels Bridge
        if(empty($codes))
            return $res;

        /* 10/11/2025 : optimisations d'appels Bridge
        // die('rararara') ;
        $ts = microtime(true);
        $products = $this->bridgeApiService->callBridgeApi('GET', '/product/get_by_request', $params);
        $ts2 = microtime(true);
        print_r($products);
        echo "\n<br/>Temps appel Bridge pour fiches couplées : " . ($ts2 - $ts) . " secondes<br/>";
        */

        $bridgeCredentials = $this->clientParams->getBridgeCredentials();
        $urlBridge = $bridgeCredentials->urlBridge;
        $urlAppel = $urlBridge . '/weblist/getBaseProductsInfos?product_codes=' . implode('|',$codes) . '&lang=' . $lang . '&first=1&max=99';
        $products = $this->requestService->cachedFileGetContent($urlAppel);
        $products = json_decode($products);
        if(!empty($products)) {
            if(isset($products->success) && $products->success && isset($products->products)) {
                $res = $products->products;
            }
        }

        return $res;
    }

    /**
     * getCoupledProducts
     * Retourne un tableau d'objets products des fiches associées à l'objet product passé, enrichis des prioriété photo et shortComment
     * @param object $fiche Objet product issu de l'API Bridge
     * @return array|false|mixed
     */
    public function getCoupledProductsData($fiche) {

        $res = array();
        $coupled = $this->getCoupledProducts($fiche);
        if (isset($coupled) && isset($coupled->products) && is_array($coupled->products) && count($coupled->products) > 0) {
            // On récupère les critères photos via un bloc dédié à l'import des photos qu'on définit dans les paramètres du plugin
            for( $i = 0 ; $i < count($coupled->products) ; $i++) {
                $coupled->products[$i]->photo = '';
            }
            $blockPhoto = $this->clientParams->getIdBlockPhotos();

            if (!empty($blockPhoto)) {
                // 12/11/2025 : on met cet appel en cache car le bloc photo ne change pas souvent
                $blockContent = $this->bridgeApiService->callBridgeApi('GET', '/api/blocks/' . $blockPhoto,array(), 20, true, 1440);
				// var_dump($blockContent);
				$blockContent = json_decode(json_encode($blockContent));
	            //var_dump($blockContent);
				//die('tt');
				if(empty($blockContent)) {
					throw new \Exception('Erreur 1 : bloc de photos introuvable (id ' . $blockPhoto . ')' );
				}
                $criteresPhotos = $this->bridgeBlockService->extractBlockCriterions($blockContent);

                for( $i = 0 ; $i < count($coupled->products) ; $i++) {
                    if (isset($criteresPhotos) && is_array($criteresPhotos) && count($criteresPhotos) > 0) {
                        $testImage = $this->bridgeBlockService->filterModalities($coupled->products[$i]->modalities, $criteresPhotos[0]);
                        if(empty($testImage) && count($criteresPhotos) > 0)
                            $testImage = $this->bridgeBlockService->filterModalities($coupled->products[$i]->modalities, $criteresPhotos[1]);

                        // print_r($coupled->products[$i]);
                        if (is_array($testImage) && count($testImage) > 0 && isset($testImage[0]->value) && trim($testImage[0]->value) != '') {
                            $photo = "https://" . $testImage[0]->value;
                            $photo = str_replace("https://http://", "http://", $photo);
                            $photo = str_replace("https://https://", "https://", $photo);
                            $coupled->products[$i]->photo = $photo;
                        }
                    }
                    if(!empty($coupled->products[$i]->comment)) {
                        if(mb_strlen($coupled->products[$i]->comment) > 150) {
                            $coupled->products[$i]->shortComment = mb_substr($coupled->products[$i]->comment,0,150) . '...';
                        } else {
                            $coupled->products[$i]->shortComment = $coupled->products[$i]->comment ;
                        }
                    } else {
                        $coupled->products[$i]->shortComment = '';
                    }

                    // Lien
                    $link = $this->clientContentService->getPostLinkFromProductCode($coupled->products[$i]->productCode);
                    $coupled->products[$i]->link = $link;
                }
            }
            $res = $coupled->products;
        }
        return $res;
    }


    /**
     * Retourne les données d'une fiche SIT à partir du contexte (appelée depuis la page fiche)
     * @return mixed objet product issu de l'API Bridge
     */
    public function getDataFiche($productCode='', $idFicheBridge='', $langueForcee = '') {
        // Dans l'afichage de la page fiche, on a 2 appels consécutifs : l'un dans add_css_on_body puis dans signle_fiche_sit
        // Pour éviter de refaire un appel à l'API Bridge, on récupère les données de la fiche d'une variable globale
        $debugTrace[] = "Début de récupération des infos de fiche";
        if($productCode === '' && $idFicheBridge === '' && $langueForcee === '' && isset($GLOBALS['bridgeDataFiche']) && is_object($GLOBALS['bridgeDataFiche'])) {
            $debugTrace[] = 'Zone -1 : Un appel a déjà été fait : on récupère les données en cache depuis le GLOBALS pour éviter un double appel';
            return $GLOBALS['bridgeDataFiche'];
        }

        $debugTrace = array();
        $debugTrace[] = 'Zone 0 - Arguments passés à la fonction : productCode : ' . $productCode . ' - idFicheBridge : ' . $idFicheBridge ;
        // Récupération depuis le contexte
        if($idFicheBridge === '') {
            // Nota : sous Drupal, idFicheBridge n'est jamais vide
            $debugTrace[] = 'Zone 1 Récupération de la query_var idFicheBridge :' . $this->clientContentService->getQueryVar('idFicheBridge');
            $idFicheBridge = $this->clientContentService->getQueryVar('idFicheBridge');
        }
        $postId = null ;
        if($productCode === '') {
            // Nota : sous Drupal, productCode n'est jamais vide
            $postId = $this->clientContentService->getCurrentPostId();
            if (!empty($postId)) {
                $productCode = $this->clientContentService->getProductCodeFromPostId($postId);
                $debugTrace[] = 'Zone 2 - Le CMS a détecté un post courant : postID : ' . $postId . ' - en correspondance avec le productCode : ' . $productCode;

                if(empty($idFicheBridge)) {
                    $idFicheBridge = $this->clientContentService->getFicheIdFromPostId($postId);
                }
            }
        }
        $debugTrace[] = 'Zone 3 - a-t-on trouvé un id de fiche à ce stade ? idFicheBridge : ' . $idFicheBridge;

        // ajout 29/06/22 : si pas de modèle de fiche passé, on essaie de retrouver le modèle associé à la première weblist attaché à la fiche
        if(empty($idFicheBridge)) {
            $debugTrace[] = 'Zone 4 - on n\'a pas de modèle de fiche, on essaie de le retrouver via une weblist attachée à la fiche';
            if(!empty($productCode)) {
                $postId = $this->clientContentService->getPostIdFromProductCode($productCode);
                $debugTrace[] = 'Zone 5 - idFicheBridge ataché à la fiche courante : ' . $idFicheBridge;
            } else {
                $postId = $this->clientContentService->getCurrentPostId();
                if(!empty($postId)) {
                    $debugTrace[] = 'Zone 6 - On tente de récupérer le post CMS courant : postId : ' . $postId;
                }
            }

            if(isset($postId) && $postId !== false && !empty($postId) && empty($idFicheBridge)) {
                $idFicheBridge = $this->clientContentService->getFicheIdFromPostId($postId);
	            $debugTrace[] = 'Zone 7 - On récupère l\'id de fiche Bridge depuis la catégorie principale du post courant : idFicheBridge : ' . $idFicheBridge . ' - postId : ' . $postId;
            }
        }
        if (empty($productCode)) {
            $debugTrace[] = 'Zone 8 : N\'ayant pas pu récupérer de post courant via le mécanisme intégré de Wordpress, on récupère le post_id depuis le query_var fiche_sit (il doit contenir un productCode car sil avait contenu un slug ça aurait matché avant :' . $this->clientContentService->getQueryVar('fiche_sit');
            $productCode = $this->clientContentService->getQueryVar('fiche_sit');
        }

        $dataFiche = new \stdClass();

        if(!empty($productCode)) {
            $debugTrace[] = 'Zone 9 - A ce stde on a bien un productCode : ' . $productCode;
            $urlBridge = $this->clientParams->getBridgeUrl();

            if(empty($idFicheBridge)) {
                $debugTrace[] = 'Zone 10 - pas de modèle de fiche détecté à ce stade, flute, on affiche une erreur';
                echo "Erreur de paramétrage - pas de modèle de fiche";
            } else {
                $debugTrace[] = 'Zone 11 - On a toutes les données nécessaires pour récupérer les infos de Bridge : productCode : ' . $productCode . ' - idFicheBridge : ' . $idFicheBridge;
                $lang = $this->clientParams->getLanguage();
                // Nouveau 10/08/2024 : possibilité de forcer la langue pour récupérer du FR et faire une traduction automatique par exemple
                if($langueForcee !== '') $lang = $langueForcee;
                // Patch 13/03/2024 : on passe le numéro de site pour que les parmamètres au niveau site puissent petre pris en compte
                $siteId = $this->clientParams->getSiteBridge();
                $lurl = $urlBridge . '/webpage/getProductForFiche/' . $idFicheBridge . '?productCode=' . $productCode . "&lang=" . $lang . '&siteId=' . $siteId ;
                if(isset($_REQUEST['dncdebug']) && $_REQUEST['dncdebug']=="Y") echo '<a href="'.$lurl.'" class="uk-width-1-1 uk-button uk-button-primary uk-margin">Détails fiche</a>';

                // 11/04/2024 : implémentation d'un système de cache volatile fichier
                $dataFiche = $this->requestService->cachedFileGetContent($lurl);
                $dataFiche = json_decode($dataFiche);

                // On insère l'information que la fiche se trouve dans le carnet de voyages ou non
                $isInSession = false;
                if(!empty($dataFiche) && is_object($dataFiche) && property_exists($dataFiche, 'product') && is_object($dataFiche->product)) {
                    // ND 07.11.23 patch sessions bloquantes
                    if (session_status() === PHP_SESSION_NONE && !headers_sent()) {
                        session_start();
                    }
                    if(!empty($_SESSION) && !empty($_SESSION["cv"])) {
                        foreach ($_SESSION["cv"] as $f) {
                            if ($f["id"] == $dataFiche->product->productCode) {
                                $isInSession = true;
                                break;
                            }
                        }
                    }
                    $geoSearch = false;
                    $geoDistance = 0;
                    $geoCity = '';
                    // calcul de la distance au point de recherche
                    if(!empty($_SESSION) && !empty($_SESSION["bridge_geolat"]) && !empty($dataFiche->product->latitude) && !empty($dataFiche->product->longitude)) {
                        $geoLat = (float) $_SESSION['bridge_geolat'];
                        $geoLon = (float) $_SESSION['bridge_geolon'];
                        $geoSearch = true;
                        $geoCity = $_SESSION['bridge_geocity'] ;
                        $geoDistance = $this->bridgeBlockService->distance($dataFiche->product->latitude, $dataFiche->product->longitude, $geoLat, $geoLon );
                    }

                    $dataFiche->product->isInSession = $isInSession;
                    $dataFiche->product->geoSearch = $geoSearch;
                    $dataFiche->product->geoDistance = $geoDistance;
                    $dataFiche->product->geoCity = $geoCity;

                    // ND 07.11.23 patch sessions bloquantes
                    // session_write_close();
                }
                if(!empty($dataFiche)) {
                    // On traite les paramètres

                    if(isset($dataFiche->ficheTemplate) && empty($dataFiche->ficheTemplate)) $dataFiche->ficheTemplate = 'fiche1';
                    // $dataFiche->specificCssPath = !empty($dataFiche->specificCssPath) ? $dataFiche->specificCssPath : '';
                    // $dataFiche->specificJsPath = !empty($dataFiche->specificJsPath) ? $dataFiche->specificJsPath : '';


                    if(isset($dataFiche->parameters) && empty($dataFiche->parameters)) {
                        $dataFiche->parameters = new \stdClass();
                    }
                    $dataFiche->parameters->displayCvButton = (isset($dataFiche->parameters->FICHE_DISPLAY_BUTTON)  && isset($dataFiche->parameters->FICHE_DISPLAY_BUTTON->value) && $dataFiche->parameters->FICHE_DISPLAY_BUTTON->value == 1);
                    $dataFiche->parameters->displayToc = (isset($dataFiche->parameters->AFFICHER_SOMMAIRE)  && isset($dataFiche->parameters->AFFICHER_SOMMAIRE->value) && $dataFiche->parameters->AFFICHER_SOMMAIRE->value == 1);
                    $dataFiche->parameters->limitDescriptionFieldHeight = (isset($dataFiche->parameters->LIMITER_HAUTEUR_DESCRIPTIF)  && isset($dataFiche->parameters->LIMITER_HAUTEUR_DESCRIPTIF->value) && $dataFiche->parameters->LIMITER_HAUTEUR_DESCRIPTIF->value == 1);
                    $dataFiche->parameters->tocPosition = (isset($dataFiche->parameters->POSITION_SOMMAIRE)  && isset($dataFiche->parameters->POSITION_SOMMAIRE->value))?$dataFiche->parameters->POSITION_SOMMAIRE->value:'GAUCHE' ;
                    $dataFiche->parameters->sliderAutoplay = (isset($dataFiche->parameters->SLIDER_AUTOPLAY)  && isset($dataFiche->parameters->SLIDER_AUTOPLAY->value) && $dataFiche->parameters->SLIDER_AUTOPLAY->value == 1);
                    $dataFiche->parameters->hidePhoneWithXX = (isset($dataFiche->parameters->HIDE_PHONE)  && isset($dataFiche->parameters->HIDE_PHONE->value) && $dataFiche->parameters->HIDE_PHONE->value == 1);
                    $dataFiche->parameters->emailLinkType = (isset($dataFiche->parameters->EMAIL_LINK_TYPE)  && isset($dataFiche->parameters->EMAIL_LINK_TYPE->value))?$dataFiche->parameters->EMAIL_LINK_TYPE->value:'MAILTO' ;
                    $dataFiche->parameters->emailPrestSubject = (isset($dataFiche->parameters->EMAIL_PREST_SUBJECT)  && isset($dataFiche->parameters->EMAIL_PREST_SUBJECT->value))?$dataFiche->parameters->EMAIL_PREST_SUBJECT->value:'Demande de renseignements' ;
                    $dataFiche->parameters->emailPrestBody = (isset($dataFiche->parameters->EMAIL_PREST_BODY)  && isset($dataFiche->parameters->EMAIL_PREST_BODY->value))?$dataFiche->parameters->EMAIL_PREST_BODY->value:'' ;
                    $dataFiche->parameters->emailPrestBodyEnd = (isset($dataFiche->parameters->EMAIL_PREST_BODY_END)  && isset($dataFiche->parameters->EMAIL_PREST_BODY_END->value))?$dataFiche->parameters->EMAIL_PREST_BODY_END->value:'' ;
                    $dataFiche->parameters->emailFormAntispam = (isset($dataFiche->parameters->EMAIL_FORM_ANTISPAM)  && isset($dataFiche->parameters->EMAIL_FORM_ANTISPAM->value))?$dataFiche->parameters->EMAIL_FORM_ANTISPAM->value:'' ;
                    $dataFiche->parameters->recaptchav3_site_key = (isset($dataFiche->parameters->RECAPTCHAV3_SITE_KEY)  && isset($dataFiche->parameters->RECAPTCHAV3_SITE_KEY->value))?$dataFiche->parameters->RECAPTCHAV3_SITE_KEY->value:'' ;
                    $dataFiche->parameters->recaptchav3_secret_key = (isset($dataFiche->parameters->RECAPTCHAV3_SECRET_KEY)  && isset($dataFiche->parameters->RECAPTCHAV3_SECRET_KEY->value))?$dataFiche->parameters->RECAPTCHAV3_SECRET_KEY->value:'' ;
                    $dataFiche->parameters->recaptchav2_site_key = (isset($dataFiche->parameters->RECAPTCHAV2_SITE_KEY)  && isset($dataFiche->parameters->RECAPTCHAV2_SITE_KEY->value))?$dataFiche->parameters->RECAPTCHAV2_SITE_KEY->value:'' ;
                    $dataFiche->parameters->recaptchav2_secret_key = (isset($dataFiche->parameters->RECAPTCHAV2_SECRET_KEY)  && isset($dataFiche->parameters->RECAPTCHAV2_SECRET_KEY->value))?$dataFiche->parameters->RECAPTCHAV2_SECRET_KEY->value:'' ;
                    $dataFiche->parameters->emailTestEmailAddress = (isset($dataFiche->parameters->EMAIL_TEST_EMAIL_ADDRESS)  && isset($dataFiche->parameters->EMAIL_TEST_EMAIL_ADDRESS->value))?$dataFiche->parameters->EMAIL_TEST_EMAIL_ADDRESS->value:'' ;
                    $dataFiche->parameters->emailModeTest = (isset($dataFiche->parameters->EMAIL_MODE_TEST)  && isset($dataFiche->parameters->EMAIL_MODE_TEST->value) && $dataFiche->parameters->EMAIL_MODE_TEST->value == 1);

                    // Update du 23.01 : nouveaux templates fiches associées : on fait le rendu ici
                    $props = array_keys((array) $dataFiche->product->ficheBlocks);
                    foreach($props as $prop) {
                        $test = $dataFiche->product->ficheBlocks->$prop;
                        if(is_array($test)) {
                            for ($tmpIdx = 0; $tmpIdx < count($dataFiche->product->ficheBlocks->$prop); $tmpIdx++) {
                                $bloc = $dataFiche->product->ficheBlocks->$prop[$tmpIdx];
                                $specialComponent = false;
                                $specialComponentHtml = '';
                                $specialComponentData = null ;
                                $elevationData = '';
                                switch ($bloc->bridgeComponent) {
                                    case 'fiches_associees_carte':
                                    case 'fiches_associees_liste':
                                    case 'fiches_a_proximite':
                                        $specialComponent = true;
                                        $tmpSpecialComponent = $this->renderBridgeSpecialComponent($dataFiche->product, $bloc->bridgeComponent, $bloc);
                                        $specialComponentHtml = $tmpSpecialComponent->html;
                                        $specialComponentData = $tmpSpecialComponent->data;

                                        break;
                                    case 'profil_altimetrique':
                                        // enqueueScript('chart-js'); TODO : Faudra charger charts toutjours sous wp
                                        $elevationData = $this->getElevationData($dataFiche->product);
                                        break;
                                }
                                $dataFiche->product->ficheBlocks->$prop[$tmpIdx]->specialComponent = $specialComponent;
                                $dataFiche->product->ficheBlocks->$prop[$tmpIdx]->specialComponentHtml = $specialComponentHtml;
                                $dataFiche->product->ficheBlocks->$prop[$tmpIdx]->specialComponentData = $specialComponentData;
                                $dataFiche->product->ficheBlocks->$prop[$tmpIdx]->elevationData = $elevationData;
                            }
                        } elseif(is_object($test)) {
                            $specialComponent = false;
                            $specialComponentHtml = '';
                            $specialComponentData = null ;
                            switch ($test->bridgeComponent) {
                                case 'fiches_associees_carte':
                                case 'fiches_associees_liste':
                                case 'fiches_a_proximite':
                                    $specialComponent = true;
                                    $tmpSpecialComponent = $this->renderBridgeSpecialComponent($dataFiche->product, $test->bridgeComponent, $test);
                                    $specialComponentHtml = $tmpSpecialComponent->html;
                                    $specialComponentData = $tmpSpecialComponent->data;
                                    break;
                            }
                            $dataFiche->product->ficheBlocks->$prop->specialComponent = $specialComponent;
                            $dataFiche->product->ficheBlocks->$prop->specialComponentHtml = $specialComponentHtml;
                            $dataFiche->product->ficheBlocks->$prop->specialComponentData = $specialComponentData;
                        }
                    }
                    if($specialComponent) {
                        // echo $specialComponentHtml;
                        // die('fin : ' . $test->bridgeComponent);
                    }
                }
            }
        } else {
			$debugTrace[] = 'Zone 15 - pas de productCode';
        }

        if(isset($_REQUEST['dncdebug']) && strtolower($_REQUEST['dncdebug']) === 'datafiche') {
            echo implode("<br/>\n", $debugTrace);
        }
        return $dataFiche;
    }

    /**
     * Effectue le rendu HTML d'un composant spécial (avec appel imbriqué à Bridge)
     *
     * @param $product
     * @param $componentName
     * @param $blockObject
     *
     * @return string Résultat HTML du composant
     */
    public function renderBridgeSpecialComponent($product, $componentName, $blockObject) {
        $res = (object) array('data' => null, 'html' => '');
        // Cas particulier des fiches associées
        if($componentName == 'fiches_associees_carte' || $componentName == 'fiches_associees_liste') {
            $codes = array();
            $data = 'RIEN' ;
            if(is_object($product) && isset($product->coupled)) {
                foreach ($product->coupled as $tmpObj) {
                    // Vérification si filtre par type de couplage
                    if(isset($blockObject->parameters) && is_object($blockObject->parameters) && isset($blockObject->parameters->coupleType)
                        && !empty($blockObject->parameters->coupleType) && isset($tmpObj->coupleType)
                        && ( !is_array($blockObject->parameters->coupleType) && $blockObject->parameters->coupleType != $tmpObj->coupleType
                            || (is_array($blockObject->parameters->coupleType) && !in_array($tmpObj->coupleType, $blockObject->parameters->coupleType)))
                    )
                        continue;

                    // Patch 15.05.2023 : on exclut les fiches qui n'existent pas dans le CMS
                    $tmpProductCode = $tmpObj->coupledProductCode;
                    $tmpPostId = $this->clientContentService->getPostIdFromProductCode($tmpProductCode);
                    if ($tmpPostId !== false) {
                        $codes[] = $tmpObj->coupledProductCode;
                    }
                    // Fin patch 15.05.2023
                }
            }
            if(isset($_REQUEST['dncdebug']) && $_REQUEST['dncdebug'] === 'COUPLED') {
                die("DEBUG coupled. Code : " . implode(', ', $codes));
            }
            if (count($codes) > 0) {
                $attributes['id'] = -1; // Pour tromper l'ennemi
                $attributes['useRequestFilters'] = false;
                $attributes['first'] = 1;
                $attributes['max'] = 50;
                $attributes['change'] = 1;
                $attributes['product_codes'] = $codes;
                $attributes['relatedList'] = true; // Pour ne pas avoir à se base sur une weblist existante maiss
                $attributes['relatedListBlocksEntity'] = 'WebPageBlock';
                if (isset($blockObject->blockId))
                    $attributes['relatedListBlocksId'] = $blockObject->blockId; // WebPageBlock->id

                if(isset($blockObject->parameters) && is_object($blockObject->parameters) && isset($blockObject->parameters->listItemTemplate)  && is_object($blockObject->parameters->listItemTemplate) && isset($blockObject->parameters->listItemTemplate->gabaritName) && !empty($blockObject->parameters->listItemTemplate->gabaritName)) {
                    $attributes['item_template'] = $blockObject->parameters->listItemTemplate->gabaritName;
                }
                // On recherche si un maxi a été défini
                if(isset($blockObject->parameters) && is_object($blockObject->parameters) && isset($blockObject->parameters->parameters) && is_array($blockObject->parameters->parameters) && !empty($blockObject->parameters->parameters)) {
                    foreach($blockObject->parameters->parameters as $param) {
                        if(is_object($param) && isset($param->bridgeParameter) && is_object($param->bridgeParameter) && isset($param->value) && is_object($param->value)) {
                            if($param->bridgeParameter->key == 'WEBLIST_MAX_ITEMS' && !empty($param->value->value)) {
                                $attributes['max'] = $param->value->value ;
                            }
                        }
                    }
                }
                $data = $this->getListData($attributes);
            }
        }

        // Cas particulier des fiches associées
        if($componentName == 'fiches_a_proximite') {
            $attributes['id'] = -1; // Pour tromper l'ennemi
            $attributes['useRequestFilters'] = false;
            $attributes['first'] = 1;
            $attributes['max'] = 12;
            $attributes['change'] = 1;
            $attributes['relatedList'] = true; // Pour ne pas avoir à se base sur une weblist existante maiss
            $attributes['relatedListBlocksEntity'] = 'WebPageBlock';
            $attributes['relatedProductCode'] = $product->productCode;
            if(isset($blockObject->blockId))
                $attributes['relatedListBlocksId'] = $blockObject->blockId ; // WebPageBlock->id

            // Rayon par défaut : 10km
            $rayon = 10000;
            if(isset($blockObject->parameters) && is_object($blockObject->parameters) && isset($blockObject->parameters->rayon) && !empty($blockObject->parameters->rayon) && is_numeric($blockObject->parameters->rayon)) {
                $rayon = (int) $blockObject->parameters->rayon;
            }
            if(isset($blockObject->parameters) && is_object($blockObject->parameters) && isset($blockObject->parameters->selection) && is_object($blockObject->parameters->selection) && !empty($blockObject->parameters->selection->id)) {
                $attributes['selection_id'] = $blockObject->parameters->selection->id;
            }

            if(isset($blockObject->parameters) && is_object($blockObject->parameters) && isset($blockObject->parameters->listItemTemplate)  && is_object($blockObject->parameters->listItemTemplate) && isset($blockObject->parameters->listItemTemplate->gabaritName) && !empty($blockObject->parameters->listItemTemplate->gabaritName)) {
                $attributes['item_template'] = $blockObject->parameters->listItemTemplate->gabaritName;
            }
            // On recherche si un maxi a été défini
            if(isset($blockObject->parameters) && is_object($blockObject->parameters) && isset($blockObject->parameters->parameters) && is_array($blockObject->parameters->parameters) && !empty($blockObject->parameters->parameters)) {
                foreach($blockObject->parameters->parameters as $param) {
                    if(is_object($param) && isset($param->bridgeParameter) && is_object($param->bridgeParameter) && isset($param->value) && is_object($param->value)) {
                        if($param->bridgeParameter->key == 'WEBLIST_MAX_ITEMS' && !empty($param->value->value)) {
                            $attributes['max'] = $param->value->value ;
                        }
                    }
                }
                $attributes['selection_id'] = $blockObject->parameters->selection->id;
            }
            if(!empty($product->latitude) && !empty($product->longitude)) {
                $attributes['filter_lat'] = $product->latitude;
                $attributes['filter_lon'] = $product->longitude;
                $attributes['filter_rayon'] = $rayon;
            }

            $attributes['debugthis'] = 1;
            $data = $this->getListData($attributes);
        }

        if(is_array($data) && $data['success'] == true && is_object($data['data'])) {
            $res->data = $data;
            $res->data['mapOptions'] = $this->clientParams->getMapsDefaultSettings();
            $res->data['bridgeParameters'] = json_decode(json_encode($this->clientParams->getBridgeParameters()));

            $listData = $this->prepareListDataForRender($data);

            $res->html = $this->renderer->renderListeAjaxTemplate($listData);
            /*
            // On charge les JS et CSS de liste Bridge
            BridgeUtils::loadListScriptsAndStyles();
            ob_start();
            // $template = self::bridge_locate_template('templates-bridge/fiche/gabarits/' . $componentName . '.php');
            $template = self::bridge_locate_template('templates-bridge/ajax.php');
            include($template);
            $res->html = ob_get_clean();
            */
        } else {
            // print_r($data);
            // echo "PROBLEME dans le gabarit $componentName : pas de data";
        }
        return $res;
    }


    /**
     * prepareListDataForRender
     * Prépare un tableau contenant les données à passer au renderer de liste
     * @param array $data tableau ou objet data tel que renvoyé par BridgeUtils::getListData
     * @return array tableau associatif contenant les donnée sutilisées par les vues listes de rendu
     */
    public function prepareListDataForRender($data) {
        $lang = $this->clientParams->getLanguage();
        $langUpper = strtoupper($lang);

        // Pour que tout soit objet : plus simple
        $data = json_decode(json_encode($data));

        // On stocke les infos relatives à l'URL et aux filtres et tris passés en paramètre
        $uri_parts = explode('?', $_SERVER['REQUEST_URI'], 2);
        if(is_object($data) && isset($data->infos) && is_object($data->infos)) {
            $data->infos->currentUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://$_SERVER[HTTP_HOST]$uri_parts[0]" ;
            $data->infos->activeFilters = (isset($_REQUEST["braf"])) ? $_REQUEST["braf"] : "" ;
            $data->infos->activeItems = (isset($_REQUEST["brai"])) ? $_REQUEST["brai"] : "" ;
            $data->infos->activeSorts = (isset($_REQUEST["bras"])) ? $_REQUEST["bras"] : "" ;
            $data->infos->sortDirections = (isset($_REQUEST["brsd"])) ? $_REQUEST["brsd"] : "" ;
            $data->infos->httpRequest = $_REQUEST;
        }


        $bridgeParameters = json_decode(json_encode($this->clientParams->getBridgeParameters()));
        // On override les paramètres de balises HTML avec ce qu'on pourrait trouver dans $data->infos->attributes
        if(isset($data->infos->attributes) && is_object($data->infos->attributes)) {
            $markupOptionGroups = $this->clientParams->getMarkupOptionGroups();
            foreach($markupOptionGroups as $groupKey => $group) {
                foreach($group as $attributeKey => $attribute) {
                    if(isset($data->infos->attributes->$attributeKey) && !empty($data->infos->attributes->$attributeKey)) {
                        $bridgeParameters->$attributeKey = $data->infos->attributes->$attributeKey;
                    }
                }
            }
        }


        // On calcule les url des pages suivantes et précédentes pour le rel=next/prev
        $separator = '&';
        if(empty($bridgeParameters->paginationUrlParams)) {
            $separator = '?';
        }

        if($data->infos->currentPage == 2 ) {
            // Pas de paramètre brpa pour la première page
            $data->infos->prevPageURL = $bridgeParameters->currentUrl . $bridgeParameters->paginationUrlParams ;
        } elseif($data->infos->currentPage > 2 ) {
            $data->infos->prevPageURL = $bridgeParameters->currentUrl . $bridgeParameters->paginationUrlParams . $separator . 'brpa=' . (((int)$data->infos->currentPage) - 1);
        } else {
            $data->infos->prevPageURL = '';
        }
        if($data->infos->currentPage < $data->infos->lastPage ) {
            $data->infos->nextPageURL = $bridgeParameters->currentUrl . $bridgeParameters->paginationUrlParams . $separator . 'brpa=' . (((int)$data->infos->currentPage) + 1);
        } else {
            $data->infos->nextPageURL = '';
        }

        // Pour réutilisation plus simple dans les templates de moteurs / filtres
        $moteur = null;
        if(isset($data->data->moteur))
            $moteur = $data->data->moteur;

        // On enrichit la réponse avec les infos de la Weblist
        $pageTitle = '';
        if (!empty($data->data->title)) {
            $pageTitle = $data->data->title;
            if($lang != 'fr' && isset($data->data->titleTranslations->$langUpper) && ! empty($data->data->titleTranslations->$langUpper))
                $pageTitle = $data->data->titleTranslations->$langUpper;
        }

        $pageDescription = '';
        if (!empty($data->data->description)) {
            $pageDescription = $data->data->description;
            if($lang != 'fr' && isset($data->data->descriptionTranslation->$langUpper) && ! empty($data->data->descriptionTranslation->$langUpper))
                $pageDescription = $data->data->descriptionTranslation->$langUpper;
        }

        // Annonce liée à la liste (on n'en gère qu'une)
        $annonce = null;
        if (isset($data->data->annonce)) {
            $annonce = $data->data->annonce;
        }

        $ret =  array (
            'listInfos' => $data->infos,
            'listParameters' => $data->parameters,
            'data' => $data->data,
            'moteur' => $moteur,
            'bridgeParameters' => $bridgeParameters,
            'pageTitle' => $pageTitle,
            'pageDescription' => $pageDescription,
            'annonce' => $annonce,
        );

        return $ret;
    }

    /**
     * prepareMapDataForRender
     * Prépare un tableau contenant les données à passer au renderer de carte interactive
     * @param mixed $data objet data tel que renvoyé par BridgeUtils::getMapData
     * @return array tableau associatif contenant les données utilisées par les vues map
     */
    public function prepareMapDataForRender($data) {
        $lang = $this->clientParams->getLanguage();
        $langUpper = strtoupper($lang);

        // Pour que tout soit objet : plus simple
        $data = json_decode(json_encode($data));

        // On stocke les infos relatives à l'URL et aux filtres et tris passés en paramètre
        $uri_parts = explode('?', $_SERVER['REQUEST_URI'], 2);
        if(is_object($data) && isset($data->infos) && is_object($data->infos)) {
            $data->infos->currentUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://$_SERVER[HTTP_HOST]$uri_parts[0]" ;
            $data->infos->activeFilters = (isset($_REQUEST["braf"])) ? $_REQUEST["braf"] : "" ;
            $data->infos->activeItems = (isset($_REQUEST["brai"])) ? $_REQUEST["brai"] : "" ;
            $data->infos->httpRequest = $_REQUEST;
        }

        $bridgeParameters = json_decode(json_encode($this->clientParams->getBridgeParameters()));


        return array (
            'mapInfos' => $data->infos,
            'mapParameters' => $data->parameters,
            'data' => $data->data,
            'bridgeParameters' => $bridgeParameters,
        );
    }

    /**
     * prepareMapPopupDataForRender
     * Prépare un tableau contenant les données à passer au renderer de popup de carte interactive
     * @param mixed $data objet data tel que renvoyé par BridgeUtils::getMapData
     * @return array tableau associatif contenant les données utilisées par les vues map
     */
    public function prepareMapPopupDataForRender($data) {
        $lang = $this->clientParams->getLanguage();
        $langUpper = strtoupper($lang);

        // Pour que tout soit objet : plus simple
        $data = json_decode(json_encode($data));

        // On stocke les infos relatives à l'URL et aux filtres et tris passés en paramètre
        $uri_parts = explode('?', $_SERVER['REQUEST_URI'], 2);
        if(is_object($data) && isset($data->infos) && is_object($data->infos)) {
            $data->infos->currentUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://$_SERVER[HTTP_HOST]$uri_parts[0]" ;
            $data->infos->activeFilters = (isset($_REQUEST["braf"])) ? $_REQUEST["braf"] : "" ;
            $data->infos->activeItems = (isset($_REQUEST["brai"])) ? $_REQUEST["brai"] : "" ;
            $data->infos->httpRequest = $_REQUEST;
        }

        $bridgeParameters = json_decode(json_encode($this->clientParams->getBridgeParameters()));

        return array (
            'mapPopupInfos' => $data->infos,
            'data' => $data->data,
            'bridgeParameters' => $bridgeParameters,
        );
    }


    /**
     * prepareFicheDataForRender
     * Prépare un tableau contenant les données à passer au renderer de Fiche
     * @param $data objet data tel que renvoyé par BridgeUtils::getDataFiche
     * @return array tableau associatif contenant les données utilisées par les vues fiches de rendu
     */
    public function prepareFicheDataForRender($data, $attributes = array()) {
        $lang = $this->clientParams->getLanguage();
        $langUpper = strtoupper($lang);

        // Pour que tout soit objet : plus simple
        $data = json_decode(json_encode($data));

        // Si pas d'infos, on retourne false
        if(!is_object($data) || !isset($data->product)) {
            return array();
        }
        if(is_object($data) && (!isset($data->infos) || !is_object($data->infos))) {
            $data->infos = new \stdClass();
        }

        // On stocke les infos relatives à l'URL et aux filtres et tris passés en paramètre
        $uri_parts = explode('?', $_SERVER['REQUEST_URI'], 2);
        if(is_object($data) && isset($data->infos) && is_object($data->infos)) {
            $data->infos->currentUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://$_SERVER[HTTP_HOST]$uri_parts[0]" ;
            $data->infos->activeFilters = (isset($_REQUEST["braf"])) ? $_REQUEST["braf"] : "" ;
            $data->infos->activeItems = (isset($_REQUEST["brai"])) ? $_REQUEST["brai"] : "" ;
            $data->infos->httpRequest = $_REQUEST;
        }

        $bridgeParameters = $this->clientParams->getBridgeParameters();
        $bridgeParameters = json_decode(json_encode($bridgeParameters)); // Conversion en objet
        // On override les paramètres de balises HTML avec ce qu'on pourrait trouver dans $data->infos->attributes
        if(isset($attributes) && !empty($attributes)) {
            $attributes = json_decode(json_encode($attributes));
            $markupOptionGroups = $this->clientParams->getMarkupOptionGroups();
            foreach($markupOptionGroups as $groupKey => $group) {
                foreach($group as $attributeKey => $attribute) {
                    if(isset($attributes->$attributeKey) && !empty($attributes->$attributeKey)) {
                        $bridgeParameters->$attributeKey = $attributes->$attributeKey;
                    }
                }
            }
        }

        // Initilisation des fiches associées
        // Update du 23/01/2023 : il faudra bientôt supprimer la ligne coupledLegacy
        // Update du 23.01.2023 : on ne charge la donnée coupled que si le composant des fiches associées est chargé
        $coupledLegacy = $this->getCoupledProducts($data->product);
        $coupledProducts = $this->getCoupledProductsData($data->product);

        // Update du 23.01.2023 : si le composant fiches associées liste ou fiches associées carte est présent, on prépare l'affichage de la liste


        // On injecte le JS et le CSS spécifique // Faudra faire différemment sur wp
        /*
        if($data->specificCssPath!='')
            $this->bridgeCmsAbstractLayer->enqueueStyle('specCssFiche',get_site_url() . '/wp-content/plugins/plugin-sit/'.$data->specificCssPath);
        if($data->specificJsPath!='')
            $this->bridgeCmsAbstractLayer->enqueueScript('specJsFiche',get_site_url() . '/wp-content/plugins/plugin-sit/'.$data->specificJsPath);
        */

        return array (
            'ficheInfos' => $data->infos,
            'ficheParameters' => $data->parameters,
            'dataFicheBridge' => $data,
            'ficheTemplate' => $data->ficheTemplate,
            'fiche' => $data->product,
            'bridgeParameters' => $bridgeParameters,
            'coupledLegacy' => $coupledLegacy,
            'coupledProducts' => $coupledProducts,
        );

    }
    /**
     * Prépare un tableau contenant les données à passer au renderer de pub
     * @param $data objet data tel que renvoyé par BridgeUtils::getAnnonceData
     * @return array tableau associatif contenant les données utilisées par les vues annonce
     */
    public function prepareAnnonceDataForRender($data) {
        $data = json_decode(json_encode($data));
        $bridgeParameters = json_decode(json_encode($this->clientParams->getBridgeParameters()));
        return array (
            'annonceInfos' => $data->infos,
            'data' => $data->data,
            'bridgeParameters' => $bridgeParameters,
        );
    }
    /**
     * Prépare un tableau contenant les données à passer au renderer d'espace pub
     * @param array $data data tel que renvoyé par BridgeUtils::getAnnonceSpaceData
     * @return array tableau associatif contenant les données utilisées par les vues annonce-space
     */
    public function prepareAnnonceSpaceDataForRender(array $data): array {
        $data = json_decode(json_encode($data));
        $bridgeParameters = json_decode(json_encode($this->clientParams->getBridgeParameters()));

        return array (
            'annonceSpaceInfos' => $data->infos,
            'data' => $data->data,
            'bridgeParameters' => $bridgeParameters,
        );
    }


    private function formatParametersListe($params){
        $res = new \stdClass();
        if(!isset($params)) return $res;
        foreach($params as $key=>$param) {
            if (isset($param) && is_object($param)) {
                $res->$key = $param->value;
            }
        }
        return $res;
    }

    /**
     * Retourne une liste de communes répondant aux paramètres de recherche passés
     * @param array $attributes : paramètres d'affichage
     * @param bool $isAjax : si true, la fonction se comporte en mode ajax et ne retourne que les templateItems
     *
     * @return array
     */
    public function getCitiesData($attributes) {

        $res = array(
            'success' => false,
            'message' => '',
            'data' => null,
            'infos' => array(),
            'parameters' => array()
        );

        $bridgeCredentials = $this->clientParams->getBridgeCredentials();
        $urlBridge = $bridgeCredentials->urlBridge;

        if(empty($urlBridge)) {
            $res['message'] = "L'URL de Bridge n'est pas définie - contrôlez les paramètres du plugin";
            return $res;
        }

        if(!is_array($attributes) || empty($attributes['id'])){
            $res['message'] = 'Le paramètre id est obligatoire dans le shortcode';
            return $res;
        }

        // La liste est filtrée par défaut en lisant les paramètres GET POST du contexte
        // l'attribut useRequestFilters permet de désactiver ce comportement
        if(!isset($attributes['useRequestFilters']))
            $attributes['useRequestFilters'] = true;

        // Variable utilisée à différents endroits du code
        $webListId = $attributes['id'];
        $res['infos']['debug'] = array() ;

        // Pagination des résultats
        // Première fiche
        $first = '1' ;
        $lang = $this->clientParams->getLanguage();
        $res['infos']['lang'] = $lang;

        // On ajoute les paramètres de filtre du moteur et les paramètres de tri
        $brParamsUrlFiltre = '';
        $brParamsUrlSort = '';

        if (count($_REQUEST) > 0 && $attributes['useRequestFilters']) {
            foreach ($_REQUEST as $key=>$value) {
                // braf: Bridge Active Filter
                // brai: Bridge Active Item
                if ($key == 'braf' || $key == 'brai' || (strlen($key) > 6 && substr($key,0,6) == 'brflt_')) {
                    // On doit encoder en URI les paramètres de type valeur saisie par l'internaute
                    if (strpos($key, '_value') !== false)
                        $brParamsUrlFiltre .= "&$key=" . urlencode($value);
                    elseif (strpos($key, '_city') !== false)
                        $brParamsUrlFiltre .= "&$key=" . urlencode($value);
                    else
                        $brParamsUrlFiltre .= "&$key=$value";
                }
            }
        }

        // JM 22/03/2022 : on change le nom du paramètre (update -> v2), moins risqué
        $url = $urlBridge . "/weblist/getCities/" . $webListId . "?v2=1&lang=" . $lang ;

        // Filtres et tris forcés
        if(isset($attributes['braf']) && !empty($attributes['braf'])) {
            $brParamsUrlFiltre .= "&braf=" . $attributes['braf'];
        }
        if(isset($attributes['brai']) && !empty($attributes['brai'])) {
            $brParamsUrlFiltre .= "&brai=" . $attributes['brai'];
        }
        if(isset($attributes['bras']) && !empty($attributes['bras'])) {
            $brParamsUrlSort .= "&bras=" . $attributes['bras'];
        }
        if(isset($attributes['brsd']) && !empty($attributes['brsd'])) {
            $brParamsUrlSort .= "&brsd=" . $attributes['brsd'];
        }

        if ($brParamsUrlFiltre != '') {
            $url .= $brParamsUrlFiltre;
        }

        if ($brParamsUrlSort != '') {
            $url .= $brParamsUrlSort;
        }

        // Feat : 25/05/2022 : possibilité de passer une liste de productCodes pour forcer les fiches à afficher dans le shortcode
        $productCodes = '';
        if(isset($attributes['product_codes']) && !empty($attributes['product_codes'])) {
            $productCodes = $attributes['product_codes'];
            $url .= "&productCodes=" . $productCodes;
        }
        // Support des requetes minimalistes pour les points sur la carte
        if(isset($attributes['minimal_select']) && !empty($attributes['minimal_select'])) {
            $url .= "&minimalSelect=1" ;
        }
        // Ajout 13/02/2023 :  possibilité de passer un nom de commune pour cumulr un filtre par commune à la séelction de la liste
        if(isset($attributes['filter_city']) && !empty($attributes['filter_city'])) {
            $url .= "&filter_city=" . $attributes['filter_city'];
        }
        // 13/02/2023 : application d'un scoring
        if(isset($attributes['filter_lat']) && !empty($attributes['filter_lat'])) {
            $url .= "&filter_lat=" . $attributes['filter_lat'];
        }
        // 13/02/2023 : application d'un scoring
        if(isset($attributes['filter_lon']) && !empty($attributes['filter_lon'])) {
            $url .= "&filter_lon=" . $attributes['filter_lon'];
        }
        // 13/02/2023 : application d'un scoring
        if(isset($attributes['filter_rayon']) && !empty($attributes['filter_rayon'])) {
            $url .= "&filter_rayon=" . $attributes['filter_rayon'];
        }
        // 13/02/2023 : filtre par critères / mdalités
        if(isset($attributes['filter_criterions']) && !empty($attributes['filter_criterions'])) {
            // Syntaxes pour l'attribut criterion (multicritères avec ou sans valeur) - on parle bien de criterions_codes et de modality_codes
            // CRIT1:MODA1|MODA2|MODA3,CRIT2:MODA4=VALEUR,CRIT3:MODA5=MIN-MAX,CRIT4,CRIT5:MODA6|MODA7=MIN-MAX
            $url .= "&filter_criterions=" . urlencode($attributes['filter_criterions']);
        }
        // 13/02/2023 : application d'un scoring
        if(isset($attributes['scoring']) && !empty($attributes['scoring'])) {
            $url .= "&scoring=" . (int) $attributes['scoring'];
        }
        // 13/02/2023 : application d'un scoring
        if(isset($attributes['sort_one']) && !empty($attributes['sort_one'])) {
            $url .= "&sort_one=" . (int) $attributes['sort_one'];
        }
        // 13/02/2023 : application d'un scoring
        if(isset($attributes['sort_two']) && !empty($attributes['sort_two'])) {
            $url .= "&sort_two=" . (int) $attributes['sort_two'];
        }
        // 13/02/2023 : application d'un scoring
        if(isset($attributes['sort_three']) && !empty($attributes['sort_three'])) {
            $url .= "&sort_three=" . (int) $attributes['sort_three'];
        }
        // 13/02/2023 : application d'un scoring
        if(isset($attributes['duplicate_by']) && !empty($attributes['duplicate_by'])) {
            $url .= "&duplicate_by=" . $attributes['duplicate_by'];
        }
        // 13/02/2023 : application d'un scoring
        if(isset($attributes['selection_id']) && !empty($attributes['selection_id'])) {
            $url .= "&selection_id=" . (int) $attributes['selection_id'];
        }
        // 08/01/2024 : ajout d'un filter id pour avoir toutes les communes d'un département si nécessaire
        if(isset($attributes['filter_id']) && !empty($attributes['filter_id'])) {
            $url .= "&filter_id=" . (int) $attributes['filter_id'];
        }
        if(isset($_REQUEST['dncdebug']) && $_REQUEST['dncdebug'] == 'Y')
            echo "<a href='".$url."' target='_blank' class='uk-button uk-button-primary'>Détails de liste</a>";

        $tmpData = $this->requestService->cachedFileGetContent($url);
        $data = null;

        if (!empty($tmpData)) {
            $tmpDataDecoded = json_decode($tmpData);
            if ($tmpDataDecoded->success && $tmpDataDecoded->data) {
                $data = $tmpDataDecoded->data;
            } else {
                $res['message'] = $tmpDataDecoded->message;
                return $res;
            }
        } else {
            $res['message'] = 'Échec de la récupération de la liste - 2';
            return $res;
        }

        if (!is_array($data)) {
            $res['message'] = "ERREUR d'appel à Bridge : aucune donnée valide renvoyée";
            return $res;
        }
        $res['data'] = $data;
        $res['success'] = true;
        return $res;
    }

    /**
     * getElevationData
     * Renvoit les données d'altitude du tracé d'une fiche
     * @param object $fiche objet product tel que retourné par l'API Bridge
     * @return mixed|void|null données altimétriques du tracé
     */
    public function getElevationData($fiche)
    {
        if (isset($fiche) && is_object($fiche) && property_exists($fiche, 'kml') && !empty($fiche->kml)) {
            $urlkml = $fiche->kml;
            $data = $this->bridgeApiService->callBridgeApi('GET', '/api/gis/get_elevation_data', 'urlkml=' . urlencode($urlkml));
            if(!empty($data)) {
                // $data = json_decode($textData);
                if(isset($data) && $data !== false && is_object($data) && $data->success) {
                    // print_r($data);
                    return $data->data;
                }
            }
            return null;
        }
    }

    /**
     * getProductDispos
     * Renvoit les disponibilités en temps réel d'une fiche (donnée chaudes)
     * @param $fiche
     * @return mixed|null
     */
    public function getProductDispos($fiche) {
        $urlBridge = $this->clientParams->getUrlBridge();
        // 11/04/2024 : implémentation d'un système de cache volatile fichier
        $json = $this->requestService->cachedFileGetContent($urlBridge . '/product/getDispos?productCode=' . $fiche->productCode);
        $dispos = json_decode( $json );
        if(!empty($dispos))
            return $dispos;
        else
            return null;
    }

    /**
     * getFicheData
     * Renvoit l'adresse mail de l'administrateur du site
     * @param $siteID
     * @return mixed|null
     */
    public function getAdminMailAddress($siteID) {
        $site = $this->bridgeApiService->callBridgeApi('GET', '/api/sites/' . $siteID);

        if(!is_object($site)) {
            return false;
        }

        if(isset($site->adminMailAddress) && !empty($site->adminMailAddress)) {
            return $site->adminMailAddress;
        }

        $params = $this->bridgeApiService->callBridgeApi('GET', '/params/get_params');

        if(!is_object($params)) {
            return false;
        }

        if(isset($params->import_notification_email) && !empty($params->import_notification_email))
        {
            return $params->import_notification_email;
        }

        return false;
    }

}
