<?php
namespace Bridge\Weblibs;

use Bridge\Weblibs\BridgeRequestServiceInterface;

class BridgeImageService
{
	private BridgeRequestServiceInterface $requestService;

	public function __construct(BridgeRequestServiceInterface $requestService)
	{

		$this->requestService = $requestService;
	}

	/**
	 * @param string $url
	 * @param string $lemode : 'contain', 'cover', 'remplir', 'crop', 'tronq'
	 *      'contain' : l'image est redimensionnée pour être contenue dans les dimensions souhaitées
	 *      'ajust' : identique à contain
	 *      'cover' : l'image est redimensionnée pour couvrir les dimensions souhaitées
	 *      'remplir' : identique à cover
	 *      'crop' : l'image est redimensionnée pour remplir les dimensions souhaitées et est ensuite recadrée à ces dimensions exactes
	 *      'tronq' : identique à crop
	 * @param int|string $selwidth : largeur souhaitée
	 * @param int|string $selheight : hauteur souhaitée
	 * @param int|string $def : Définition de l'image (de 1 à 100 pour jpeg, 0 à 9 pour png)
	 * @param bool|string $debug : mode débug
	 *
	 * @return array|bool
	 *
	 * @since version
	 */
	public function resizeImage($url = "", $lemode = "", $selwidth = "", $selheight = "", $quality = 60, $debug = false, $returnType = 'RESOURCE')
	{

		$ret = $this->blankErrorImage();

		if ($url == '' || ($selwidth == '' && $selheight == '')) {
			$ret['error'] = 'Paramètres manquants ';
			return $ret;
		}
		if ($lemode == '' || $lemode == 'ajust' || $lemode == 'adjust') {
			$lemode = 'contain';
		}

		if ($quality == '') {
			$quality = 60;
		}
		$quality = (int)$quality;

		$file = str_replace(' ', '%20', $url);
		$file = str_replace("https://http://", "https://", $file);
		$file = str_replace("https://https://", "https://", $file);

		// Patch ND 12.06.2023 : on télécharge localement l'image pour pouvoir gérer un timeout raisonnable
		$options = array(
			'http' => array(
				'timeout' => 4   // Timeout in seconds
			),
			"ssl"=>array(
				"verify_peer"=>false,
				"verify_peer_name"=>false,
			),
		);

		// Fetch the URL's contents
		$binaryData = $this->requestService->cachedFileGetContent($file,false,$options);
		//@file_get_contents($file, 0, $context);

		if ($binaryData === false) {
			$ret['error'] = 'Fichier source introuvable';
			return $ret;
		}
		try {
			list($width, $height, $image_type) = getimagesizefromstring($binaryData);
		} catch (\Exception $e) {
			// en cas d'erreur réseau, on renvoit rien
			$ret['error'] = 'Erreur réseau lors de la récupération de l\'image';
			return $ret;
		}


		// Si jamais on nous passe pas une URL d'image, on sort tout de suite
		if (!isset($image_type)) {
			$ret['error'] = 'Le fichier source n\'est pas une image valide';
			return $ret;
		}

		// Si aucune dimension d'image n'est fournie (ce qui est anormal) on se cale sur les dimensions de l'image source
		if ($selwidth == "" && $selheight == "") {
			$selwidth = $width;
			$selheight = $height;
		}

		if ($selwidth != "" || $selheight != "") {
			try {
				$src = imagecreatefromstring($binaryData);
				if ($src === false) {
					$ret['error'] = 'Le fichier source n\'est pas une image valide (création impossible)';
					return $ret;
				}
			} catch (\Exception $e) {
				// en cas d'erreur réseau, on renvoit rien
				$ret['error'] = 'Le fichier source n\'est pas une image valide (erreur de la librairie GD)';
				return $ret;
			}

			// Pach 11/2021 : gestion des photos portraits iPhone qui se retrouvent tournées
			// On detcte l'orientation de l'image et on adapte les dimensions
			$inverserWidthHeight = false;
			try {
				$exif = @exif_read_data($file);

				if (is_array($exif) && !empty($exif['Orientation'])) {
					switch ($exif['Orientation']) {
						case 3:
							$src = imagerotate($src, 180, 0);
							break;
						case 6:
							$src = imagerotate($src, -90, 0);
							$inverserWidthHeight = true;
							break;
						case 8:
							$src = imagerotate($src, 90, 0);
							$inverserWidthHeight = true;
							break;
						default:
							break;
					}
				}
			} catch (\Exception $e) {
				// On peut avoir une erreur EXIF, tant pis, c'est pas grave
				$exif = array();
			}

			if ($inverserWidthHeight) {
				$tmpW = $width;
				$tmpH = $height;
				$width = $tmpH;
				$height = $tmpW;
			}

			// dimensions finales de l'image souhaitée
			$final_width = $selwidth * 1;
			$final_height = $selheight * 1;
			// Fin de la gestion des photos iPhone portrait

			if (isset($_REQUEST['imagedebug'])) {
				// die('ouf3');
			}
			// On détermine les proportions de l'image pour adapter les dimensions lors du redimensionnement
			$x_ratio = $final_width / $width;
			$y_ratio = $final_height / $height;

			if ($lemode == "contain") {
				if ($selwidth == "") {
					$ratio = $y_ratio;
				} elseif ($selheight == "") {
					$ratio = $x_ratio;
				} else {
					$ratio = min($x_ratio, $y_ratio);
				}
				// Dimensions de l'image redimensionnée
				$resized_tmpwidth = ceil($ratio * $width);
				$resized_tmpheight = ceil($ratio * $height);
			} elseif ($lemode == "cover" || $lemode == "remplir" || $lemode == "crop" || $lemode == "tronq") {
				if ($selwidth == "") {
					$ratio = $y_ratio;
				} elseif ($selheight == "") {
					$ratio = $x_ratio;
				} else {
					$ratio = max($x_ratio, $y_ratio);
				}
				// dimensions de l'image finale
				$resized_tmpwidth = ceil($ratio * $width);
				$resized_tmpheight = ceil($ratio * $height);
				// Dimensions de l'image redimensionnée
			} else {
				$resized_tmpwidth = ceil($x_ratio * $width);
				$resized_tmpheight = ceil($y_ratio * $height);
				// Dimensions de l'image redimensionnée
				$scaled_width = ceil($x_ratio * $width);
				$scaled_height = ceil($y_ratio * $height);
			}


			$tmp = imagecreatetruecolor($resized_tmpwidth, $resized_tmpheight);

			/* Check if this image is PNG or GIF, then set if Transparent */
			// TODO : Ne permet pas de faire un tronq sur des png et gif!!!
			if ($lemode == "crop" || $lemode == "tronq") {
				if ($resized_tmpheight > $final_height || $resized_tmpwidth > $final_width) {
					$tmp = imagecreatetruecolor($final_width, $final_height);
				}
				imagecopyresampled($tmp, $src, 0, 0, round($resized_tmpwidth / $ratio / 2) - round($final_width / $ratio / 2), round($resized_tmpheight / $ratio / 2) - round($final_height / $ratio / 2), $resized_tmpwidth, $resized_tmpheight, $width, $height);
			} else {
				if (($image_type == 1) || ($image_type == 3)) {
					if ($image_type == 1) {
						imagecolortransparent($tmp, imagecolorallocatealpha($tmp, 255, 255, 255, 127));
					}
					imagealphablending($tmp, false);
					imagesavealpha($tmp, true);
					$transparent = imagecolorallocatealpha($tmp, 255, 255, 255, 127);
					imagefilledrectangle($tmp, 0, 0, $resized_tmpwidth, $resized_tmpheight, $transparent);
				}
				// ND 22/01/2016 : probème avec double utilisation sur gif transprent : tranpasrent devient blanc -> resized au lieu de resampled
				//imagecopyresampled($tmp, $src, 0, 0, 0, 0, $resized_tmpwidth, $resized_tmpheight, $width, $height);
				try {
					imagecopyresized($tmp, $src, 0, 0, 0, 0, $resized_tmpwidth, $resized_tmpheight, $width, $height);
				} catch (\Exception $e) {
					$ret['error'] = 'Erreur lors du redimensionnement de l\'image';
					return $ret;
				}
			}
		}
		switch ($image_type) {
			case 1:
				$headers = array('Content-Type' => 'image/gif');
				ob_start();
				imagegif($tmp, null);
				$image_data = ob_get_contents();
				imagedestroy($tmp);
				return array('headers' => $headers, 'data' => $image_data, 'ext' => 'gif', 'success' => true);
			case 2:
				$headers = array('Content-Type' => 'image/jpeg',
				                 'Content-Disposition' => 'inline; filename="image.jpg"');
				ob_start();
				imagejpeg($tmp, null, $quality);
				imagedestroy($tmp);
				$image_data = ob_get_clean();
				return array('headers' => $headers, 'data' => $image_data, 'ext' => 'jpg', 'success' => true);
			case 3:
				$headers = array('Content-Type' => 'image/png');
				// Attention ! en png, c'est le taux de compression qui va de 0 à 9
				$compression = round(($quality / 100) * 9);
				ob_start();
				imagepng($tmp, null, $compression);
				$image_data = ob_get_contents();
				imagedestroy($tmp);
				return array('headers' => $headers, 'data' => $image_data, 'ext' => 'png', 'success' => true);
			default:
				$ret['error'] = 'Format d\'image non supporté';
				return $ret;
		}
		return $ret;

	}


	private function blankErrorImage()
	{
		$blankImg = imagecreatetruecolor(10, 10);
		$headers = array('Content-Type' => 'image/jpeg',
		                 'Content-Disposition' => 'inline; filename="blank.jpg"');
		ob_start();
		imagejpeg($blankImg, null, 25);
		imagedestroy($blankImg);
		$image_data = ob_get_clean();
		return array('headers' => $headers, 'data' => $image_data, 'ext' => 'jpg', 'success' => false, 'error' => 'Erreur inconnue');
	}
}
