<?php

namespace Drupal\pylot_bridge\Services;

use Bridge\Weblibs\BridgeClientParamsInterface;
use Bridge\Weblibs\BridgeCmsAbstractLayerInterface;
use Bridge\Weblibs\BridgeClientRendererInterface;


class BridgeClientRendererDrupal implements BridgeClientRendererInterface
{

    private BridgeClientParamsInterface $clientParams;
	private BridgeCmsAbstractLayerInterface $bridgeCmsAbstractLayer;

    public function __construct(BridgeClientParamsInterface $clientParams, BridgeCmsAbstractLayerInterface $bridgeCmsAbstractLayer)
	{
        $this->clientParams = $clientParams;
		$this->bridgeCmsAbstractLayer = $bridgeCmsAbstractLayer;
    }

    /**
     * Effectue le rendu d'un template avec des données passées
     * @param string $templatePath
     * @param array $data
     * @return array Render array for Drupal
     */
    public function renderTemplate(string $templatePath, array $data = array()) {
        // In Drupal, we return a render array, not an HTML string directly.
        // The template path will be converted to a theme hook.
        $theme_hook = str_replace(['/', '.html.twig'], ['_', ''], $templatePath);
        return [
            '#theme' => $theme_hook,
            '#data' => $data,
        ];
    }

    /**
     * Effectue le rendu d'un template de liste Ajax avec des données passées
     * @param array $listData
     * @return array Render array for Drupal
     */
    public function renderListeAjaxTemplate(array $listData = array()) {
        $build = $this->prepareListViewRenderer($listData, 'liste_sit_ajax');
        $html = \Drupal::service('renderer')->renderPlain($build);
        return $html;
    }

	/**
	 * Effectue le rendu d'un template de liste avec des données passées
	 * @param array $listData
	 * @return array Render array for Drupal
	 */
	public function renderListeTemplate(array $listData = array()) {

        // $build = $this->prepareListViewRenderer($listData);
        $html = \Drupal::service('renderer')->renderPlain($listData);
        return $html;
	}

    /**
     * Retourne un tableau des librairies Js utilisée pour afficher les listes / moteurs
     * @return array
     */
    public function getListScripts() {
        $libraries = array();
        $libraries[] = 'uikit-js';
        $libraries[] = 'fontawesome';
        // $libraries[] = 'fontawesome-all';
        $libraries[] = 'leaflet';
        $libraries[] = 'leaflet-fs';
        $libraries[] = 'leaflet-mc';
        $libraries[] = 'leaflet-control-button';
        $libraries[] = 'leaflet-easy-button';
        $libraries[] = 'togeojson';
        $libraries[] = 'dnc-leaflet-services';
        $libraries[] = 'mc-datepickerjs';
        $libraries[] = 'infinite-scroll';
        $libraries[] = 'plugin-sit-script';
        $libraries[] = 'sit-custom-js';
        $libraries[] = 'bridge-autocomplete';
        $libraries[] = 'bridge-moteur';
        $libraries[] = 'bridge-datepicker';
        $libraries[] = 'bridge-datepicker-locale';
        $libraries[] = 'bridge-flatpickr';
        $libraries[] = 'bridge-flatpickr-locale';
        $libraries[] = 'infinite-scroll-ajax';
        return $libraries;
    }

    /**
     * Retourne un tableau des librairies css utilisée pour afficher les listes / moteurs
     * @return array
     */
    public function getListStyles() {
        $libraries = array();
        $libraries[] = 'uikit-css';
        $libraries[] = 'fontawesome-v6-all';
        $libraries[] = 'leaflet';
        $libraries[] = 'leaflet-fs';
        $libraries[] = 'leaflet-mc';
        $libraries[] = 'leaflet-mc-theme';
        $libraries[] = 'leaflet-easy-button';
        $libraries[] = 'leaflet-control-button';
        $libraries[] = 'mc-datepicker';
        $libraries[] = 'bridge-datepicker';
        $libraries[] = 'bridge-flatpickr';
        //$libraries[] = 'infiniscroll-style';
        $libraries[] = 'bridge-autocomplete';
        $libraries[] = 'plugin-sit-style';
        $libraries[] = 'sit-custom-css';
        return $libraries;
    }

	/**
	 * Effectue le rendu d'un template de fiche avec des données passées
	 * @param array $ficheData
	 * @return array|string Render array for Drupal or empty string
	 */
	public function renderFicheTemplate(array $ficheData = array()) {
		if (!empty($ficheData)) {
            // The template name is dynamic.
            $theme_hook = 'bridge_fiche_' . str_replace('-', '_', $ficheData['ficheTemplate']);
			return [
                '#theme' => $theme_hook,
                '#data' => $ficheData,
                '#attached' => [
                    'library' => $this->getFicheLibraries(),
                ],
            ];
		}
		return '';
	}

    /**
     * Charge une feuille de style css dans la page
     * @param $handle : id texte du css
     * @param string $src : url du css
     * @param array $deps : tableau de dépendances (liste d'id texte)
     * @param false $ver : version
     * @param string $media
     * @return mixed
     */
    public function BridgeEnqueueStyle($handle, $src = "", $deps = array(), $ver = false, $media = 'all') {
        return $this->bridgeCmsAbstractLayer->enqueueStyle($handle, $src, $deps, $ver, $media);
    }

    /**
     * Charge un script js dans la page
     * @param $name identifiant texte du script
     * @param $link URL du script
     * @return bool
     */
    public function BridgeEnqueueScript($handle, $src = "", $deps = array(), $ver = false, $in_footer = false ) {
        return $this->bridgeCmsAbstractLayer->enqueueScript($handle, $src, $deps, $ver, $in_footer);
    }

    /**
     * Retourne le header du theme front-end du site pour affichage avant le code de la fichede détail (WordPress)
     * @return mixed
     */
    public function getHeaderBridge($arg = null) {
        return $this->bridgeCmsAbstractLayer->getHeaderBridge($arg);
    }

    /**
     * Retourne le footer du theme front-end du site pour affichage après le code de la fichede détail (WordPress)
     * @return mixed
     */
    public function getFooterBridge() {
        return $this->bridgeCmsAbstractLayer->getFooterBridge();
    }

    /**
     * registerAllStyles
     * In Drupal, styles are registered in a *.libraries.yml file.
     * This method is no longer needed.
     */
    public function registerAllStyles() {
        // This function is deprecated for Drupal. Libraries are defined in YAML files.
    }

    /**
     * Enregistre tous les scripts utilisés par Bridge pour traiter l'affichage
     * In Drupal, scripts are registered in a *.libraries.yml file.
     * This method is no longer needed.
     */
    public function registerAllScripts() {
        // This function is deprecated for Drupal. Libraries are defined in YAML files.
    }

    /**
     * Returns an array of JS/CSS libraries used for lists/search engines.
     * These should correspond to libraries defined in your_module.libraries.yml.
     * @return array
     */
    public function getListLibraries() {
        $libraries = array();
        $libraries[] = 'core/jquery';
        $libraries[] = 'pylot_bridge/uikit';
        $libraries[] = 'pylot_bridge/bridge-autocomplete';
        $libraries[] = 'pylot_bridge/fontawesome-v6';
        $libraries[] = 'pylot_bridge/leaflet';
        $libraries[] = 'pylot_bridge/leaflet-addons-bridge';
        $libraries[] = 'pylot_bridge/dncleafletservices';
        $libraries[] = 'pylot_bridge/bridge-base';
        $libraries[] = 'pylot_bridge/mc-datepicker';
        $libraries[] = 'pylot_bridge/bridge-datepicker';
        $libraries[] = 'pylot_bridge/bridge-flatpickr';
        //$libraries[] = 'pylot_bridge/bridge-datepicker-locale';
        //$libraries[] = 'pylot_bridge/bridge-flatpickr-locale';
        $libraries[] = 'pylot_bridge/calendar';
        $libraries[] = 'pylot_bridge/bridge-pagination';
        $libraries[] = 'pylot_bridge/bridge-moteur';
        $libraries[] = 'pylot_bridge/bridge-custom';
        $libraries[] = 'pylot_bridge/bridge-simple-list-map-loader';

        $lang = $this->clientParams->getLanguage();
        // For datepicker locale
        $libraries[] = 'bridge/datepicker-locale-' . $lang;
        // For flatpickr locale
        $libraries[] = 'bridge/flatpickr-locale-' . $lang;

        return $libraries;
    }

    /**
     * Returns an array of JS/CSS libraries used for detail pages.
     * These should correspond to libraries defined in your_module.libraries.yml.
     * @return array
     */
    public function getFicheLibraries() {
        $libraries = [];
        // On injecte les scripts et feuilles de style nécessaires
        $libraries[] = 'core/jquery';
        $libraries[] = 'pylot_bridge/uikit';
        $libraries[] = 'pylot_bridge/fontawesome-v6';
        $libraries[] = 'pylot_bridge/leaflet';
        $libraries[] = 'pylot_bridge/leaflet-addons-bridge';
        $libraries[] = 'pylot_bridge/dncleafletservices';
        $libraries[] = 'pylot_bridge/calendar';
        $libraries[] = 'pylot_bridge/mc-datepicker';
        $libraries[] = 'pylot_bridge/bridge-datepicker';
        // $libraries[] = 'pylot_bridge/bridge-datepicker-locale';
        $libraries[] = 'pylot_bridge/bridge-base';
        $libraries[] = 'pylot_bridge/bridge-simple-list-map-loader';
        $libraries[] = 'pylot_bridge/bridge-custom';
        $libraries[] = 'pylot_bridge/chart-js';
        $lang = $this->clientParams->getLanguage();
        $libraries[] = 'bridge/datepicker-locale-' . $lang;

        return $libraries;
    }

    /**
     * Charge tous les scripts et styles utilisés sur les listes
     * This is handled by attaching libraries to render arrays in Drupal.
     */
    public function loadListScriptsAndStyles() {
        // Deprecated for Drupal.
    }

    /**
     * Retourne un tableau des librairies Js utilisée pour afficher une fiche de détail
     * @return array
     */
    public function getFicheScripts() {
        // Deprecated. Use getFicheLibraries() instead.
        return [];
    }

    /**
     * Retourne un tableau des librairies CSS utilisée pour afficher une fiche de détail
     * @return array
     */
    public function getFicheStyles() {
        // Deprecated. Use getFicheLibraries() instead.
        return [];
    }

    /**
     * Charge tous les scripts et styles utilisés sur les listes
     * This is handled by attaching libraries to render arrays in Drupal.
     */
    public function loadFicheScriptsAndStyles() {
        // Deprecated for Drupal.
    }

    /**
     * Locate template.
     * In Drupal, template location is handled by the Twig system and theme discovery.
     * This method is no longer needed.
     */
    public function bridge_locate_template( $template_name, $template_path = '', $default_path = '' ) {
        // Deprecated for Drupal.
        return null;
    }

    /**
     * Permet de localiser une feuille de style dynamique dans /public/bridgewebfrontlibs/css
     * In Drupal, this is handled by the library system.
     * This method is no longer needed.
     */
    public function bridge_locate_style($styleFileName, $styleFilePath = '', $default_path = '') {
        // Deprecated for Drupal.
        return null;
    }

	/**
	 * Permet de déclarer une fonction de callback pour retrvailler du contenu dans les vues Bridge sans avoir à les overrider
	 * équivalent à wordpress apply_filter
	 * @param $hookName : nom du hook
	 * @param mixed $value : The value to be filtered.
	 * @param mixed ...$args : Additional arguments passed to the hook.
	 * @return mixed The filtered value.
	 */
	public function applyFilters($hookName, $callBack, $priority = 10, $acceptedArgs = 1) {
        return $callBack;
	}

	/**
	 * prepareListViewRenderer : utile uniquement pour Drupal et Symfony (pour le moment)
	 * Prépare l'affichage d'une liste dans une page (non ajax)
	 * Attache les scripts et CSS et retourne un tableau associatif contenant toutes les données qui seront utilisées dans les vues
	 * $data Objet data tel que renvoyé par BridgeUtils::prepareListdataForRender
	 * $theme nom du fichier twig initial à charger
	 * @return array tableau associatif contenant toutes les données qui seront utilisées dans les vues
	 */
	public function prepareListViewRenderer($data, $theme = 'liste_sit') {
        $libraries = $this->getListLibraries();
        $js_settings = [
            'bridgeTradsJs' => [
                'voir_carnet' => t('Voir le Carnet de Voyage'),
            ],
        ];

        $build = array (
            '#theme' => $theme,
            '#attached' => array('library' => array(), 'drupalSettings' => ['bridge' => $js_settings],),
            '#children' => array(),
            '#listInfos' => $data['listInfos'],
            '#listParameters' => $data['listParameters'],
            '#data' => $data['data'],
            '#moteur' => $data['moteur'],
            '#bridgeParameters' => $data['bridgeParameters'],
            '#pageTitle' => $data['pageTitle'],
            '#libraries' => $libraries,
            '#pageDescription' => $data['pageDescription'],
        );

        // On injecte les scripts et feuilles de style nécessaires

        foreach( $libraries as $lib)
            $build['#attached']['library'][] = $lib;

        return $build;
	}

	/**
	 * prepareFicheViewRenderer : utile uniquement pour Drupal et Symfony (pour le moment)
	 * Prépare l'affichage d'une fiche dans une page (non ajax)
	 * Attache les scripts et CSS et retourne un tableau associatif contenant toutes les données qui seront utilisées dans les vues
	 * @param $dataFiche Objet data tel que renvoyé par BridgeUtils::prepareFichedataForRender
	 * @param $theme nom du fichier twig initial à charger
	 * @return array tableau associatif contenant toutes les données qui seront utilisées dans les vues
	 */
	public function prepareFicheViewRenderer($datafiche, $theme = 'fiche_sit_ajax') {
        $libraries = $this->getFicheLibraries();
        $theme_hook = 'bridge_fiche_' . str_replace('-', '_', $datafiche['ficheTemplate']);

        return [
            '#theme' => $theme_hook,
            '#data' => $datafiche,
            '#attached' => [
                'library' => $libraries,
            ],
        ];
	}

}
