<?php

namespace Bridge\Weblibs\Wordpress;

use Bridge\Weblibs\BridgeClientParamsInterface;
use Bridge\Weblibs\Wordpress\BridgeCmsAbstractLayerWordpress;
use Bridge\Weblibs\Wordpress\BridgeRequestServiceWordpress;


class BridgeClientParamsWordpress implements BridgeClientParamsInterface
{
    private $requestService;
	private $bridgeCmsAbstractLayer;

    public function __construct(BridgeRequestServiceWordpress $requestService, BridgeCmsAbstractLayerWordpress $bridgeCmsAbstractLayer)
	{
		$this->requestService = $requestService;
		$this->bridgeCmsAbstractLayer = $bridgeCmsAbstractLayer;
    }

    /**
     * Retourne un tableau de Groupes d'options de markup HTML pour les listes (titres, conteneurs, etc.)
     * @return array Tableau de Groupes d'options de markup HTML pour les listes
     */
    public function getMarkupOptionGroups() {
        return array(
            'Listes et carrousels' => array (
                'tag_list_container' => 'Conteneur de liste',
                'tag_list_pagination' => 'Conteneur de pagination',
                'tag_listitem_container' => 'Conteneur d\'élément de liste',
                // 'list_titre' => 'Titre de liste', // kept commented like original
            ),
            'Eléments des listes et carrousels' => array(
                'tag_listitem_surtitre' => 'Surtitre',
                'tag_listitem_titre' => 'Titre',
                'tag_listitem_soustitre' => 'Sous-titre',
                'tag_listitem_commentaire' => 'Commentaire',
                'tag_listitem_pictogrammes' => 'Pictogrammes',
                'tag_listitem_info1' => 'Info1',
                'tag_listitem_info2' => 'Info2',
                'tag_listitem_info3' => 'Info3',
                'tag_listitem_info4' => 'Info4',
            ),
            'Moteur' => array(
                'tag_moteur_titre_section' => 'Titre de section',
                'tag_moteur_titre_subsection' => 'Titre de sous-section',
                'tag_moteur_titre_filtre' => 'Titre de filtre',
            ),
            'Fiche' => array(
                'tag_fiche_titre_infosupplementaires1' => 'Titre de bloc infos supplémentaires 1',
                'tag_fiche_titre_infosupplementaires2' => 'Titre de bloc infos supplémentaires 2',
                'tag_fiche_titre_infosupplementaires3' => 'Titre de bloc infos supplémentaires 3',
                'tag_fiche_titre_infosupplementaires4' => 'Titre de bloc infos supplémentaires 4',
                'tag_fiche_titre_infosupplementaires5' => 'Titre de bloc infos supplémentaires 5',
                'tag_fiche_titre_groupe_blocperso' => 'Titre de groupe de blocs perso',
                'tag_fiche_titre_blocperso' => 'Titre de bloc perso',
            ),
        );
    }

    /**
     * Retourne un tableau des options de balises HTML recommandées pour les listes (titres, conteneurs, etc.)
     * @return array Tableau des options de balises HTML recommandées pour les listes
     */
    public function getRecommendedMarkupOptions() {
        return array(
            'tag_list_container' => 'div',
            'tag_list_pagination' => 'nav',
            'tag_listitem_container' => 'div',
            'tag_listitem_titre' => 'h2',
            'tag_listitem_commentaire' => 'p',
            'tag_moteur_titre_section' => 'h2',
            'tag_moteur_titre_subsection' => 'h2',
            'tag_moteur_titre_filtre' => 'span',
            'tag_fiche_titre_infosupplementaires1' => 'h2',
            'tag_fiche_titre_infosupplementaires2' => 'h2',
            'tag_fiche_titre_infosupplementaires3' => 'h2',
            'tag_fiche_titre_infosupplementaires4' => 'h2',
            'tag_fiche_titre_infosupplementaires5' => 'h2',
            'tag_fiche_titre_groupe_blocperso' => 'h2',
            'tag_fiche_titre_blocperso' => 'h2',
        );
    }
    /**
     * @return array[] Tableau des options de balises HTML pour les listes déroulantes de sélection dans l'administration
     */
    public function getHtmlTagsForMarkupOptions() {
        return array(
            array('label' => '-- Sélectionnez --', 'value' => ''),
            array('label' => 'H1', 'value' => 'h1'),
            array('label' => 'H2', 'value' => 'h2'),
            array('label' => 'H3', 'value' => 'h3'),
            array('label' => 'H4', 'value' => 'h4'),
            array('label' => 'DIV', 'value' => 'div'),
            array('label' => 'SPAN', 'value' => 'span'),
            array('label' => 'P', 'value' => 'p'),
            array('label' => 'MAIN', 'value' => 'main'),
            array('label' => 'NAV', 'value' => 'nav'),
        );
    }
    /**
     * Retourne l'URL relative pour les appels AJAX des templates front-end qui lancent la récupération des données des listes
     */
    public function getAjaxURL() {
        // 08/11/2025 : fix point entrée ajax générique pour compatibilité bedrock setup
        $adminAjaxUrl = admin_url('admin-ajax.php');
        $relativeUrl = wp_parse_url($adminAjaxUrl, PHP_URL_PATH);

        if (!empty($relativeUrl)) {
            return $relativeUrl;
        }

        return '/wp-admin/admin-ajax.php';
    }


    /**
     * Met à jour un réglage de Bridge en base de données
     *
     * @param $optionName
     * @param $optionValue
     */
    public function bridgeUpdateOption($optionName, $optionValue) {
        global $wpdb;

        $serializedOptionValue = serialize($optionValue);
        // DN 30.06.2023 ; on répare la chaine car lorsqu'elle contient des \ ça merde
        //        $data = preg_replace('!s:(\d+):"(.*?)";!e', "'s:'.strlen('$2').':\"$2\";'", $data);
        $sql = "INSERT INTO " . $wpdb->prefix . "options (option_name, option_value, autoload) VALUES ";
        $sql .= "(%s, %s, 'yes') ";
        $sql .= "ON DUPLICATE KEY UPDATE option_value = %s ;";
        $sql = $wpdb->prepare($sql, $optionName, $serializedOptionValue, $serializedOptionValue);
        return $wpdb->query($sql);
    }

    /**
     * @return string code de la langue courante sur 2 lettres
     */
    public function getLanguage()
    {
        return $this->bridgeCmsAbstractLayer->getCurrentLanguage();
    }

    /**
     * @return string code de la langue courante sous la forme fr-FR ou en-GB (langue - pays)
     */
    public function getLanguageLocale()
    {
        return $this->bridgeCmsAbstractLayer->getCurrentLanguageLocale();
    }

    /**
     * DRUPAL : Fonction indiquant le préfixe de langue à ajouter aux liens URL - sans objet dans Wordpress
     */
    public function getLanguagePrefix() {
        return '';
    }

    /**
     * @return mixed|string URL du serveur Pylot Bridge (application distante)
     */
    public function getBridgeUrl()
    {
        $options = get_option('reglages_sit_bridge_option');
        if (isset($options['url_bridge']))
            return $options['url_bridge'];
        else
            return '';
    }
    /**
     * Retounre l'URL de Pylot Bridge telle qu'enregistrée dans les paramètres
     * @return array|mixed|null URL de Pylot Bridge
     */
    public function getUrlBridge() {
        return $this->getBridgeUrl();
    }

    /**
     * @return mixed|string URL de la feuille de styles personnalisée définie dans le paramètres de l'extension
     */
    public function getCustomCssURl()
    {
        $options = get_option('reglages_sit_design_option');
        if (isset($options['custom_css_sit']))
            return $options['custom_css_sit'];
        else
            return '';
    }

    /**
     * @return mixed|string URL du script personnalisé défini dans le paramètres de l'extension
     */
    public function getCustomJsURl()
    {
        $options = get_option('reglages_sit_design_option');
        if (isset($options['custom_js_sit']))
            return $options['custom_js_sit'];
        else
            return '';
    }

    /**
     * @return object Paramètres de l'extension relatifs à l'affichage des cartes
     */
    public function getMapsDefaultSettings()
    {
        $options = get_option('reglages_sit_design_option');
        $obj = new \stdClass();
        $obj->latitude = '48.8534';
        if (isset($options['latitude']) && !empty($options['latitude']))
            $obj->latitude = $options['latitude'];
        $obj->longitude = '2.3488';
        if (isset($options['longitude']) && !empty($options['longitude']))
            $obj->longitude = $options['longitude'];

        $assetsBaseUrl = trailingslashit($this->bridgeDirUrl() . 'assets/img');
        $obj->marker = $assetsBaseUrl . 'map-marker-icon.png';
        if (!empty($options['marker_map']))
            $obj->marker = $options['marker_map'];

        $obj->activeMarker = $assetsBaseUrl . 'map-marker-icon.png';
        if (!empty($options['marker_map_actif']))
            $obj->activeMarker = $options['marker_map_actif'];

        $obj->activeGeoloc = false;
        if (!empty($options['active_geoloc_maps']) && $options['active_geoloc_maps'] == '1')
            $obj->activeGeoloc = true;

        $obj->defaultGeolocIconUrl = $assetsBaseUrl . 'geoloc-icon.svg';
        if (!empty($options['marker_geoloc_actif']) )
            $obj->defaultGeolocIconUrl = $options['marker_geoloc_actif'];

        return $obj;
    }

    /**
     * @return false|mixed|string Chemin de base relatif à la réécriture des URL de taxonomies (hors permalien personnalisé)
     */
    public function getTaxonomyRootUrl()
    {
        $options = get_option('reglages_sit_avance_option');
        if (!empty($options['taxonomy_root_url'])) {
            $temp = $options['taxonomy_root_url'];
            if (substr($temp, 0, 1) == '/')
                $temp = substr($temp, 1);
            return $temp;
        } else {
            return 'sit';
        }
    }

    /**
     * @return false|mixed|string Chemin de base relatif à la réécriture des URL des fiches (hors permalien personnalisé)
     */
    public function getFicheRootUrl()
    {
        $options = get_option('reglages_sit_avance_option');
        if (!empty($options['fiche_root_url'])) {
            $temp = $options['fiche_root_url'];
            if (substr($temp, 0, 1) == '/')
                $temp = substr($temp, 1);
            return $temp;
        } else {
            return 'fiche-sit';
        }
    }

    /**
     * Retourne l'URL du dossier du plugin Pylot Bridge
     * @return string URL du dossier du plugin Pylot Bridge
     */
    public function bridgeDirUrl()
    {
        return plugin_dir_url(realpath(dirname(__FILE__) . '/../../plugin-sit.php'));
    }

    /**
     * Retournre le chemin physique du dossier du plugin Pylot Bridge
     * @return string Chemin du dossier du plugin Pylot Bridge
     */
    public function bridgeDirPath()
    {
        return plugin_dir_path(realpath(dirname(__FILE__) . '/../../plugin-sit.php'));
    }

    /**
     * Retourne le chemin physique du dossier des telmplates d'affichage du plugin Pylot Bridge
     * @return string URL du dossier des telmplates d'affichage du plugin Pylot Bridge
     */
    public function bridgeTemplatesPath()
    {
        return $this->bridgeDirPath().'public/templates/';
    }

    /**
     * Retourne l'URL du script php de resizer d'image Bridge
     * @return string URL du resizer d'image Bridge
     */
    public function getResizeImage() {
        return $this->bridgeDirUrl().'assets/resize_image_dnc.php';
    }

    /**
     * getResizeImageLink
     * Génère un lien d'image redimensionnée selon les paramètres passés
     * @param $url URL de la photo originale
     * @param string $mode mode de redimensonnement :
     * - ajust pour ajuster aux dimensions passées (ne déborde pas)
     * - remplir : pour remplir le cadre de dimensions passées
     * - deform : pour déformer l'image et forcer les dimensions exactes
     * @param string $width largeur souhaitée
     * @param string $height hauteur souhaitée
     * @param string $quality qualité de compresion (sur 100 pour le jpg et de 0 à 9 pour le png)
     * @param int $timeToCache durée de cache navigateur à indiquer dans l'en-tête http
     * @return string l'URL du service qui affiche les données d'image redimensionnée
     */
    public function getResizeImageLink($url, $mode = "ajust", $width = "150", $height = "150", $quality = "60", $timeToCache = "1800") {
        // Update 06/09/2024 : on utilise désormais un point d'entrée dans piblic sit public qui gère la mise en cache des images
        $queryArgs = array(
            'brImageResize' => 1,
            'lemode' => $mode,
            'selwidth' => $width,
            'selheight' => $height,
            'def' => $quality,
            'timeToCache' => $timeToCache,
            'file' => $url,
        );

        return add_query_arg($queryArgs, home_url('/'));
        // return $this->bridgeDirUrl()."assets/resize_image_dnc.php?lemode=$mode&selwidth=$width&selheight=$height&def=$quality&timeToCache=$timeToCache&file=" . urlencode($url);
    }

    /**
     * Retourne un objet contenant les paramètres de connexion au serveur Bridge (serveur et identifiants Bridge)
     *
     * @return object objets contenant les paramètres de connexion au serveur Bridge
     */
    public function getBridgeCredentials()
    {
        $options = get_option('reglages_sit_bridge_option');
        $obj = new \stdClass();
        $obj->urlBridge = '';
        if (isset($options['url_bridge']))
            $obj->urlBridge = $options['url_bridge'];
        $obj->loginBridge = '';
        if (isset($options['login_bridge']))
            $obj->loginBridge = $options['login_bridge'];
        $obj->passwordBridge = '';
        if (isset($options['password_bridge']))
            $obj->passwordBridge = $options['password_bridge'];

        return $obj;
    }

    /**
     * Permet de récupérer un token de connexion à l'API Bridge à partir d'identifiants
     *
     * @param string $url Url du serveur Bridge
     * @param string $login login Bridge
     * @param string $pw mot de passe Bridge
     * @throws \Exception
     * @return string token de connexion
     */
    public function getBridgeTokenFromCredentials(string $url, string $login, string $pw) {

        if ($url != '' && $login != '' && $pw != '') {
            $data = array(
                'username' => $login,
                'password' => $pw
            );
            $options = array(
                'http' => array(
                    'method' => 'POST',
                    'header' => "Content-Type: application/json",
                    'ignore_errors' => true,
                    'timeout' => 10,
                    'content' => json_encode($data),
                )
            );

            $context = stream_context_create($options);
            $res = json_decode(file_get_contents($url . '/login_check', false, $context));

            if (isset($res) && isset($res->token)) {
                return $res->token;
            } else {
                throw new \Exception("Invalid credentials");
            }
        }
        return '';
    }

    /**
     * Permet de récupérer le token de connexion à l'API Bridge
     * @return string token de connexion à l'API Bridge
     * @throws \Exception
     */
    public function getBridgeToken(bool $useCache = true)
    {
        if($useCache) {
          $resCache = $this->requestService->getCachedData('bridgeApiToken', 120);
          if(!empty($resCache)) {
              return $resCache;
          }
        }

        $tmpOptions = $this->getBridgeCredentials();
        $urlBridge = $tmpOptions->urlBridge;
        $loginBridge = $tmpOptions->loginBridge;
        $passwordBridge = $tmpOptions->passwordBridge;

        $token = $this->getBridgeTokenFromCredentials($urlBridge, $loginBridge, $passwordBridge);
        if($useCache) {
            $this->requestService->storeCachedData('bridgeApiToken', $token);
        }
        return $token;
    }

    /**
     * Retounre l'id de site Pylot Bridge telle qu'enregistré dans les paramètres
     * @return array|mixed|null id de site Pylot Bridge
     */
    public function getSiteBridge() {
        $options = get_option('reglages_sit_import');
        if (isset($options['site']))
            return $options['site'];
        else
            return '';
    }

    /**
     * Retourne l'id du bock photos par défaut de Pylot Bridge tel qu'enregistré dans les paramètres
     * @return array|mixed|null id de bloc photos
     */
    public function getIdBlockPhotos() {
        $options = get_option('reglages_sit_import'); // iste et
        if (isset($options['product_media_block']))
            return $options['product_media_block'];
        else
            return '';
        // fiche_root_url
    }

    /**
     * Retourne un tableau contenant les paramètres généraux du plugin Pylot Bridge ainsi que des données d'environnement
     * @return array tablea de paramètres
     */
    public function getBridgeParameters() {
        // On récupère les paramètres de Bridge pour les passer au Twig
        $bridgeParameters = array();
        $optionsImport = get_option('reglages_sit_import');
        $optionsBridge = get_option('reglages_sit_bridge_option');
        $optionsDesign = get_option('reglages_sit_design_option');
        $optionsBalises = get_option('reglages_sit_markup_option');

        // Paramètres d'affichage des balises sémantiques
        if (is_array($optionsBalises) && !empty($optionsBalises)) {
            foreach ($optionsBalises as $key => $value) {
                $bridgeParameters[$key] = $value;
            }
        }

        $bridgeParameters['fiches_iframe'] = '';
        if(isset($optionsDesign['fiches_iframe']) && !empty($optionsDesign['fiches_iframe'])) {
            $bridgeParameters['fiches_iframe'] = $optionsDesign['fiches_iframe'];
        }

        $bridgeParameters['marker_map'] = '';
        $bridgeParameters['marker_map_actif'] = '';
        if(isset($optionsDesign['marker_map']) && !empty($optionsDesign['marker_map'])) {
            // $idmarker_map = BridgeUtils::get_attachment_id((string)$optionsDesign['marker_map']);
            $bridgeParameters['marker_map'] = $optionsDesign['marker_map']; // wp_get_attachment_url($idmarker_map);
        }
        if(isset($optionsDesign['marker_map_actif']) && !empty($optionsDesign['marker_map_actif'])) {
            //$idmarker_map = BridgeUtils::get_attachment_id((string)$optionsDesign['marker_map_actif']);
            $bridgeParameters['marker_map_actif'] = $optionsDesign['marker_map_actif'] ; // wp_get_attachment_url($idmarker_map);
        }

        if(isset($optionsDesign['marker_render']) && !empty($optionsDesign['marker_render'])) {
            $bridgeParameters['marker_render'] = $optionsDesign['marker_render'] ;
        }

        $bridgeParameters['url_bridge'] = '';
        if(isset($optionsBridge['url_bridge']))
            $bridgeParameters['url_bridge'] = $optionsBridge['url_bridge'];

        $bridgeParameters['custom_css_sit'] = '';
        if(isset($optionsDesign['custom_css_sit']))
            $bridgeParameters['custom_css_sit'] = $optionsDesign['custom_css_sit'];

        $bridgeParameters['custom_js_sit'] = '';
        if(isset($optionsDesign['custom_js_sit']))
            $bridgeParameters['custom_js_sit'] = $optionsDesign['custom_js_sit'];

        $bridgeParameters['bridge_site'] = '';
        if(isset($optionsImport['site']))
            $bridgeParameters['bridge_site'] = $optionsImport['site'];

        $bridgeParameters['product_media_block'] = '';
        if(isset($optionsImport['product_media_block']))
            $bridgeParameters['product_media_block'] = $optionsImport['product_media_block'];


        $bridgeParameters['maps_center_lat'] = '';
        if(isset($optionsDesign['latitude']))
            $bridgeParameters['maps_center_lat'] = $optionsDesign['latitude'];

        $bridgeParameters['maps_center_lon'] = '';
        if(isset($optionsDesign['longitude']))
            $bridgeParameters['maps_center_lon'] = $optionsDesign['longitude'];

        if(empty($bridgeParameters['maps_center_lat']))
            $bridgeParameters['maps_center_lat'] = 48.856614;
        if(empty($bridgeParameters['maps_center_lon']))
            $bridgeParameters['maps_center_lon'] = 2.3522219;

        $rootUrl = home_url();
        $bridgePath = $this->bridgeDirUrl();
        $bridgeParameters['url_root'] = $rootUrl;
        $bridgeParameters['url_module_bridge'] = $bridgePath;
        $uri_parts = explode('?', $_SERVER['REQUEST_URI'], 2);
        $bridgeParameters['currentUrl'] = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://$_SERVER[HTTP_HOST]$uri_parts[0]" ;

        // Paramètres d'URL pour la pagination des listes
        $paginationUrlParams = $_SERVER['QUERY_STRING'] ;
        if(!empty($paginationUrlParams)) {
            $tabUrlParams = array();
            parse_str($paginationUrlParams, $tabUrlParams);
            // On retire le paraetre brpa de l'url courante pour construire les liens de pagination
            if(isset($tabUrlParams['brpa'])) {
                unset($tabUrlParams['brpa']);
                $paginationUrlParams = http_build_query($tabUrlParams);
            }
            if(!empty($paginationUrlParams))
                $paginationUrlParams = '?' . $paginationUrlParams;
        }
        $bridgeParameters['paginationUrlParams'] = $paginationUrlParams;

        if(empty($bridgeParameters['marker_map']))
            $bridgeParameters['marker_map'] = $bridgeParameters['url_module_bridge'] . '/assets/img/map-marker-icon.png';
        if(empty($bridgeParameters['marker_map_actif']))
            $bridgeParameters['marker_map_actif'] = $bridgeParameters['marker_map'] ;

        // On donne des dimensions proportionnelles au marker avec en visée 50 px de haut
        if(strtolower(substr($bridgeParameters['marker_map'], -3)) == 'svg') {
            $markerImage = $this->requestService->cachedFileGetContent($bridgeParameters['marker_map']);
            if(!empty($markerImage)) {
                try {
                    $xml = simplexml_load_file( $bridgeParameters['marker_map'] );

                    $attr = $xml->attributes();
                    if ( is_object( $attr ) && isset( $attr->width ) && isset( $attr->height ) ) {
                        $ratio                                 = 50 / (int) $attr->height;
                        $bridgeParameters['marker_map_width']  = round( (int) $attr->width * $ratio );
                        $bridgeParameters['marker_map_height'] = round( (int) $attr->height * $ratio );
                    } else {
                        $bridgeParameters['marker_map_width']  = 35;
                        $bridgeParameters['marker_map_height'] = 50;
                    }
                } catch ( \Exception $e ) {
                    $bridgeParameters['marker_map_width']  = 35;
                    $bridgeParameters['marker_map_height'] = 50;
                }
            }
        } else {
            $image_info = getImageSize($bridgeParameters['marker_map']);
            if(isset($image_info[1]))
                $ratio = 50 / $image_info[1];
            else
                $ratio = 1;
            $bridgeParameters['marker_map_width'] = round($image_info[0] * $ratio);
            $bridgeParameters['marker_map_height'] = round($image_info[1] * $ratio);
        }

        if(strtolower(substr($bridgeParameters['marker_map_actif'], -3)) == 'svg') {
            $markerActifImage = $this->requestService->cachedFileGetContent($bridgeParameters['marker_map_actif']);
            if(!empty($markerActifImage)) {
                try {
                    $xml = simplexml_load_file( $bridgeParameters['marker_map_actif'] );

                    $attr = $xml->attributes();
                    if ( is_object( $attr ) && isset( $attr->width ) && isset( $attr->height ) ) {
                        $ratio                                       = 50 / (int) $attr->height;
                        $bridgeParameters['marker_map_actif_width']  = round( (int) $attr->width * $ratio );
                        $bridgeParameters['marker_map_actif_height'] = round( (int) $attr->height * $ratio );
                    } else {
                        $bridgeParameters['marker_map_actif_width']  = 35;
                        $bridgeParameters['marker_map_actif_height'] = 50;
                    }
                } catch ( \Exception $e ) {
                    $bridgeParameters['marker_map_actif_width']  = 35;
                    $bridgeParameters['marker_map_actif_height'] = 50;
                }
            }
        } else {
            $image_info = getImageSize($bridgeParameters['marker_map_actif']);
            if(isset($image_info[1]))
                $ratio = 50 / $image_info[1];
            else
                $ratio = 1;
            $bridgeParameters['marker_map_actif_width'] = round($image_info[0] * $ratio);
            $bridgeParameters['marker_map_actif_height'] = round($image_info[1] * $ratio);
        }

        // On ajoute aux paramètres d'application les constantes système de manière à tout avoir sous la main et ne pas avoir à faire trop d'appels
        $bridgeParameters['url_ajax_liste'] = $rootUrl . '/pylot_bridge/list_products_json' ;
        $bridgeParameters['url_ajax_liste_json_poi'] = $rootUrl . '/pylot_bridge/list_products_json_for_map_poi' ;

        // Et la langue courante tant qu'on y est
        $bridgeParameters['lang'] = $this->getLanguage();

        // Facebook App Id pour la social box
        $bridgeParameters['facebookAppId'] = '495036869019149';
        $bridgeParameters['facebookAppNonce'] = 'awkTZqCe';

        return $bridgeParameters;
    }

	/**
	 * @return mixed|string Pour la demande d'import en tâche de fond
	 */
	public function getNeedToImport()
	{
		global $wpdb;
		$currentDbPrefix = $wpdb->prefix;
		if(is_multisite()) {
			$blogId = get_current_blog_id();
			$currentDbPrefix = $wpdb->get_blog_prefix($blogId);
		}

		// FIX ND : pb cache : on passe en SQL direct
		$sql = "SELECT option_value FROM " . $currentDbPrefix. "options WHERE option_name='need_to_import' ;";
		$test = $wpdb->get_results($sql);
		if (!empty($test) && is_array($test)) {
			return $test[0]->option_value;
		}
		return '';
	}

	/**
	 * @return mixed|string Pour la demande d'import en tâche de fond
	 */
	public function setNeedToImport($val)
	{
		global $wpdb;
		$currentDbPrefix = $wpdb->prefix;
		if(is_multisite()) {
			$blogId = get_current_blog_id();
			$currentDbPrefix = $wpdb->get_blog_prefix($blogId);
		}

		// FIX ND : pb cache : on passe en SQL direct
		$sql = "SELECT option_value FROM " . $currentDbPrefix. "options WHERE option_name='need_to_import' ;";
		$test = $wpdb->get_results($sql);
		if (!empty($test) && is_array($test)) {
			$sql = "UPDATE " . $currentDbPrefix . "options SET option_value=%s WHERE option_name='need_to_import' ;";
			$numrows = $wpdb->query($wpdb->prepare($sql, $val));
			$debugMsg = $sql;
		} else {
			$sql = "INSERT INTO " . $currentDbPrefix . "options (option_name, option_value) VALUES ('need_to_import', %s);";
			$numrows = $wpdb->query($wpdb->prepare($sql, $val));
		}
		return $numrows;
	}

	public function getCvPreviewLink() {
		$cvPreviewLink = '';
		$cvPreviewId = get_option("reglages_sit_design_option")["cv_preview_page"];
		if(!empty($cvPreviewId))
		{
			$cvPreviewLink = @get_page_link($cvPreviewId);
            if(function_exists('pll_get_post')){
              $cvPreviewLink = @get_page_link(pll_get_post($cvPreviewId));
            }
		}
		return $cvPreviewLink;
	}
	public function getCvDownloadLink() {
		return '/index.php?bridge_dl_carnet=1';
	}

	public function getLastImportLog() {

	}
	public function setLastImportLog($txt) {

	}

    /**
     * @return string URL du dossier des images de l'application
     */
    public function getBridgeImgUrl() {
        return $this->bridgeDirUrl() . 'public/bridgewebfrontlibs/img';
    }

    public function cachedFileGetContent(string $url, bool $use_include_path = false, $options = null, int $cacheTimeMinutes = 120) {
        return $this->requestService->cachedFileGetContent($url, $use_include_path, $options, $cacheTimeMinutes);
    }
}
