<?php

namespace Bridge\Weblibs\Wordpress;

use Bridge\Weblibs\BridgeCmsAbstractLayerInterface;

/**
 * Couche d'abstraction au CMS
 *
 *
 * @package    Plugin_SIT
 * @subpackage Plugin_SIT/includes
 * @author     DN Consultants <info@dnconsultants.fr>
 */

class BridgeCmsAbstractLayerWordpress implements BridgeCmsAbstractLayerInterface {

    /**
     * Retourne vrai si le script a été enregistré / chargé
     * @param $handle : clé texte du script
     * @param $list : statut à vérifier : enqueued, registered
     * @return mixed
     */
    public function scriptIs($handle, $list ) {
        return wp_script_is($handle, $list  = 'enqueued' );
    }
    /**
     * Ajouter un script JS à la page courante
     * Si src est absent, le handle du script doit faire référence à un script préalablement déclaré avec registerScript
     * @param $handle : id texte du script
     * @param string $src : url du script
     * @param array $deps : tableau de dépendances (liste d'id texte)
     * @param false $ver : version
     * @param false $in_footer : charger dans le footer
     * @return mixed
     */
    public function registerScript($handle, $src = "", $deps = array(), $ver = false, $in_footer = false ) {
        return wp_register_script($handle, $src, $deps, $ver, $in_footer);
    }

    /**
     * Ajouter un script JS à la page courante
     * Si src est absent, le handle du script doit faire référence à un script préalablement déclaré avec registerScript
     * @param $handle : id texte du script
     * @param string $src : url du script
     * @param array $deps : tableau de dépendances (liste d'id texte)
     * @param false $ver : version
     * @param false $in_footer : charger dans le footer
     * @return mixed
     */
    public function enqueueScript($handle, $src = "", $deps = array(), $ver = false, $in_footer = false ) {
        return wp_enqueue_script($handle, $src, $deps, $ver, $in_footer);
    }

    /**
     * Référencer une feuille de style CSS : associer un id texte à une URL
     * Si src est absent, le handle du css doit faire référence à un fichier préalablement déclaré avec registerStyle
     * @param $handle : id texte du css
     * @param string $src : url du css
     * @param array $deps : tableau de dépendances (liste d'id texte)
     * @param false $ver : version
     * @param string $media
     * @return mixed
     */
    public function registerStyle($handle, $src = "", $deps = array(), $ver = false, $media = 'all' ) {
        return wp_register_style($handle, $src, $deps, $ver, $media);
    }

    /**
     * Ajouter une feuille de style CSS à la page courante
     * Si src est absent, le handle du css doit faire référence à un fichier préalablement déclaré avec registerStyle
     * @param $handle : id texte du css
     * @param string $src : url du css
     * @param array $deps : tableau de dépendances (liste d'id texte)
     * @param false $ver : version
     * @param string $media
     * @return mixed
     */
    public function enqueueStyle($handle, $src = "", $deps = array(), $ver = false, $media = 'all' ) {
        return wp_enqueue_style($handle, $src, $deps, $ver, $media);
    }

    /**
     * @return Retourne le code de la langue courante (2 lettres en minuscules / fr par défaut)
     */
    public function getCurrentLanguage($noCookie = false)
    {

        $lang = 'fr';
        // Polylang
        if (function_exists('pll_current_language')) {
            $lang = pll_current_language();

        }

        // ND 17/09/2025 : on on utilise plus le cookie car pose problème avec les plugins de cache ?
        if($noCookie == false && isset($_COOKIE['pll_language']))
            $lang = $_COOKIE['pll_language'];
        // $lang = isset($_COOKIE['pll_language']) ? $_COOKIE['pll_language'] : $lang;

        // Multilingual press 2
        if (function_exists('mlp_get_current_blog_language')) {
            $lang = mlp_get_current_blog_language();
            if(mb_strlen($lang) > 2) {
                $lang = mb_substr($lang,0,2);
            }
        }
        // Gtranslate
        if(isset($_SERVER['HTTP_X_GT_LANG']) && !headers_sent()) {
            // 16/09/2025 : pourquoi ne pas retourner directement $_SERVER['HTTP_X_GT_LANG'] ? J'ajoute la ligne
            $lang = $_SERVER['HTTP_X_GT_LANG'];
            $options = get_option('reglages_sit_import');
            if(isset($options) && isset($options['gtranslate_lang_import'])) {
                $allowLang = $options['gtranslate_lang_import'];
                if(isset($allowLang) && in_array($_SERVER['HTTP_X_GT_LANG'],$allowLang)) {
                    $lang = $_SERVER['HTTP_X_GT_LANG'];
                    header('Cache-Control: no-cache');
                }
            }
        }

        return ($lang != '') ? $lang : 'fr';
    }

    /**
     * @return Retourne le code de la langue courante sous la forme fr-FR ou en-GB (langue - pays)
     */
    public function getCurrentLanguageLocale()
    {
        $lang = 'fr';
        // Polylang
        if (function_exists('pll_current_language'))
            $lang = pll_current_language('locale');
        // Multilingual press 2
        if (function_exists('mlp_get_current_blog_language')) {
            $lang = mlp_get_current_blog_language();
            if(mb_strlen($lang) > 2) {
                $lang = mb_substr($lang,0,2);
            }
        }
        // Gtranslate // On vérifie que l'option a été renseigné dans wordpress
        if(isset($_SERVER['HTTP_X_GT_LANG'])){
            $options = get_option('reglages_sit_import');
            if(isset($options)){
                $allowLang = $options['gtranslate_lang_import'];
                if(isset($allowLang) && in_array($_SERVER['HTTP_X_GT_LANG'],$allowLang)){
                    //$lang = $_SERVER['HTTP_X_GT_LANG'];
                    header('Cache-Control: no-cache');
                }
            }
        }

        return ($lang!='') ? $lang : 'fr';
    }

    /**
     * Retourne le code de langue d'un post donné (wordpress)
     * @param $postId numéro de post
     * @return string
     */
    public function getPostLanguage($postId)
    {
        $lang = 'fr';

        // Polylang
        if (function_exists('pll_get_post_language'))
            $lang = pll_get_post_language($postId);
        // Multilingual press 2 => Comme le fonctionnement est multidomaine, on regarde juste la langue du blog en cours
        if (function_exists('mlp_get_current_blog_language')) {
            $lang = mlp_get_current_blog_language();
            if(mb_strlen($lang) > 2) {
                $lang = mb_substr($lang,0,2);
            }
        }
        // Gtranslate
        if(isset($_SERVER['HTTP_X_GT_LANG'])){
            $lang = $_SERVER['HTTP_X_GT_LANG'];
            header('Cache-Control: no-cache');
        }
        return ($lang!='') ? $lang : 'fr';

    }

    /**
     * Définit le code de langue d'un post donné (surtout pour WordPress qui ne gère pas nativement le multilingue
     * @param $postId : id du post
     * @return string code de langue sur 2 lettres
     */
    public function setPostLanguage($postId, $lang)
    {
        // Polylang
        if (function_exists('pll_set_post_language'))
            return pll_set_post_language($postId, $lang);
        return true;
    }

    /**
     * Sauvegarde les traductions d'un produit (WordPress)
     * @param $translationsArray tableau associatif $lang => $postId ('fr' => 4)
     */
    public function savePostTranslations($translationsArray)
    {
        // Polylang
        if (function_exists('pll_save_post_translations'))
            return pll_save_post_translations($translationsArray);
        return true;
    }

    /**
     * Définit le code de langue d'un post donné (WordPress)
     * @param $termId : id du terme
     * @return string
     */
    public function setTermLanguage($termId, $lang)
    {
        // Polylang
        if (function_exists('pll_set_term_language'))
            return pll_set_term_language($termId, $lang);
        return true;
    }

    /**
     * Retourne la langue d'un terme (WordPress)
     * @param $termId : id du terme
     * @return string code de langue
     */
    public function getTermLanguage($termId)
    {
        $lang = 'fr';

        // GTranslate
        $dataGT = get_option('GTranslate');
        if(isset($dataGT) && !empty($dataGT) && class_exists('GTranslate')){
            // En mode Gtranslate on a pas de posts traduits => on force le FR
            return 'fr';
        }

        // Polylang
        if (function_exists('pll_get_term_language'))
            $lang = pll_get_term_language($termId);
        // Multilingual press 2 => Comme le fonctionnement est multidomaine, on regarde juste la langue du blog en cours
        if (function_exists('mlp_get_current_blog_language')) {
            $lang = mlp_get_current_blog_language();
            if(mb_strlen($lang) > 2) {
                $lang = mb_substr($lang,0,2);
            }
        }
        return ($lang!='') ? $lang : 'fr';
    }

    /**
     * Retourne la langue d'un terme de taxonomie (WordPress)
     * ND 12/09/2022 : après recherche dans le code cette fonction n'est jamais utilisée
     * @param $termId : id du terme
     * @return array taleau associatif lang => id de terme
     */
    public function getTermTranslations($termId)
    {
        $trans = array();

        // Polylang
        if (function_exists('pll_get_term_translations'))
            $trans = pll_get_term_translations($termId);

        return $trans;
    }

    /**
     * Sauvegarde les traductions d'un terme (Wordpress)
     * @param $translationsArray tableau associatif $lang => $termId ('fr' => 4)
     */
    public function saveTermTranslations($translationsArray)
    {

        // Polylang
        if (function_exists('pll_save_term_translations'))
            return pll_save_term_translations($translationsArray);
        return true;
    }

    /**
     * Retourne la liste des langues actives
     * @return string[] tableau de codes de langues
     */
    public function getLanguagesList()
    {
        $langs = array('fr');

        // Polylang
        if (function_exists('pll_languages_list')) {
            $tabLangs = pll_languages_list();
            foreach ($tabLangs as $key => $lang) {
                if ($lang != 'fr')
                    $langs[] = $lang;
            }

        }
        return $langs;
    }

    /**
     * Retourne le permalien associé à une weblist
     * @param $term_id : id de term lié à la lweblist
     * @param string $lang : langue en cours
     * @return string : Début de l'URL pour la réécriture des liens vers les fiches de la liste'
     */
    public function getPermalinkFromTermId($term_id, $lang = '') {
        $taxonomy = 'rubrique_sit_primaire';
        if(empty($lang))
            $lang = $this->getCurrentLanguage();
        $langUpper = strtoupper($lang);

        $permalink = get_term_meta($term_id, 'permalien', true);
        if($permalink != '') {
            return str_replace('//', '/', $permalink . '/');
        }

        $destinationWebListId = get_term_meta($term_id, 'weblist_destinationWebListId', true);
        if(!empty($destinationWebListId)) {
            // Lien vers une liste de destination qui n'a pas de permalien personnalisé
            // On retrouve le terme associé à la liste de destination pour en retrouver le lien
            // Nota : si la weblist de destination avait un permalien perso, on l'aurait recopié dans le terme courant
            $args = array(
                'hide_empty' => false, // also retrieve terms which are not used yet
                'meta_query' => array(
                    array(
                        'key'       => 'weblist_id',
                        'value'     => $destinationWebListId,
                        'compare'   => '='
                    )
                ),
                'taxonomy'  => $taxonomy,
            );
            $terms = get_terms( $args );
            $term = null;

            if(count($terms) > 0) {
                foreach($terms as $termTemp) {
                    $tempLang = $this->getTermLanguage($termTemp->term_id);
                    $deb = $termTemp->term_id;
                    if($tempLang == $lang) {
                        $term = $termTemp;
                        break;
                    }
                }
            }
            if(isset($term)){
                return str_replace(trailingslashit(home_url()), '', get_term_link((int) $term_id));
            } else {
                return "#errDest#Term" . $term_id . "#" . count($terms) . "#" . $deb . "#" . $term_id . "#";
            }
        } else {
            $link = get_term_link((int) $term_id);
            return str_replace(trailingslashit(home_url()), '', $link);
        }
    }


    public function get_site_url() {
        return home_url();
    }

    public function getPostPermalink($postId) {
        return get_post_permalink($postId);
    }

    /**
     * Retourne le header du theme front-end du site pour affichage avant le code de la fichede détail (WordPress)
     * @return mixed
     */
    public function getHeaderBridge($arg = null) {
        return get_header($arg);
    }

    /**
     * Retourne le footer du theme front-end du site pour affichage après le code de la fichede détail (WordPress)
     * @return mixed
     */
    public function getFooterBridge() {
        return get_footer();
    }



    /**
     * Renvoie une reponse http Json d'erreur
     * @param mixed|null $data données à renvoyer en Json
     * @param int|null $status_code code HTTP à renvoyer
     * @param int $options options d'après les specifications de la fonction wp_send_json_error
     */
    public function sendJsonError( $data = null, int $status_code = null, int $options = 0) {
        wp_send_json_error( $data, $status_code, $options);
    }

    /**
     * Renvoie une reponse http Json de succes
     * @param mixed|null $data données à renvoyer en Json
     * @param int|null $status_code code HTTP à renvoyer
     * @param int $options options d'après les specifications de la fonction wp_send_json_success
     */
    public function sendJsonSuccess( $data = null, int $status_code = null, int $options = 0) {
        wp_send_json_success( $data, $status_code, $options);
    }

    /**
     * Renvoie une reponse http Json de succes
     * @param mixed|null $data données à renvoyer en Json
     * @param int|null $status_code code HTTP à renvoyer
     * @param int $options options d'après les specifications de la fonction wp_send_json_success
     */
    public function sendJson( $data = null, int $status_code = null, int $options = 0) {
        wp_send_json( $data, $status_code, $options);
    }

    /**
     * Permet de déclarer une fonction de callback pour insérer du contenu dans les vues Bridge sans avoir à les overrider
     * éuivalent à wordpress add_action
     * @param $hookName : nom du hook
     * @param $callBack : fonction de callback
     * @param int $priority : priorité
     * @param int $acceptedArgs : nombre d'arguments acceptés
     */
    public function addDisplayHook($hookName, $callBack, $priority = 10, $acceptedArgs = 1) {
        add_action($hookName, $callBack, $priority, $acceptedArgs);
    }

    /**
     * Permet d'insérer un hook d'affichage dans les templates
     * @param string $hookName
     * @param mixed $args
     */
    public function doDisplayHook($hookName, ...$arg) {
        do_action($hookName, ...$arg);
    }

     /**
     * Permet de déclarer une fonction de callback pour retrvailler du contenu dans les vues Bridge sans avoir à les overrider
     * éuivalent à wordpress apply_filter
     * @param $hookName : nom du hook
     * @param $callBack : fonction de callback
     * @param int $priority : priorité
     * @param int $acceptedArgs : nombre d'arguments acceptés
     */
    public function applyFilters($hookName, $callBack, $priority = 10, $acceptedArgs = 1) {
       return apply_filters($hookName,$callBack,$priority,$acceptedArgs);
    }

    /**
     * Permet de déclarer une fonction de callback pour retrvailler du contenu dans les vues Bridge sans avoir à les overrider
     * éuivalent à wordpress apply_filter
     * @param $hookName : nom du hook
     * @param $callBack : fonction de callback
     * @param int $priority : priorité
     * @param int $acceptedArgs : nombre d'arguments acceptés
     */
    public function addFilter($hookName, $callBack, $priority = 10, $acceptedArgs = 1) {
       return add_filter($hookName,$callBack,$priority,$acceptedArgs);
    }

    /**
     * @return false|mixed|string Chemin de base relatif à la réécriture des URL des fiches (hors permalien personnalisé)
     */
    public function getFicheRewritePattern()
    {
        // Obsolète 25/10/2025 : on utilise plus cette option
        return '/(.*)[/]?$';
        /*
         * $options = get_option('reglages_sit_avance_option');
        if (!empty($options['fiche_rewrite_pattern'])) {
            $temp = $options['fiche_rewrite_pattern'];
            return $temp;
        } else {
            return '/(.*)[/]?$';
        } */
    }

    /**
     * Fonction pour WordPress : indique si on se trouve actuellement sur une page de taxonomie SIT
     * @return bool True si on est sur une page de taxonomie SIT
     *
     */
    public function IsFicheSitTaxonomy(){
        return is_tax( get_object_taxonomies( 'fiche_sit' ) );
        // Pareil normalement
        // return is_tax( 'rubrique_sit_primaire' );
    }

    /**
     * @return bool true si on est sur une page de fiche SIT
     */
    public function IsFicheSit() {
        // Patch nicolas 27/04/21 : c'est un underscore !
        return is_singular( array( 'fiche_sit' ) );
    }

    /**
     * fonction appelée depuis un modèle de fiche
     * Retourne le code HTML du fil d'Ariane affiché dans les fiches
     * @param string $delimiter délimiteur d'entrées
     * @param $home texte pour l'entrée 'Accueil'
     * @param int $showCurrent mettre 1 pour afficher l'entre courante
     * @param string $before texte à afficher avant l'élément courant
     * @param string $after texte à afficher après l'élément courant
     *
     * @return string code HTML du fil d'Ariane affiché dans les fiches
     */
    public function BridgeBreadcrumbs($delimiter = '&raquo;',$home = '', $showCurrent = 1,$before = '<span class="current">',$after = '</span>',$showOnHome = true)
    {

        if (!isset($home) || $home == '' && $showOnHome) $home = __('Accueil', 'plugin-sit');

        /*$showOnHome = 0; // 1 - show breadcrumbs on the homepage, 0 - don't show
        $delimiter = '&raquo;'; // delimiter between crumbs
        $home = 'Home'; // text for the 'Home' link
        $showCurrent = 1; // 1 - show current post/page title in breadcrumbs, 0 - don't show
        $before = '<span class="current">'; // tag before the current crumb
        $after = '</span>'; // tag after the current crumb */
        global $post;

        $retour = '';
        $tabBreadCrumbs = [];
        $tabBreadCrumbsRubriquePrincipale = [];
        if (get_post_type() != 'fiche_sit') {
            return "Votre article n'est pas une fiche SIT";
        }

        if($showOnHome){
            $homeLink = get_bloginfo('url');
            $tabBreadCrumbs[] = '<a href="' . $homeLink . '">' . $home . '</a> <span class="separator">' . $delimiter . '</span> ';
        }

        if (get_post_type() != 'fiche_sit') {
            $post_type = get_post_type_object(get_post_type());
            $slug = $post_type->rewrite;
            $tabBreadCrumbs[] = '<a href="' . $homeLink . '/' . $slug['slug'] . '/" class="breads-'.$slug['slug'].' term-'.$slug['term_id'].'">' . $post_type->labels->singular_name . '</a>';
        } else {
            //$retour .= 'a';
            $cats = wp_get_post_terms($post->ID, 'rubrique_sit_primaire');
/* ancien code : rubrique principale
            $cats = $this->bridge_array_sort($cats, 'term_id', SORT_ASC);
            foreach($cats as $c) {
                $cat = $c;
                $permalien = get_term_meta($c->term_id, 'permalien', true);
                $exlude = get_term_meta($c->term_id, 'exlude_breadcrumbs', true);
                $rubriquePrincipale = get_term_meta($c->term_id, 'weblist_permalinkIsCanonical', true);


                if ($exlude != '1') {
                    $tabBreadCrumbs[] = '<a href="' . $homeLink . '/' . $permalien . '/" class="term-bridge-'.$c->term_id.'">' . $c->name . '</a> <span class="separator">'.$delimiter.'</span> ';
                }
                if ($exlude != '1' && $rubriquePrincipale == true) {
                    $tabBreadCrumbsRubriquePrincipale[] = '<a href="' . $homeLink . '/' . $permalien . '/" class="term-bridge-'.$c->term_id.'">' . $c->name . '</a> <span class="separator">'.$delimiter.'</span> ';
                }

            }
*/

            $currentPath = '';
            if (isset($_SERVER['REQUEST_URI'])) {
                $currentPath = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
            }
            $currentPath = trim($currentPath, '/');
            $homeLinkParsed = parse_url($homeLink, PHP_URL_PATH);
            if($homeLinkParsed === false || $homeLinkParsed === null) {
                $homeLinkParsed = '';
            }
            $homePath = trim($homeLinkParsed, '/');
            if ($homePath !== '' && strpos($currentPath, $homePath) === 0) {
                $currentPath = trim(substr($currentPath, strlen($homePath)), '/');
            }

            $matchedTerm = null;
            $matchedLength = 0;
            if (!empty($cats) && $currentPath !== '') {
                foreach ($cats as $catCandidate) {
                    $termPermalink = trim((string) get_term_meta($catCandidate->term_id, 'permalien', true), '/');
                    if ($termPermalink === '') {
                        continue;
                    }

                    if ($currentPath === $termPermalink || strpos($currentPath, $termPermalink . '/') === 0) {
                        $termPermalinkLength = strlen($termPermalink);
                        if ($termPermalinkLength > $matchedLength) {
                            $matchedTerm = $catCandidate;
                            $matchedLength = $termPermalinkLength;
                        }
                    }
                }
            }

            $termsForBreadcrumb = array();

            if ($matchedTerm instanceof \WP_Term) {
                $ancestorIds = array_reverse(get_ancestors($matchedTerm->term_id, 'rubrique_sit_primaire'));
                foreach ($ancestorIds as $ancestorId) {
                    $ancestor = get_term($ancestorId, 'rubrique_sit_primaire');
                    if ($ancestor instanceof \WP_Term) {
                        $termsForBreadcrumb[$ancestor->term_id] = $ancestor;
                    }
                }
                $termsForBreadcrumb[$matchedTerm->term_id] = $matchedTerm;
            } else if (!empty($cats)) {
                foreach ($cats as $catCandidate) {
                    $isCanonical = get_term_meta($catCandidate->term_id, 'weblist_permalinkIsCanonical', true);
                    if (in_array($isCanonical, array('1', 1, true, 'true'), true)) {
                        $termsForBreadcrumb[$catCandidate->term_id] = $catCandidate;
                    }
                }

                if (empty($termsForBreadcrumb)) {
                    $cats = $this->bridge_array_sort($cats, 'term_id', SORT_ASC);
                    foreach ($cats as $catCandidate) {
                        $termsForBreadcrumb[$catCandidate->term_id] = $catCandidate;
                    }
                }
            }

            $termsForBreadcrumb = array_values($termsForBreadcrumb);

            foreach ($termsForBreadcrumb as $cat) {
                $permalien = get_term_meta($cat->term_id, 'permalien', true);
                $exlude = get_term_meta($cat->term_id, 'exlude_breadcrumbs', true);
                if (isset($permalien) && !empty($permalien))
                    $permalien = $permalien;
                else
                    $permalien = substr(get_locale(), 0, 2) . '/' . $cat->slug;
                $tabBreadCrumbs[] = '<a href="' . $homeLink . '/' . $permalien . '/" class="term-bridge-' . $cat->term_id . '">' . $cat->name . '</a> <span class="separator">' . $delimiter . '</span> ';
            }
            //$retour .= 'b';
            //$retour .= $cats ;
        }
        //echo 'b';
        /*
        $finalBreadCrumbs = [];
        if(!empty($tabBreadCrumbsRubriquePrincipale)) {
            $finalBreadCrumbs = $tabBreadCrumbsRubriquePrincipale;
        } else {
            $finalBreadCrumbs = $tabBreadCrumbs;
        }
        */
        if ($showCurrent == 1) {
            $tabBreadCrumbs[] = ' ' . $before . get_the_title() . $after;
        }

        if(!empty($tabBreadCrumbs))
            $retour = '<div id="crumbs" aria-label="breadcrumbs">' . implode('', $tabBreadCrumbs)  . '</div>';

        return $retour;
    }

    /* Permet de trier un tableau */

    public function bridge_array_sort($array, $on, $order=SORT_ASC)
    {
        $new_array = array();
        $sortable_array = array();

        if (count($array) > 0) {
            foreach ($array as $k => $v) {
                if (is_array($v)) {
                    foreach ($v as $k2 => $v2) {
                        if ($k2 == $on) {
                            $sortable_array[$k] = $v2;
                        }
                    }
                } else {
                    $sortable_array[$k] = $v;
                }
            }

            switch ($order) {
                case SORT_ASC:
                    asort($sortable_array);
                    break;
                case SORT_DESC:
                    arsort($sortable_array);
                    break;
            }

            foreach ($sortable_array as $k => $v) {
                $new_array[$k] = $array[$k];
            }
        }

        return $new_array;
    }
    public function _n($singular, $plural, $number, $domain = 'default') {
        return _n($singular, $plural, $number, $domain);
    }
    public function _nmp($keyStr, $singular, $plural, $number, $domain = 'default') {
        // On ignore $keyStr qui ne concerne pas Wordpress
		return _n($singular, $plural, $number, $domain);
    }

    public function __( $text, $domain = 'default' ) {
        return __( $text, $domain );
    }
	public function __mp($keyStr, $text, $domain = 'default') {
		// On ignore $keyStr qui ne concerne pas Wordpress
        return __( $text, $domain );
    }

    public function esc_attr($text) {
        return esc_attr($text);
    }
    public function esc_html($text) {
        return esc_html($text);
    }
    public function esc_js($text) {
        return esc_js($text);
    }

    public function sendMail($to, $subject, $message, $headers = '', $attachments = array())
    {
        return wp_mail($to, $subject, $message, $headers, $attachments);
    }

    public function send404Error($text = '') {
        global $wp_query;
        if(!empty($text))
            echo $text;
        $wp_query->set_404();
        status_header(404);
        nocache_headers();
    }
}
