<?php

namespace Drupal\pylot_bridge\Services;

use Bridge\Weblibs\BridgeClientContentInterface;
use Bridge\Weblibs\BridgeClientParamsInterface;
use Bridge\Weblibs\BridgeCmsAbstractLayerInterface;
use Drupal\nuull;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\node\NodeInterface;
use Drupal\taxonomy\TermInterface;
use Symfony\Component\HttpFoundation\RequestStack;


class BridgeClientContentDrupal implements BridgeClientContentInterface
{

	private $bridgeCmsAbstractLayer;
	private $bridgeClientParams;
    /**
     * @var \Drupal\Core\Entity\EntityTypeManagerInterface
     */
    protected $entityTypeManager;

    /**
     * @var \Symfony\Component\HttpFoundation\RequestStack
     */
    protected $requestStack;

    /**
     * @var \Drupal\Core\Routing\RouteMatchInterface
     */
    protected $routeMatch;

	public function __construct(
        BridgeCmsAbstractLayerInterface $bridgeCmsAbstractLayer,
        BridgeClientParamsInterface $bridgeClientParams,
        EntityTypeManagerInterface $entityTypeManager,
        RequestStack $requestStack,
        RouteMatchInterface $routeMatch
    ) {
		$this->bridgeCmsAbstractLayer = $bridgeCmsAbstractLayer;
        $this->entityTypeManager = $entityTypeManager;
        $this->requestStack = $requestStack;
        $this->routeMatch = $routeMatch;
        $this->bridgeClientParams = $bridgeClientParams;
	}
    /**
     *  Retourne L'id d'une fiche Bridge pour un produit lui même
     * @param int|string $productCode
     * @param int|string $idFicheBridge
     * @return int|string id fiche issu de l'API Bridge ou '' si non trouvé
     */
    public function getIdficheBridge($productCode='', $idFicheBridge='') {
        // non testé sous Drupal
        throw new \Exception("Méthode getIdficheBridge non encore implémentée pour Drupal");
        $node = $this->routeMatch->getParameter('node');

        // Récupération depuis le contexte
        if($idFicheBridge === '') {
            $idFicheBridge = $this->requestStack->getCurrentRequest()->get('idFicheBridge', '');
        }

        if($productCode === '' && $node instanceof NodeInterface) {
            if ($node->hasField('field_bridge_productcode')) {
                $productCode = $node->get('field_bridge_productcode')->value;
            }
            // TODO : ne pas extraire la rubrique d'ici mais plutôt de l'URL actuelle (passage du numéro de fiche SIT et Weblist ID)
            if ($node->hasField('field_rubrique_sit_primaire')) {
                $terms = $node->get('field_rubrique_sit_primaire')->referencedEntities();

                // Normalement, le modèle de fiche est issu des règles de réécriture mises en place à l'import
                // Si jamais il n'y en a pas (appel depuis /fiche-sit/slug ou pas de fiche saisie dans la weblist
                if (empty($idFicheBridge) && !empty($terms)) {
                    foreach ($terms as $term) {
                        if ($term->hasField('field_weblist_ficheid') && !empty($term->get('field_weblist_ficheid')->value)) {
                            $idFicheBridge = $term->get('field_weblist_ficheid')->value;
                            break;
                        }
                    }
                }
            }
        }
        $nodeId = null;
        // ajout 29/06/22 : si pas de modèle de fiche passé, on essaie de retrouver le modèle associé à la première weblist attaché à la fiche
        if(empty($idFicheBridge)) {
            if(!empty($productCode)) {
                $nodeId = $this->getPostIdFromProductCode($productCode);
            } elseif($node instanceof NodeInterface) {
                $nodeId = $node->id();
            }

            if(isset($nodeId)) {
                $nodeStorage = $this->entityTypeManager->getStorage('node');
                $node = $nodeStorage->load($nodeId);
                if ($node && $node->hasField('field_rubrique_sit_primaire')) {
                    $terms = $node->get('field_rubrique_sit_primaire')->referencedEntities();
                    // Normalement, le modèle de fiche est issu des règles de réécriture mises en place à l'import
                    // Si jamais il n'y en a pas (appel depuis /fiche-sit/slug ou pas de fiche saisie dans la weblist
                    if (empty($idFicheBridge) && !empty($terms)) {
                        foreach ($terms as $term) {
                            if ($term->hasField('field_weblist_ficheid') && !empty($term->get('field_weblist_ficheid')->value)) {
                                $idFicheBridge = $term->get('field_weblist_ficheid')->value;
                                break;
                            }
                        }
                    }
                }
            }
        }
        return $idFicheBridge;
    }

	/**
	 * Retourne l'objet term du CMS courant associé à une liste Bridge
	 * @param int $webListId ID de weblist issu de l'appel à Bridge
	 * @param string $lang langue de récupération
	 * @return int|null Objet term du CMS
	 */
	public function getTermFromListId(int $webListId, string $lang = 'fr')
	{
        // Récupéré de Drupal
        $query = \Drupal::entityQuery('taxonomy_term');
        $query->accessCheck(TRUE);
        $query->condition('vid', 'rubrique_sit');
        $query->condition('field_sit_list_id', $webListId );
        $query->condition('langcode', $lang) ; // On récupère en fr
        $results = $query->execute();

        //$terms = \Drupal\taxonomy\Entity\Term::loadMultiple($query->execute());
        if(count($results) > 0 ) {
            // $term =  reset($terms);
            $temp = array_keys($results);
            if(count($temp) > 0)
                return $temp[0];
            else
                return null;
        } else {
            return null;
        }
	}

	/**
	 * Retourne l'objet term du CMS courant associé à une liste Bridge
	 * @param int $webListId ID de weblist issu de l'appel à Bridge
	 * @param string $lang langue de récupération
	 * @return object|null Objet term du CMS
	 */
	public function getTermFromId(int $id, string $lang = '')
	{
        // non utilisé sous Drupal
        throw new \Exception("Méthode getTermFromId non utilisée sous Drupal");
	}


    /**
     * Retourne le chemin URL d'une liste Bridge (pour créer des liens vers les fiches et vers les listes)
     * @param object $webList objet weblist issu de l'appel à Bridge
     * @return array|string|string[] Portion d'URL de la liste
     */
    public function getListPermalink(object $webList) {

        $deb = '';
        $lang = $this->bridgeCmsAbstractLayer->getCurrentLanguage();

        // Update 23/01/2023 : on supporte les listes sans id créées à la volée => lien automatique
        if(!is_object($webList) || !isset($webList->id) || empty($webList->id))
            return 'RUBRIQUE_PRINCIPALE';

        $termId = $this->getTermFromListId($webList->id) ; //  self::getTermIdByWeblistId($webList->id);
        if(empty($termId))
            return '###ERR-WEBLIST-TERM###';

        $path = '/taxonomy/term/' . $termId ;
        $alias_manager = \Drupal::entityTypeManager()->getStorage('path_alias');
        // On recherche s'il existe un alias pour ce node
        $aliasNode = $alias_manager->loadByProperties([
            'path'     => $path,
            'langcode' => $lang
        ]);
        if(empty($aliasNode)) {
            return $this->bridgeClientParams->getLanguagePrefix() . $path; // Route par défaut de Drupal ; '/taxonomy/term/' . $term->get('tid')->value;
        } else {

            foreach($aliasNode as $alias_object) {
                $alias = $alias_object->getAlias();
                // $type = gettype($alias_object->alias);
                $type = gettype($alias);
                if($type == 'string')
                    return $this->bridgeClientParams->getLanguagePrefix() . $alias ;
                else
                    return '#ErrPermalink#TermTypeIsNotString#' . $type ;
            }
        }
	    $term = null;

	    // Update 23/01/2023 : on supporte les listes sans id créées à la volée => lien automatique
	    if(!isset($webList->id) || empty($webList->id) || $webList->permalinkUrl == 'RUBRIQUE_PRINCIPALE') {
		    return 'RUBRIQUE_PRINCIPALE';
	    }

		if(isset($webList->id) && !empty($webList->id)) {
		    $term = $this->getTermFromListId($webList->id);
	    }
	    if($term instanceof TermInterface) {
		    $permalink = $this->bridgeCmsAbstractLayer->getPermalinkFromTermId($term->id());
		    if($this->requestStack->getCurrentRequest()->query->has('EETR')) {
			    echo "PERMALINK : $permalink";
		    }
		    return $permalink;
	    } else {
		    $termid = 'pas de terme => veuillez lancer un import';
		    if($this->requestStack->getCurrentRequest()->query->has('EETR')) {
			    echo "PAS DE TERME !!!";
		    }
		    if($term instanceof TermInterface)
			    $termid = $term->id();
		    return "#err" . $webList->id . "#" . $termid . "#";
	    }

    }

	public function getAllTerms($langfilter = '')
	{
        if($langfilter != '') {
            $lang = $langfilter;
        } else {
            $lang = 'fr';
        }
        // Récupéré de Drupal
        $tids = \Drupal::entityQuery('taxonomy_term')
            ->accessCheck(TRUE)
            ->condition('vid', 'rubrique_sit')
            ->condition('langcode', $lang)
            ->execute();

        if(!empty($tids)) {
            $controller = \Drupal::entityTypeManager()->getStorage('taxonomy_term');
            $entities = $controller->loadMultiple($tids);
            return $entities;
        } else {
            return array();
        }
	}

	public function getWebListIdFromTermObject($term)
	{
        // non testé sous Drupal
        if ($term instanceof TermInterface && $term->hasField('field_weblist_id')) {
		    return $term->get('field_weblist_id')->value;
        }
        return null;
	}

	public function deleteTermObject(object $term)
	{
        // non testé sous Drupal
        if ($term instanceof TermInterface) {
		    return $term->delete();
        }
        return null;
	}

	/**
	 * Créé un nouveau terme dans le CMS
	 * @param   string  $termName
	 * @param   string  $termSlug
	 * @param   string|null  $termDescription
	 * @param   object  $webList
	 * @param   string  $lang
	 *
	 * @return mixed objet term du CMS
	 *
	 * @since version 1.0.0
	 */
	public function createTerm(string $termName, string $termSlug, $termDescription, object $webList, $permalink = '', string $lang = 'fr')
	{
        // non testé sous Drupal
        $term_values = [
            'name' => $termName,
            'description' => [
                'value' => $termDescription,
                'format' => 'full_html',
            ],
            'vid' => 'rubrique_sit_primaire',
            'langcode' => $lang,
            'path' => ['alias' => '/' . $termSlug],
        ];

		$term = $this->entityTypeManager->getStorage('taxonomy_term')->create($term_values);
        $term->save();

		if (!empty($permalink)) {
			if (substr($permalink, -1, 1) == '/')
				$permalink = substr($permalink, 0, -1);
			if (substr($permalink, 0, 1) == '/')
				$permalink = substr($permalink, 1);
			$term->set('field_permalien', $permalink);
		}

		// On mémorise l'id de weblist dans le meta du terme
		$term->set('field_weblist_id', $webList->id);
		$term->set('field_weblist_ficheid', $webList->ficheId);
		$term->set('field_weblist_destinationweblis', $webList->destinationWebListId);
		$term->set('field_weblist_donotcreaterewri', $webList->doNotCreateRewriteRuleForList);
		$term->set('field_weblist_permalinkiscanon', $webList->permalinkIsCanonical);
        $term->save();

		return $term;
	}

	public function updateTermFromWebList(object $term, string $termName, string $termSlug, $termDescription, object $webList, $permalink = '', string $lang = 'fr') {
        // non testé sous Drupal
        if (!$term instanceof TermInterface) {
            return;
        }

        if ($term->language()->getId() !== $lang && $term->isTranslatable()) {
            if (!$term->hasTranslation($lang)) {
                $term = $term->addTranslation($lang, $term->toArray());
            } else {
                $term = $term->getTranslation($lang);
            }
        }

        $term->setName($termName);
        $term->setDescription($termDescription);
        $term->set('path', ['alias' => '/' . $termSlug]);


		if (!empty($permalink)) {
			if (substr($permalink, -1, 1) == '/')
				$permalink = substr($permalink, 0, -1);
			if (substr($permalink, 0, 1) == '/')
				$permalink = substr($permalink, 1);
			$term->set('field_permalien', $permalink);
		} else {
			$term->set('field_permalien', null);
		}

		// On mémorise l'id de weblist dans le meta du terme
		$term->set('field_weblist_id', $webList->id);
		$term->set('field_weblist_ficheid', $webList->ficheId);
		$term->set('field_weblist_destinationweblis', $webList->destinationWebListId);
		$term->set('field_weblist_donotcreaterewri', $webList->doNotCreateRewriteRuleForList);
		$term->set('field_weblist_permalinkiscanon', $webList->permalinkIsCanonical);
        $term->save();
	}

    /**
     * Permet de récupérer l'url d'une fiche d'après son productCode
     * @param int|string $productCode champ productCode de Bridge
     * @return string URL de la fiche
     */
    public function getPostLinkFromProductCode($productCode) {
        // non testé sous Drupal
        $nid = $this->getPostIdFromProductCode($productCode);
        if ($nid) {
            return $this->bridgeCmsAbstractLayer->getPostPermalink($nid);
        }
        return "#ErrorPermaLink#";
    }

    /**
     * Permet de récupérer un Id de post à partir d'un numéro de produit SIT
     * @param string|int $productCode champ productCode de Bridge
     * @return string URL de la fiche
     */
    public function getPostIdFromProductCode($productCode) {
        $post = $this->findPostByProductCode($productCode);
        if($post === false) {
            return false;
        }
        return $post->id();
    }

    /**
     * Renvoit un objet Fiche SIT du CMS à partir d'un code produit
     * @param $productCode Code produit
     * @param string $lang Langue
     * @return false|mixed Objet Fiche SIT CMS
     * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
     * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
     */
    public function findPostByProductCode($productCode, $lang = 'fr') {
        $node_manager = \Drupal::entityTypeManager()->getStorage('node');
        $nodes = $node_manager->loadByProperties([
            'field_code_sit'     => $productCode,
            'langcode' => $lang
        ]);
        if(is_array($nodes) && count($nodes) > 0) {
            return reset($nodes);
        } else {
            return false;
        }
    }

    /**
     * Permet de récupérer l'url d'une fiche d'après son productCode
     * @param $productCode champ productCode de Bridge
     * @return string URL de la fiche
     */
    public function getPostPermalinkFromProductCode($productCode) {

        $lang = $this->bridgeCmsAbstractLayer->getCurrentLanguage();
        $permalink = '';
        $post = $this->findPostByProductCode($productCode, 'fr');
        if(!empty($post)) {
            $path = '/node/'.$post->id() ;
            $alias_manager = $permalink = \Drupal::service('path_alias.manager') ;
            $permalink = \Drupal::service('path_alias.manager')->getAliasByPath('/node/'.$post->id());

            // On ajoute le prefixe de langue si nécessaire
            if(empty($permalink))
                $permalink = $this->bridgeClientParams->getLanguagePrefix() . $path;
            else
                $permalink = $this->bridgeClientParams->getLanguagePrefix() . $permalink;

            // Correction double /
            if(mb_substr($permalink, 0, 2) == '//') {
                $permalink = substr($permalink, 1);
            }
        } else {
            $permalink = "#ErrorPermalink#PostNotFoundForProduct#" . $productCode;
        }
        if(!empty($permalink))
            return $permalink;

        return "#ErrorPermalink#";
    }

    /**
     * Retourne le terme de taxonomie associé à une liste bridge - Drupal
     * @param $id id de terme
     * @param string $lang langue dans laquelle on veut récupérer le terme
     * @return \Drupal\Core\Entity\EntityBase[]|\Drupal\Core\Entity\EntityInterface[]|Term[] tableau associatif $tid => $term
     */
    public function getWebPageIdFromTermId($id, $lang = 'fr') {
        $pageId = null;
        $term = \Drupal\taxonomy\Entity\Term::load($id);
        if(!empty($term)) {
            $pageId = $term->get('field_sit_fiche_id')->value;
        }
        return $pageId;
    }
    /**
     * Wordpress : Permet de récupérer le lien de la rubrique principale d'une fiche
     * @param int $productCode code SIT de la fiche
     * @param string $productSlug slug de la fiche (pour éviter une récupération en BDD si on l'a sous la main)
     * @return string Lien vers la fiche dans sa rubrique principale ou lien par défaut sinon
     */
    public function getMainTermLinkFromProductCode($productCode, $productSlug = '') {
        // non testé sous Drupal
        $link = '';
        $postId = $this->getPostIdFromProductCode($productCode);
        if(!empty($postId)) {
            $node = $this->entityTypeManager->getStorage('node')->load($postId);
            if ($node && $node->hasField('field_rubrique_principale') && !$node->get('field_rubrique_principale')->isEmpty()) {
                $mainTermId = $node->get('field_rubrique_principale')->target_id;
                if(!empty($mainTermId)) {
                    if(empty($productSlug)) {
                        $productSlug = $node->toUrl()->toString(); // This gives full URL, might need adjustment
                        $productSlug = basename($productSlug);
                    }
                    $tempPermalink = $this->bridgeCmsAbstractLayer->getPermalinkFromTermId($mainTermId);
                    $link = '/' . $tempPermalink . '/' . $productSlug;
                }
            }
        }
        // Voiture balais : s'il n'existe pas de liste cochée 'Rubrique principale' contenant cette fiche
        // On renvoit le permalien par défaut de la fiche
        if(empty($link)) {
            $link = $this->getPostLinkFromProductCode($productCode);
        }

        return $link;
    }

    /**
     * Permet de récupérer le lien vers une fiche)
     * @param int $productCode code SIT de la fiche
     * @param string $productSlug slug de la fiche (pour éviter une récupération en BDD si on l'a sous la main)
     * @param string $permalink Permalink de la liste courante (Wordpress réécritures d'URL)
     * @return string Lien vers la fiche dans sa rubrique principale ou lien par défaut sinon
     */
    public function getProductStandardLink($productCode, $productSlug = '', $permalink = '') {
        if(!empty($productSlug) && !empty($permalink)) {
            // Cas normal d'utilisation pour Drupal
            return '/'.$permalink . '/' . $productSlug;
        } else {
            // Sinon lien par défaut
            return $this->getPostLinkFromProductCode($productCode);
        }
    }

    /**
     * Retourne l'ID de la fiche Bridge associée à un post
     *
     * @param int $postId
     * @return nuull|int id de fiche Bridge reliée au post (préférence pour la rubrique principale)
     */
    public function getFicheIdFromPostId(int $postId) {
        // non testé sous Drupal
        $node = $this->entityTypeManager->getStorage('node')->load($postId);
        if (!$node || !$node->hasField('field_rubrique_sit_primaire')) {
            return null;
        }
        $terms = $node->get('field_rubrique_sit_primaire')->referencedEntities();
        $idFicheBridge = null;
        // Normalement, le modèle de fiche est issu des règles de réécriture mises en place à l'import
        // Si jamais il n'y en a pas (appel depuis /fiche-sit/slug ou pas de fiche saisie dans la weblist
        if (!empty($terms)) {
            $debugTrace[] = 'Zone 3 - terms : ' . var_export(array_map(function($t){return $t->id();}, $terms), true);
            // Patch 14/06/2023 : on prend la fiche de la rubrique principale en priorité
            $idFicheBridgeParDepit = '';
            $hasRubriquePrincipale = false;
            $nomsRubriques = [];
            $nomsRubriquesPrincipales = [];
            foreach ($terms as $term) {
                $idFiche = $term->hasField('field_weblist_ficheid') ? $term->get('field_weblist_ficheid')->value : null;
                $rubriquePrincipale = $term->hasField('field_weblist_permalinkiscanon') ? $term->get('field_weblist_permalinkiscanon')->value : false;
                if($rubriquePrincipale == true && !empty($idFiche)) {
                    if($hasRubriquePrincipale) {
                        $debugTrace[] = 'Zone 3,3 - Attention, il y a plusieurs rubriques principales - idFiche : ' . $idFiche;
                    }
                    $nomsRubriquesPrincipales[] = $term->id() . '-' . $term->getName();
                    $hasRubriquePrincipale = true;
                    $idFicheBridge = $idFiche;
                    $debugTrace[] = 'Zone 3,5 - on prend la fiche de la rubrique principale - idFiche : ' . $idFiche;
                } else if (!empty($idFiche) ) {
                    $idFicheBridgeParDepit = $idFiche;
                    $nomsRubriques[] = $term->id() . '-' . $term->getName();
                }
            }

            if($hasRubriquePrincipale) {
                $debugTrace[] = 'Zone 3,9 - rubriques principales : ' . implode(', ', $nomsRubriquesPrincipales);
                $debugTrace[] = 'Zone 3,91 - autres rubriques (non utilisées) : ' . implode(', ', $nomsRubriques);
            }
            // S'il n'y a pas de rubrique principale, on se rabat par dépit sur la fiche de la dernière webList
            if($hasRubriquePrincipale === false && !empty($idFicheBridgeParDepit)) {
                $debugTrace[] = 'Zone 4 - on prend une fiche en dépit de rubrique principale - idFiche : ' . $idFiche;
                $debugTrace[] = 'Zone 4,1 - rubriques par dépit : ' . implode(', ', $nomsRubriques);
                $idFicheBridge = $idFicheBridgeParDepit;
            }
        }
        return $idFicheBridge;
    }

	public function deletePost($postId) {
        // non testé sous Drupal
        $node = $this->entityTypeManager->getStorage('node')->load($postId);
        if ($node) {
            // Attachments are fields in Drupal, they will be deleted with the node.
            // If they are separate entities, custom logic is needed here.
            $node->delete();
            return true;
        }
		return false;
	}

    /**
     * Alias de get_query_var pour wordpress
     * @param string $var
     * @return mixed
     */
    public function getQueryVar(string $var) {
        // non testé sous Drupal
        return $this->requestStack->getCurrentRequest()->get($var);
    }

    /**
     * Retourne le code produit d'un post ous la forme de chaine
     * @param int $postId
     * @return string code produit
     */
    public function getProductCodeFromPostId(int $postId) {
        // non testé sous Drupal
        $node = $this->entityTypeManager->getStorage('node')->load($postId);
        if ($node && $node->hasField('field_bridge_productcode')) {
            return $node->get('field_bridge_productcode')->value;
        }
        return '';
    }

    /**
     * Récupère le post courant (Wordpress)
     * @return null|int id du post courant
     */
    public function getCurrentPostId() {
        // non testé sous Drupal
        $node = $this->routeMatch->getParameter('node');
        if ($node instanceof NodeInterface) {
            return $node->id();
        }
        return null;
    }

	public function getPostObjectFrom($field, $value, $lang) {

	}

	public function createPostFromBridgeProduct($product, $lang) {

	}

	public function updatePostFromBridgeProduct($postId, $product, $lang) {

	}
}
