<?php

namespace Bridge\Weblibs;

/**
 * Couche d'abstraction au CMS
 *
 *
 * @package    Plugin_SIT
 * @subpackage Plugin_SIT/includes
 * @author     DN Consultants <info@dnconsultants.fr>
 */

interface BridgeCmsAbstractLayerInterface {

    /**
     * Retourne vrai si le script a été enregistré / chargé
     * @param $handle : clé texte du script
     * @param $list : statut à vérifier : enqueued, registered
     * @return mixed
     */
    public function scriptIs($handle, $list );
    /**
     * Ajouter un script JS à la page courante
     * Si src est absent, le handle du script doit faire référence à un script préalablement déclaré avec registerScript
     * @param $handle : id texte du script
     * @param string $src : url du script
     * @param array $deps : tableau de dépendances (liste d'id texte)
     * @param false $ver : version
     * @param false $in_footer : charger dans le footer
     * @return mixed
     */
    public function registerScript($handle, $src = "", $deps = array(), $ver = false, $in_footer = false );

    /**
     * Ajouter un script JS à la page courante
     * Si src est absent, le handle du script doit faire référence à un script préalablement déclaré avec registerScript
     * @param $handle : id texte du script
     * @param string $src : url du script
     * @param array $deps : tableau de dépendances (liste d'id texte)
     * @param false $ver : version
     * @param false $in_footer : charger dans le footer
     * @return mixed
     */
    public function enqueueScript($handle, $src = "", $deps = array(), $ver = false, $in_footer = false );

    /**
     * Référencer une feuille de style CSS : associer un id texte à une URL
     * Si src est absent, le handle du css doit faire référence à un fichier préalablement déclaré avec registerStyle
     * @param $handle : id texte du css
     * @param string $src : url du css
     * @param array $deps : tableau de dépendances (liste d'id texte)
     * @param false $ver : version
     * @param string $media
     * @return mixed
     */
    public function registerStyle($handle, $src = "", $deps = array(), $ver = false, $media = 'all' );

    /**
     * Ajouter une feuille de style CSS à la page courante
     * Si src est absent, le handle du css doit faire référence à un fichier préalablement déclaré avec registerStyle
     * @param $handle : id texte du css
     * @param string $src : url du css
     * @param array $deps : tableau de dépendances (liste d'id texte)
     * @param false $ver : version
     * @param string $media
     * @return mixed
     */
    public function enqueueStyle($handle, $src = "", $deps = array(), $ver = false, $media = 'all' );

    /**
     * @return Retourne le code de la langue courante (2 lettres en minuscules / fr par défaut)
     */
    public function getCurrentLanguage();

    /**
     * @return Retourne le code de la langue courante sous la forme fr-FR ou en-GB (langue - pays)
     */
    public function getCurrentLanguageLocale();

    /**
     * Retourne le code de langue d'un post donné (wordpress)
     * @param $postId numéro de post
     * @return string
     */
    public function getPostLanguage($postId);

    /**
     * Définit le code de langue d'un post donné (surtout pour WordPress qui ne gère pas nativement le multilingue
     * @param $postId : id du post
     * @return string code de langue sur 2 lettres
     */
    public function setPostLanguage($postId, $lang);

    /**
     * Sauvegarde les traductions d'un produit (WordPress)
     * @param $translationsArray tableau associatif $lang => $postId ('fr' => 4)
     */
    public function savePostTranslations($translationsArray);

    /**
     * Définit le code de langue d'un post donné (WordPress)
     * @param $termId : id du terme
     * @return string
     */
    public function setTermLanguage($termId, $lang);

    /**
     * Retourne la langue d'un terme (WordPress)
     * @param $termId : id du terme
     * @return string code de langue
     */
    public function getTermLanguage($termId);

    /**
     * Retourne la langue d'un terme de taxonomie (WordPress)
     * ND 12/09/2022 : après recherche dans le code cette fonction n'est jamais utilisée
     * @param $termId : id du terme
     * @return array taleau associatif lang => id de terme
     */
    public function getTermTranslations($termId);

    /**
     * Sauvegarde les traductions d'un terme (Wordpress)
     * @param $translationsArray tableau associatif $lang => $termId ('fr' => 4)
     */
    public function saveTermTranslations($translationsArray);

    /**
     * Retourne la liste des langues actives
     * @return string[] tableau de codes de langues
     */
    public function getLanguagesList();

    /**
     * Retourne le permalien associé à une weblist
     * @param $term_id : id de term lié à la lweblist
     * @param string $lang : langue en cours
     * @return string : Début de l'URL pour la réécriture des liens vers les fiches de la liste'
     */
    public function getPermalinkFromTermId($term_id, $lang = '');


    public function get_site_url() ;

    public function getPostPermalink($postId);

    /**
     * Retourne le header du theme front-end du site pour affichage avant le code de la fichede détail (WordPress)
     * @return mixed
     */
    public function getHeaderBridge($arg = null) ;

    /**
     * Retourne le footer du theme front-end du site pour affichage après le code de la fichede détail (WordPress)
     * @return mixed
     */
    public function getFooterBridge() ;


    /**
     * Renvoie une reponse http Json d'erreur
     * @param mixed|null $data données à renvoyer en Json
     * @param int|null $status_code code HTTP à renvoyer
     * @param int $options options d'après les specifications de la fonction wp_send_json_error
     */
    public function sendJsonError( $data = null, int $status_code = null, int $options = 0);

    /**
     * Renvoie une reponse http Json de succes
     * @param mixed|null $data données à renvoyer en Json
     * @param int|null $status_code code HTTP à renvoyer
     * @param int $options options d'après les specifications de la fonction wp_send_json_success
     */
    public function sendJsonSuccess( $data = null, int $status_code = null, int $options = 0) ;

    /**
     * Renvoie une reponse http Json de succes
     * @param mixed|null $data données à renvoyer en Json
     * @param int|null $status_code code HTTP à renvoyer
     * @param int $options options d'après les specifications de la fonction wp_send_json_success
     */
    public function sendJson( $data = null, int $status_code = null, int $options = 0);

    /**
     * Permet de déclarer une fonction de callback pour insérer du contenu dans les vues Bridge sans avoir à les overrider
     * éuivalent à wordpress add_action
     * @param $hookName : nom du hook
     * @param $callBack : fonction de callback
     * @param int $priority : priorité
     * @param int $acceptedArgs : nombre d'arguments acceptés
     */
    public function addDisplayHook($hookName, $callBack, $priority = 10, $acceptedArgs = 1);

    /**
     * Permet d'insérer un hook d'affichage dans les templates
     * @param string $hookName
     * @param mixed $args
     */
    public function doDisplayHook($hookName, ...$arg) ;

     /**
     * Permet de déclarer une fonction de callback pour retrvailler du contenu dans les vues Bridge sans avoir à les overrider
     * éuivalent à wordpress apply_filter
     * @param $hookName : nom du hook
     * @param $callBack : fonction de callback
     * @param int $priority : priorité
     * @param int $acceptedArgs : nombre d'arguments acceptés
     */
    public function applyFilters($hookName, $callBack, $priority = 10, $acceptedArgs = 1) ;

    /**
     * Permet de déclarer une fonction de callback pour retrvailler du contenu dans les vues Bridge sans avoir à les overrider
     * éuivalent à wordpress apply_filter
     * @param $hookName : nom du hook
     * @param $callBack : fonction de callback
     * @param int $priority : priorité
     * @param int $acceptedArgs : nombre d'arguments acceptés
     */
    public function addFilter($hookName, $callBack, $priority = 10, $acceptedArgs = 1);

    /**
     * @return false|mixed|string Chemin de base relatif à la réécriture des URL des fiches (hors permalien personnalisé)
     */
    public function getFicheRewritePattern();

    /**
     * Fonction pour WordPress : indique si on se trouve actuellement sur une page de taxonomie SIT
     * @return bool True si on est sur une page de taxonomie SIT
     *
     */
    public function IsFicheSitTaxonomy();

    /**
     * @return bool true si on est sur une page de fiche SIT
     */
    public function IsFicheSit() ;

    /**
     * fonction appelée depuis un modèle de fiche
     * Retourne le code HTML du fil d'Ariane affiché dans les fiches
     * @param string $delimiter délimiteur d'entrées
     * @param $home texte pour l'entrée 'Accueil'
     * @param int $showCurrent mettre 1 pour afficher l'entre courante
     * @param string $before texte à afficher avant l'élément courant
     * @param string $after texte à afficher après l'élément courant
     *
     * @return string code HTML du fil d'Ariane affiché dans les fiches
     */
    public function BridgeBreadcrumbs($delimiter = '&raquo;',$home = '', $showCurrent = 1,$before = '<span class="current">',$after = '</span>',$showOnHome = true);

    /* Permet de trier un tableau */

    public function bridge_array_sort($array, $on, $order=SORT_ASC);

	public function _n($singular, $plural, $number, $domain = 'default') ;

	public function _nmp($keyStr, $singular, $plural, $number, $domain = 'default') ;

	public function __( $text, $domain = 'default' );

	public function __mp($keyStr, $text, $domain = 'default') ;

	public function esc_attr($text) ;
	public function esc_html($text) ;

	public function sendMail($to, $subject, $message, $headers = '', $attachments = array()) ;

	public function send404Error($text = '') ;
}
