<?php

namespace Bridge\Weblibs;

use Bridge\Weblibs\BridgeApiService;
use Bridge\Weblibs\BridgeClientParamsInterface;
use Bridge\Weblibs\BridgeClientRendererInterface;
use Bridge\Weblibs\BridgeCmsAbstractLayerInterface;
use Bridge\Weblibs\BridgeClientContentInterface;
use Bridge\Weblibs\BridgeFrontEndAjaxInterface;
use Bridge\Weblibs\BridgeRequestServiceInterface;
use Bridge\Weblibs\BridgeDataGetter;


class BridgeShortCodeParser
{

	// Regex101 reference: https://regex101.com/r/pJ7lO1
	const SHORTOCODE_REGEXP = "/(?P<shortcode>(?:(?:\\s?\\[))(?P<name>[\\w\\-]{3,})(?:\\s(?P<attrs>[\\w\\d,\\s=\\\"\\'\\-\\+\\#\\%\\!\\~\\`\\&\\.\\s\\:\\/\\?\\|]+))?(?:\\])(?:(?P<content>[\\w\\d\\,\\!\\@\\#\\$\\%\\^\\&\\*\\(\\\\)\\s\\=\\\"\\'\\-\\+\\&\\.\\s\\:\\/\\?\\|\\<\\>]+)(?:\\[\\/[\\w\\-\\_]+\\]))?)/u";

	// Regex101 reference: https://regex101.com/r/sZ7wP0
	const ATTRIBUTE_REGEXP = "/(?<name>\\S+)=[\"']?(?P<value>(?:.(?![\"']?\\s+(?:\\S+)=|[>\"']))+.)[\"']?/u";

	private BridgeClientParamsInterface $clientParams;

	private BridgeRequestServiceInterface $requestService;
	private BridgeApiService $bridgeApiService;

	private BridgeClientContentInterface $bridgeClientContent;

	private BridgeClientRendererInterface $bridgeClientRenderer;

	private BridgeCmsAbstractLayerInterface $bridgeCmsAbstractLayer;

	private BridgeDataGetter $bridgeDataGetter;

	private $listRequestVars = ['id', 'product_codes', 'max', 'first', 'itemTemplate', 'item_template', 'change', 'minimal_select', 'filter_city', 'filter_criterions', 'scoring', 'filter_lat', 'filter_lon', 'filter_rayon', 'selection_id', 'excluded_codes', 'sort_one', 'sort_two', 'sort_three', 'duplicate_by', 'filter_id'];
	public function __construct(BridgeClientParamsInterface $clientParams, BridgeClientContentInterface $bridgeClientContent, BridgeCmsAbstractLayerInterface $bridgeCmsAbstractLayer, BridgeRequestServiceInterface $requestService, BridgeApiService $bridgeApiService, BridgeClientRendererInterface $bridgeClientRenderer, BridgeDataGetter $bridgeDataGetter)
	{
		$this->bridgeCmsAbstractLayer = $bridgeCmsAbstractLayer;
		$this->clientParams = $clientParams;
		$this->bridgeClientContent = $bridgeClientContent;
		$this->requestService = $requestService;
		$this->bridgeApiService = $bridgeApiService;
		$this->bridgeClientRenderer = $bridgeClientRenderer;
		$this->bridgeDataGetter = $bridgeDataGetter;
	}

	public function parse_shortcodes($text)
	{
		$SHORTOCODE_REGEXP = "/(?P<shortcode>(?:(?:\\s?\\[))(?P<name>[\\w\\-]{3,})(?:\\s(?P<attrs>[\\w\\d,\\s=\\\"\\'\\-\\+\\#\\%\\!\\~\\`\\&\\.\\s\\:\\/\\?\\|]+))?(?:\\])(?:(?P<content>[\\w\\d\\,\\!\\@\\#\\$\\%\\^\\&\\*\\(\\\\)\\s\\=\\\"\\'\\-\\+\\&\\.\\s\\:\\/\\?\\|\\<\\>]+)(?:\\[\\/[\\w\\-\\_]+\\]))?)/u";

		preg_match_all($SHORTOCODE_REGEXP, $text, $matches, PREG_SET_ORDER);
		$shortcodes = array();
		foreach ($matches as $i => $value)
		{
			$shortcodes[$i]['shortcode'] = $value['shortcode'];
			$shortcodes[$i]['name']      = $value['name'];
			if (isset($value['attrs']))
			{
				$attrs                   = $this->parse_attrs($value['attrs']);
				$shortcodes[$i]['attrs'] = $attrs;
			}
			if (isset($value['content']))
			{
				$shortcodes[$i]['content'] = $value['content'];
			}
		}

		return $shortcodes;
	}

	private function parse_attrs($attrs)
	{
		$ATTRIBUTE_REGEXP = "/(?<name>\\S+)=[\"']?(?P<value>(?:.(?![\"']?\\s+(?:\\S+)=|[>\"']))+.)[\"']?/u";

		preg_match_all($ATTRIBUTE_REGEXP, $attrs, $matches, PREG_SET_ORDER);
		$attributes = array();
		foreach ($matches as $i => $value)
		{
			$key                  = $value['name'];
			$attributes[$i][$key] = $value['value'];
		}

		return $attributes;
	}


	/**
	 * BridgeShortCodeParser::process
	 *
	 * Traite tous les shortcodes Bridge du texte fourni en entrée
	 *
	 * @param   string  $text      : texte à filtrer contenant potentiellement les shortcodes
	 * @param           $langcode  : code de langue (2 lettres)
	 *
	 * @return object Objet ayant comme propriétés success (bool), text (string) : texte filtré, build (array of arrays)  : tableau de tableaux de données de rendu Drupal, shortcodeFicheMatched (bool) : true si un short de fiche a été trouvé et remplacé (nécessité de charger les librairies afférentes), , shortcodeListeMatched (bool) : true si un short de liste a été trouvé et remplacé (nécessité de charger les librairies afférentes),  errMsg : message en cas d'erreur
	 */
	public function process($text, $langcode)
	{
		$res = (object) array(
			'success'               => true,
			'text'                  => $text,
			'build'                 => array(),
			'shortcodeFicheMatched' => false,
			'shortcodeListeMatched' => false,
			'shortcodeMoteurMatched' => false,
			'shortcodeMapMatched' => false,
			'errMsg'                => ''
		);

		// On parse les shortcodes du texte original
		$shortcodes = $this->parse_shortcodes($text);

		// On traite les shortcodes de liste
		$dataLists                  = $this->processWeblists($res->text, $shortcodes, $langcode);
		$res->text                  = $dataLists->text;
		$res->success               = $res->success && $dataLists->success;
		$res->build                 = array_merge($res->build, $dataLists->build);
		$res->shortcodeListeMatched = $dataLists->shortcodeMatched;
		$res->errMsg                .= $dataLists->errMsg;

		// On traite les shortcodes de moteurs
		$dataMoteurs                  = $this->processMoteurs($res->text, $shortcodes, $langcode);
		$res->text                  = $dataMoteurs->text;
		$res->success               = $res->success && $dataMoteurs->success;
		$res->build                 = array_merge($res->build, $dataMoteurs->build);
		$res->shortcodeMoteurMatched = $dataMoteurs->shortcodeMatched;
		$res->errMsg                .= $dataMoteurs->errMsg;

		// On traite les shortcodes de maps
		$dataMaps                  = $this->processMaps($res->text, $shortcodes, $langcode);
		$res->text                  = $dataMaps->text;
		$res->success               = $res->success && $dataMaps->success;
		$res->build                 = array_merge($res->build, $dataMaps->build);
		$res->shortcodeMapMatched = $dataMaps->shortcodeMatched;
		$res->errMsg                .= $dataMaps->errMsg;

		// On traite les shortcodes de fiche
		$dataFiches                 = $this->processWebPages($res->text, $shortcodes, $langcode);
		$res->text                  = $dataFiches->text;
		$res->success               = $res->success && $dataFiches->success;
		$res->build                 = array_merge($res->build, $dataFiches->build);
		$res->shortcodeFicheMatched = $dataFiches->shortcodeMatched;
		$res->errMsg                .= $dataFiches->errMsg;

		return $res;
	}

	/**
	 * Traite le shortcode brliste
	 *
	 * @param   string  $text        : texte à filtrer contenant potentiellement le shortcode
	 * @param           $shortcodes  : tableau des shortcodes trouvés avec la fonction  BridgeShortcodeParser::parse_shortcodes
	 * @param           $langcode    : code de langue (2 lettres)
	 *
	 * @return object Objet ayant comme propriétés success (bool), text (string) : texte filtré, data (Object)  : données Bridge, shortcodeMatched (bool) : true si un short a bien été trouvé et remplacé (nécessité de charger les librairies afférentes),  errMsg : message en cas d'erreur
	 */
	public function processWeblists($text, $shortcodes, $langcode)
	{
		$res = (object) array(
			'success'          => true,
			'text'             => $text,
			'build'            => array(),
			'shortcodeMatched' => false,
			'errMsg'           => ''
		);

		if (is_array($shortcodes) && count($shortcodes) > 0)
		{
			foreach ($shortcodes as $shortcode)
			{
				if ($shortcode['name'] !== 'brliste')
					continue;
				$attributes = array();
				if (empty($shortcode['attrs']))
					continue;
				foreach ($shortcode['attrs'] as $attr)
				{
					foreach ($attr as $key => $value)
					{
						$attributes[$key] = str_replace('"', '', $value);
					}
				}
				// if(!isset($attributes['id']))
				//  continue;
				$attributes['lang']              = $langcode;
				$attributes['change']            = '1';
				$attributes['useRequestFilters'] = true;
				$res->shortcodeMatched           = true;
				try
				{
					// Appel à Bridge, récupération des données
					$data = $this->bridgeDataGetter->getListData($attributes);
					if (is_array($data) && $data['success'] === false)
					{
						$res->success = false;
						$res->errMsg  = 'Erreur : ' . $data['message'];
						$res->text    = str_replace($shortcode['shortcode'], $res->errMsg, $res->text);

						return $res;
					}
					// On prépare l'organisation des données en vue de l'affichage - avec conversion des tableaux associatifs en objets au passage pour simplifier
					$data =  $this->bridgeDataGetter->prepareListDataForRender($data);

					// On attache les librairies scripts, css ainsi que le tableau de données à passer aux templates d'affichage
					$build = $this->bridgeClientRenderer->prepareListViewRenderer($data, 'liste_sit_shortcode');
					// On retourne les données de rendu Drupal brutes Bridge ainsi que le résultat du parsing HTML
					// $res->build[] = $build;
					// On fait le rendu HTML
					$html      = $this->bridgeClientRenderer->renderListeTemplate($build);

					// renderTemplate('liste_sit_shortcode', $build);

					// $html      = \Drupal::service('renderer')->renderPlain($build);
					$res->text = str_replace($shortcode['shortcode'], $html, $res->text);

				}
				catch (\Exception $e)
				{
					$res->errMsg  = "Erreur 07 shortcode Bridge : " . $e->getMessage() ; // . ' - Trace : ' . $trace;
					$res->text    = str_replace($shortcode['shortcode'], $res->errMsg, $res->text);
					$res->success = false;
				}
			}
		}

		return $res;
	}

	/**
	 * Traite le shortcode brfiche
	 *
	 * @param   string  $text        : texte à filtrer contenant potentiellement le shortcode
	 * @param           $shortcodes  : tableau des shortcodes trouvés avec la fonction  BridgeShortcodeParser::parse_shortcodes
	 * @param           $langcode    : code de langue (2 lettres)
	 *
	 * @return object Objet ayant comme propriétés success (bool), text (string) : texte filtré, build (array of arrays)  : tableau de tableaux de données de rendu Drupal, shortcodeMatched (bool) : true si un short a bien été trouvé et remplacé (nécessité de charger les librairies afférentes),  errMsg : message en cas d'erreur
	 */
	public function processWebPages($text, $shortcodes, $langcode)
	{
		$res = (object) array(
			'success'          => true,
			'text'             => $text,
			'data'             => null,
			'build'            => array(),
			'shortcodeMatched' => false,
			'errMsg'           => ''
		);

		$new_text = $text;
		if (is_array($shortcodes) && count($shortcodes) > 0)
		{
			foreach ($shortcodes as $shortcode)
			{
				if ($shortcode['name'] !== 'brfiche')
				{
					continue;
				}
				$attributes = array();
				if (empty($shortcode['attrs']))
				{
					continue;
				}

				foreach ($shortcode['attrs'] as $attr)
				{
					foreach ($attr as $key => $value)
					{
						$attributes[$key] = str_replace('"', '', $value);
					}
				}
				if (!isset($attributes['id']) || !isset($attributes['product_code']))
				{
					continue;
				}

				$res->shortcodeMatched = true;

				$attributes['lang'] = $langcode;
				try
				{
					// Récupération des données de la fiche SIT
					$data = $this->bridgeDataGetter->getDataFiche($attributes['product_code'], $attributes['id']);
					if ($data === false)
					{
						$res->errMsg  = "Erreur 08 shortcode Bridge : aucune donnée retournée";
						$res->text    = str_replace($shortcode['shortcode'], $res->errMsg, $res->text);
						$res->success = false;
					}
					else
					{
						// On prépare les donnée spour utilisation dans les templates
						$ficheData = $this->bridgeDataGetter->prepareFicheDataForRender($data);
						// TODO : harmoniser les fonctions de preparartion de rendu entre WP/Joomla et Drupal
						// On prepare la vue
						// $build = $this->bridgeClientRenderer->prepareFicheViewRenderer($data, 'fiche_sit_ajax');
						// On retourne les données de rendu Drupal brutes Bridge ainsi que le résultat du parsing HTML
						// $res->build[] = $build;
						// On fait le rendu HTML
						$html      = $this->bridgeClientRenderer->renderTemplate('fiche_sit_ajax', $ficheData);
						// $html      = \Drupal::service('renderer')->renderPlain($build);
						$res->text = str_replace($shortcode['shortcode'], $html, $res->text);
					}
				}
				catch (\Exception $e)
				{
					$res->errMsg  = "Erreur 071 shortcode Bridge : " . $e->getMessage();
					$res->text    = str_replace($shortcode['shortcode'], $res->errMsg, $res->text);
					$res->success = false;
				}
			}
		}

		return $res;
	}



	/**
	 * Traite le shortcode brfiche
	 *
	 * @param   string  $text        : texte à filtrer contenant potentiellement le shortcode
	 * @param           $shortcodes  : tableau des shortcodes trouvés avec la fonction  BridgeShortcodeParser::parse_shortcodes
	 * @param           $langcode    : code de langue (2 lettres)
	 *
	 * @return object Objet ayant comme propriétés success (bool), text (string) : texte filtré, build (array of arrays)  : tableau de tableaux de données de rendu Drupal, shortcodeMatched (bool) : true si un short a bien été trouvé et remplacé (nécessité de charger les librairies afférentes),  errMsg : message en cas d'erreur
	 */
	public function processMoteurs($text, $shortcodes, $langcode)
	{
		$res = (object) array(
			'success'          => true,
			'text'             => $text,
			'data'             => null,
			'build'            => array(),
			'shortcodeMatched' => false,
			'errMsg'           => ''
		);

		$new_text = $text;
		if (is_array($shortcodes) && count($shortcodes) > 0)
		{
			foreach ($shortcodes as $shortcode)
			{
				if ($shortcode['name'] !== 'brmoteur')
				{
					continue;
				}
				$attributes = array();
				if (empty($shortcode['attrs']))
				{
					continue;
				}

				foreach ($shortcode['attrs'] as $attr)
				{
					foreach ($attr as $key => $value)
					{
						$attributes[$key] = str_replace('"', '', $value);
					}
				}
				if (!isset($attributes['idmoteur']) || !isset($attributes['idliste']))
				{
					continue;
				}
				$attributes['lang'] = $langcode;
				try
				{
					$res->shortcodeMatched = true;
					$preData = $this->bridgeDataGetter->getMoteurData($attributes);
					if(! isset($preData['success']) || $preData['success'] !== true) {
						$res->errMsg  = "Erreur 8M shortcode Bridge moteur : " . $preData['message'];
						$res->text    = str_replace($shortcode['shortcode'], $res->errMsg, $res->text);
						$res->success = false;
					} else {
						$template = "moteur1";
						if(isset($attributes['template']) && !empty($attributes['template'])) {
							$template = $attributes['template'];
						}
						$moteurData = $this->bridgeDataGetter->prepareMoteurDataForRender($preData);
						$html = $this->bridgeClientRenderer->renderMoteurTemplate($moteurData, $template);
						$res->text = str_replace($shortcode['shortcode'], $html, $res->text);
					}
				}
				catch (\Exception $e)
				{
					$res->errMsg  = "Erreur 081 shortcode Bridge : " . $e->getMessage();
					$res->text    = str_replace($shortcode['shortcode'], $res->errMsg, $res->text);
					$res->success = false;
				}
			}
		}
		return $res;
	}

	/**
	 * Traite le shortcode brfiche
	 *
	 * @param   string  $text        : texte à filtrer contenant potentiellement le shortcode
	 * @param           $shortcodes  : tableau des shortcodes trouvés avec la fonction  BridgeShortcodeParser::parse_shortcodes
	 * @param           $langcode    : code de langue (2 lettres)
	 *
	 * @return object Objet ayant comme propriétés success (bool), text (string) : texte filtré, build (array of arrays)  : tableau de tableaux de données de rendu Drupal, shortcodeMatched (bool) : true si un short a bien été trouvé et remplacé (nécessité de charger les librairies afférentes),  errMsg : message en cas d'erreur
	 */
	public function processMaps($text, $shortcodes, $langcode)
	{
		$res = (object) array(
			'success'          => true,
			'text'             => $text,
			'data'             => null,
			'build'            => array(),
			'shortcodeMatched' => false,
			'errMsg'           => ''
		);

		$new_text = $text;
		if (is_array($shortcodes) && count($shortcodes) > 0)
		{
			foreach ($shortcodes as $shortcode)
			{
				if ($shortcode['name'] !== 'brmap')
				{
					continue;
				}
				$attributes = array();
				if (empty($shortcode['attrs']))
				{
					continue;
				}

				foreach ($shortcode['attrs'] as $attr)
				{
					foreach ($attr as $key => $value)
					{
						$attributes[$key] = str_replace('"', '', $value);
					}
				}
				if (!isset($attributes['id']))
				{
					continue;
				}
				$attributes['lang'] = $langcode;
				try
				{
					$res->shortcodeMatched = true;
					$preData = $this->bridgeDataGetter->getMapData($attributes);
					if(! isset($preData['success']) || $preData['success'] !== true) {
						$res->errMsg  = "Erreur 8M shortcode Bridge moteur : " . $preData['message'];
						$res->text    = str_replace($shortcode['shortcode'], $res->errMsg, $res->text);
						$res->success = false;
					} else {
						$mapPreparedData = $this->bridgeDataGetter->prepareMapDataForRender($preData);
						$html = $this->bridgeClientRenderer->renderMapTemplate($mapPreparedData);
						$res->text = str_replace($shortcode['shortcode'], $html, $res->text);
					}
				}
				catch (\Exception $e)
				{
					$res->errMsg  = "Erreur 081 shortcode Bridge : " . $e->getMessage();
					$res->text    = str_replace($shortcode['shortcode'], $res->errMsg, $res->text);
					$res->success = false;
				}
			}
		}

		return $res;
	}
}
