<?php

namespace Bridge\Weblibs\Wordpress;

use Bridge\Weblibs\BridgeClientParamsInterface;
use Bridge\Weblibs\BridgeClientRendererInterface;
use Bridge\Weblibs\Wordpress\BridgeCmsAbstractLayerWordpress;


class BridgeClientRendererWordpress implements BridgeClientRendererInterface
{

    private $clientParams;
	private $bridgeCmsAbstractLayer;

    public function __construct(BridgeClientParamsInterface $clientParams, BridgeCmsAbstractLayerWordpress $bridgeCmsAbstractLayer)
	{
        $this->clientParams = $clientParams;
		$this->bridgeCmsAbstractLayer = $bridgeCmsAbstractLayer;
    }

    /**
     * Effectue le rendu d'un template avec des données passées
     * @param string $template
     * @param array $data
     * @return string Code HTML du rendu
     */
    public function renderTemplate(string $templatePath, array $data = array()) {
        ob_start();
        // $template = self::bridge_locate_template('templates-bridge/fiche/gabarits/' . $componentName . '.php');
        $template = $this->bridge_locate_template($templatePath);
        include($template);
        $html = ob_get_clean();
        return $html;
    }

    /**
     * Effectue le rendu d'un template de liste Ajax avec des données passées
     * @param array $data
     * @return string Code HTML du rendu
     */
    public function renderListeAjaxTemplate(array $listData = array()) {
        $this->loadListScriptsAndStyles();
        ob_start();
        // $template = self::bridge_locate_template('templates-bridge/fiche/gabarits/' . $componentName . '.php');
        $template = $this->bridge_locate_template('templates-bridge/ajax.php');
        include($template);
        $html = ob_get_clean();
        return $html;
    }

	/**
	 * Effectue le rendu d'un template de liste Ajax avec des données passées
	 * @param array $data
	 * @return string Code HTML du rendu
	 */
	public function renderListeTemplate(array $listData = array()) {
		$this->loadListScriptsAndStyles();
		ob_start();
		$template = $this->bridge_locate_template('templates-bridge/liste.php');
		include($template);
		$html = ob_get_clean();
		return $html;
	}

	/**
	 * Effectue le rendu d'un template de liste Ajax avec des données passées
	 * @param array $data
	 * @return string Code HTML du rendu
	 */
	public function renderFicheTemplate(array $ficheData = array()) {
		if (!empty($ficheData)) {
			$this->loadFicheScriptsAndStyles();
			ob_start();
			include $this->bridge_locate_template('templates-bridge/init-variables-globales-fiche.php');
			include $this->bridge_locate_template('templates-bridge/fiche/bridge-document-ready.php');
			include $this->bridge_locate_template('templates-bridge/fiche/' . $ficheData['ficheTemplate'] . '.php');
			$html = ob_get_clean();
			return $html;
		}
		return '';
	}

    /**
     * Charge une feuille de style css dans la page
     * @param $handle : id texte du css
     * @param string $src : url du css
     * @param array $deps : tableau de dépendances (liste d'id texte)
     * @param false $ver : version
     * @param string $media
     * @return mixed
     */
    public function BridgeEnqueueStyle($handle, $src = "", $deps = array(), $ver = false, $media = 'all') {
        return $this->bridgeCmsAbstractLayer->enqueueStyle($handle, $src, $deps, $ver, $media);
    }

    /**
     * Charge un script js dans la page
     * @param $name identifiant texte du script
     * @param $link URL du script
     * @return bool
     */
    public function BridgeEnqueueScript($handle, $src = "", $deps = array(), $ver = false, $in_footer = false ) {
        return $this->bridgeCmsAbstractLayer->enqueueScript($handle, $src, $deps, $ver, $in_footer);
    }

    /**
     * Retourne le header du theme front-end du site pour affichage avant le code de la fichede détail (WordPress)
     * @return mixed
     */
    public function getHeaderBridge($arg = null) {
        return $this->bridgeCmsAbstractLayer->getHeaderBridge($arg);
    }

    /**
     * Retourne le footer du theme front-end du site pour affichage après le code de la fichede détail (WordPress)
     * @return mixed
     */
    public function getFooterBridge() {
        return $this->bridgeCmsAbstractLayer->getFooterBridge();
    }

    /**
     * registerAllStyles
     * Enregistre tous les scripts utilisés par Bridge pour traiter l'affichage
     * Ils seront chargés plus tard
     */
    public function registerAllStyles() {
        /* Permet de charger tous les scripts qui gères les cartes SIT */
        wp_register_style( 'leaflet', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/css/leaflet/leaflet.css');
        //wp_register_style( 'leafleft', 'https://unpkg.com/leaflet@1.3.1/dist/leaflet.css');

        // Full screnn
        // wp_register_style( 'leaflet-fs', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/css/leaflet/Control.FullScreen.css');
        wp_register_style( 'leaflet-fs', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/css/leaflet/leaflet.fullscreen.css');

        // Clusters
        wp_register_style( 'leaflet-mc', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/css/leaflet/MarkerCluster.css');
        wp_register_style( 'leaflet-mc-theme', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/css/leaflet/MarkerCluster.Default.css');
        wp_register_style( 'leaflet-mc-theme', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/css/leaflet/MarkerCluster.Default.css');
        wp_register_style( 'leaflet-easy-button', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/css/leaflet/easy-button.css');

        wp_register_style('plugin-sit-style', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/css/plugin-sit-public.css');

        // Infinite scroll : inclus à plugin-sit.css
        // wp_register_style('infiniscroll-style', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/css/infiniscroll.css');

        // Uikit - Maj 13/11/2019
        wp_register_style( 'uikit-css', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/css/uikit/uikit.dn.min.css');

        // Datepicker
        wp_register_style('mc-datepicker', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/css/mc-calendar.min.css');

        // Font aweosome - MAJ 11/03/2022
        wp_register_style( 'fontawesome-v6-all', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/css/font-awesome/css/all.min.css');
        wp_register_style( 'fontawesome-v6-brands', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/css/font-awesome/css/brands.min.css');
        wp_register_style( 'fontawesome-v6-duotone', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/css/font-awesome/css/duotone.min.css');
        wp_register_style( 'fontawesome-v6-light', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/css/font-awesome/css/light.min.css');
        wp_register_style( 'fontawesome-v6-regular', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/css/font-awesome/css/regular.min.css');
        wp_register_style( 'fontawesome-v6-solid', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/css/font-awesome/css/solid.min.css');
        wp_register_style( 'fontawesome-v6-thin', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/css/font-awesome/css/thin.min.css');
        wp_register_style( 'fontawesome-v6-fa', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/css/font-awesome/css/fontawesome.min.css');

        // Compatibilté Font Awesome 5
        wp_register_style( 'fontawesome-v5', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/css/font-awesome/css/v5-font-face.min.css');

        wp_register_style('bridge-autocomplete', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/css/autocomplete/autoComplete.01.css');
        wp_register_style('bridge-datepicker', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/css/datepicker.min.css');
        wp_register_style('bridge-flatpickr', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/css/flatpickr/flatpickr.min.css');

        $customCss = $this->clientParams->getCustomCssURl();
        if(!empty($customCss))
            wp_register_style('sit-custom-css', $customCss);

        // Fiche : calendriers
        wp_register_style( "calendar", $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/css/calendar/calendar.css');
        wp_register_style( "calendar_custom", $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/css/calendar/calendar_custom.css');
    }

    /**
     * Enregistre tous les scripts utilisés par Bridge pour traiter l'affichage
     * Ils seront chargés plus tard
     */
    public function registerAllScripts() {

        /* Permet de charger tous les scripts qui gères les cartes SIT */
        // wp_register_script( 'jquery', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/jquery/jquery-3.6.0.min.js');

        /* Permet de charger tous les scripts qui gères les cartes SIT */
        // wp_register_script( 'leaflet', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/leaflet/leaflet1.2.0/leaflet.js');
        wp_register_script( 'leaflet', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/leaflet/leaflet1.8/leaflet.js');

        // Full screnn
        // wp_register_script( 'leaflet-fs', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/leaflet/Control.FullScreen.js');
        wp_register_script( 'leaflet-fs', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/leaflet/Leaflet.fullscreen.min.js');

        // Librairie DNC
        wp_register_script( 'dnc-leaflet-services', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/dncleafletservices.js');
        wp_register_script( 'togeojson', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/togeojson.js');

        // Clusters
        wp_register_script( 'leaflet-mc', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/leaflet/leaflet.markercluster.js');

        wp_register_script( 'leaflet-control-button', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/leaflet/L.Control.Button.js');
        wp_register_script( 'leaflet-easy-button', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/leaflet/easy-button.js');

        // Datepicker
        wp_register_script('mc-datepickerjs', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/mc-calendar.min.js');

        // Infinite scroll
        wp_register_script( 'infinite-scroll', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/infinite-scroll.pkgd.min.js', array('jquery'));

        // Uikit
        wp_register_script( 'uikit-js',$this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/uikit/uikit.min.js');

        wp_register_script( 'plugin-sit-script', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/plugin-sit-public.js',array('jquery'));

        wp_register_script( 'bridge-simple-list-map-loader', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/bridgeSimpleListMapLoader.js',array('jquery'));

        // wp_register_script( 'fontawesome', 'https://use.fontawesome.com/releases/v6.0.0/js/all.js' );
        wp_register_script( 'fontawesome', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/fontawesome/fontawesome.min.js' );
        wp_register_script( 'fontawesome-all', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/fontawesome/all.min.js', array( 'fontawesome' ) );
        wp_register_script( 'bridge-autocomplete', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/autoComplete.min.js', array( 'jquery' ) );
        // wp_register_script( 'bridge-autocomplete', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/autoComplete.js', array( 'jquery' ) );
        wp_register_script( 'bridge-moteur', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/bridge-moteur.js', array( 'jquery' ) );

        wp_register_script( 'bridge-datepicker', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/datepicker.min.js', array('jquery'), null, true);
        // Liste : date range picker
        wp_register_script( 'bridge-flatpickr', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/flatpickr/flatpickr.js' );

        $locale = $this->clientParams->getLanguageLocale();
        $lang = $this->clientParams->getLanguage();

        if(file_exists( $this->clientParams->bridgeDirPath() . 'public/bridgewebfrontlibs/js/datepicker/i18n/datepicker.' . $locale . '.js')) {
            $pickerLanguage = $locale;
            wp_register_script( 'bridge-datepicker-locale', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/datepicker/i18n/datepicker.' . $pickerLanguage . '.js', array('bridge-datepicker'), null, true);
        } elseif(file_exists( $this->clientParams->bridgeDirPath() . 'public/bridgewebfrontlibs/js/datepicker/i18n/datepicker.' . $lang . '-' . strtoupper($locale) . '.js')) {
            $pickerLanguage = $lang . '-' . strtoupper($locale) ;
            wp_register_script('bridge-datepicker-locale', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/datepicker/i18n/datepicker.' . $pickerLanguage . '.js', array('bridge-datepicker'), null, true);
        } elseif(file_exists( $this->clientParams->bridgeDirPath() . 'public/bridgewebfrontlibs/js/datepicker/i18n/datepicker.' . $lang . '-' . strtoupper($lang) . '.js')) {
            $pickerLanguage = $lang . '-' . strtoupper($lang) ;
            wp_register_script('bridge-datepicker-locale', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/datepicker/i18n/datepicker.' . $pickerLanguage . '.js', array('bridge-datepicker'), null, true);
        } else {
            $pickerLanguage = 'fr-FR';
            wp_register_script('bridge-datepicker-locale', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/datepicker/i18n/datepicker.' . $pickerLanguage . '.js', array('bridge-datepicker'), null, true);
        }
        // Flatpickr intl
        if(file_exists( $this->clientParams->bridgeDirPath() . 'public/bridgewebfrontlibs/js/flatpickr/i18n/' . $lang . '.js')) {
            wp_register_script('bridge-flatpickr-locale', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/flatpickr/i18n/' . $lang . '.js', array('bridge-flatpickr'), null, true);
        }
        // Fiche : calendriers
        wp_register_script( "modernizer_calendar", $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/calendar/modernizer.custom.js');
        wp_register_script( 'calendario', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/calendar/jquery.calendario.js');

        // Fiche : profil altimetrique
        // wp_register_script( "chart-js", 'https://cdnjs.cloudflare.com/ajax/libs/Chart.js/2.9.4/Chart.bundle.min.js');
        wp_register_script( "chart-js", $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/chart/chart.umd.min.js');

        // Librairie de pagination
        wp_register_script( 'infinite-scroll-ajax', $this->clientParams->bridgeDirUrl() . 'public/bridgewebfrontlibs/js/infinite-ajax-scroll.min.js' );


        $customJs = $this->clientParams->getCustomJsURl();
        if(!empty($customJs))
            wp_register_script('sit-custom-js', $customJs);

        // On ajoute un système de traduction pour certains scripts
        $bridgeTradsJs = array(
            'voir_carnet'   => __('Voir le Carnet de Voyage', "plugin-sit")
        );
        wp_localize_script('plugin-sit-script', 'bridgeTradsJs', $bridgeTradsJs);

    }

    /**
     * Retourne un tableau des librairies Js utilisée pour afficher les listes / moteurs
     * @return array
     */
    public function getListScripts() {
        $libraries = array();
        $libraries[] = 'uikit-js';
        $libraries[] = 'fontawesome';
        // $libraries[] = 'fontawesome-all';
        $libraries[] = 'leaflet';
        $libraries[] = 'leaflet-fs';
        $libraries[] = 'leaflet-mc';
        $libraries[] = 'leaflet-control-button';
        $libraries[] = 'leaflet-easy-button';
        $libraries[] = 'togeojson';
        $libraries[] = 'dnc-leaflet-services';
        $libraries[] = 'mc-datepickerjs';
        $libraries[] = 'infinite-scroll';
        $libraries[] = 'plugin-sit-script';
        $libraries[] = 'sit-custom-js';
        $libraries[] = 'bridge-autocomplete';
        $libraries[] = 'bridge-moteur';
        $libraries[] = 'bridge-datepicker';
        $libraries[] = 'bridge-datepicker-locale';
        $libraries[] = 'bridge-flatpickr';
        $libraries[] = 'bridge-flatpickr-locale';
        $libraries[] = 'infinite-scroll-ajax';
        return $libraries;
    }

    /**
     * Retourne un tableau des librairies css utilisée pour afficher les listes / moteurs
     * @return array
     */
    public function getListStyles() {
        $libraries = array();
        $libraries[] = 'uikit-css';
        $libraries[] = 'fontawesome-v6-all';
        $libraries[] = 'leaflet';
        $libraries[] = 'leaflet-fs';
        $libraries[] = 'leaflet-mc';
        $libraries[] = 'leaflet-mc-theme';
        $libraries[] = 'leaflet-easy-button';
        $libraries[] = 'leaflet-control-button';
        $libraries[] = 'mc-datepicker';
        $libraries[] = 'bridge-datepicker';
        $libraries[] = 'bridge-flatpickr';
        //$libraries[] = 'infiniscroll-style';
        $libraries[] = 'bridge-autocomplete';
        $libraries[] = 'plugin-sit-style';
        $libraries[] = 'sit-custom-css';
        return $libraries;
    }

    /**
     * Charge tous les scripts et styles utilisés sur les listes
     */
    public function loadListScriptsAndStyles() {
        $styles = $this->getListStyles();
        $scripts = $this->getListScripts();
        foreach($styles as $style)
            wp_enqueue_style($style);
        foreach($scripts as $script)
            wp_enqueue_script($script);
    }

    /**
     * Retourne un tableau des librairies Js utilisée pour afficher une fiche de détail
     * @return array
     */
    public function getFicheScripts() {
        $libraries = array();
        $libraries[] = 'uikit-js';
        $libraries[] = 'fontawesome';
        // $libraries[] = 'fontawesome-all';
        $libraries[] = 'leaflet';
        $libraries[] = 'leaflet-fs';
        $libraries[] = 'leaflet-mc';
        $libraries[] = 'leaflet-control-button';
        $libraries[] = 'leaflet-easy-button';
        $libraries[] = 'togeojson';
        $libraries[] = 'dnc-leaflet-services';
        $libraries[] = 'mc-datepickerjs';
        $libraries[] = 'infinite-scroll';
        $libraries[] = 'plugin-sit-script';
        $libraries[] = 'bridge-simple-list-map-loader';
        $libraries[] = 'sit-custom-js';
        $libraries[] = 'bridge-datepicker';
        $libraries[] = 'bridge-datepicker-locale';
        $libraries[] = 'modernizer_calendar';
        $libraries[] = 'calendario';

        return $libraries;
    }

    /**
     * Retourne un tableau des librairies CSS utilisée pour afficher une fiche de détail
     * @return array
     */
    public function getFicheStyles() {
        $libraries = array();
        $libraries[] = 'uikit-css';
        $libraries[] = 'fontawesome-v6-all';
        $libraries[] = 'leaflet';
        $libraries[] = 'leaflet-fs';
        $libraries[] = 'leaflet-mc';
        $libraries[] = 'leaflet-control-button';
        $libraries[] = 'leaflet-easy-button';
        $libraries[] = 'leaflet-mc-theme';
        $libraries[] = 'mc-datepicker';
        $libraries[] = 'bridge-datepicker';
        // $libraries[] = 'infiniscroll-style';
        $libraries[] = 'plugin-sit-style';
        $libraries[] = 'sit-custom-css';
        $libraries[] = 'calendar';
        $libraries[] = 'calendar_custom';

        return $libraries;
    }

    /**
     * Charge tous les scripts et styles utilisés sur les listes
     */
    public function loadFicheScriptsAndStyles() {
        $styles = $this->getFicheStyles();
        $scripts = $this->getFicheScripts();
        foreach($styles as $style)
            wp_enqueue_style($style);
        foreach($scripts as $script)
            wp_enqueue_script($script);
    }

    /**
     * Locate template.
     *
     * Localise un fichier template demandé (prend en compte les dossiers d'override du theme)
     * Pour wordpress, le recherche s'effectue dans cet ordre :
     * 1. /themes/theme/templates/$template_name
     * 2. /themes/theme/$template_name
     * 3. /plugins/plugin/templates/$template_name.
     *
     * @param   string  $template_name          Template to load.
     * @param   string  $string $template_path  Path to templates.
     * @param   string  $default_path           Default path to template files.
     * @return  string                          Path to the template file.
     */
    public function bridge_locate_template( $template_name, $template_path = '', $default_path = '' ) {
        // Set variable to search in the templates folder of theme.
        $tmpl = get_stylesheet_directory() . '/plugin-sit/' . $template_name;
        if ( ! file_exists( $tmpl ) ) {
            // If the override template does NOT exist, fallback to the default template.
            $tmpl = realpath( $this->clientParams->bridgeDirPath() . '/public/templates') . '/' . $template_name;
        }
        return apply_filters( 'bridge_locate_template', $tmpl, $template_name, $template_path, $default_path );
    }

    /**
     * Permet de localiser une feuille de style dynamique dans /public/bridgewebfrontlibs/css
     *
     * @param   string  $styleFileName          Feuille de style recherchée
     * @param   string  $styleFilePath          Chemin des styles
     * @param   string  $default_path           Chemin des styles par défaut
     * @return  string                          Chemin de la feuille de style recherchée
     */

    public function bridge_locate_style($styleFileName, $styleFilePath = '', $default_path = '') {
        $tmpl = realpath($this->clientParams->bridgeDirPath() . '/public/bridgewebfrontlibs/css') . '/' . $styleFileName;
        return apply_filters('bridge_locate_style', $tmpl, $styleFileName, $styleFilePath, $default_path);
    }

	/**
	 * Permet de déclarer une fonction de callback pour retrvailler du contenu dans les vues Bridge sans avoir à les overrider
	 * éuivalent à wordpress apply_filter
	 * @param $hookName : nom du hook
	 * @param $callBack : fonction de callback
	 * @param int $priority : priorité
	 * @param int $acceptedArgs : nombre d'arguments acceptés
	 */
	public function applyFilters($hookName, $callBack, $priority = 10, $acceptedArgs = 1) {
		return apply_filters($hookName,$callBack,$priority,$acceptedArgs);
	}

	/**
	 * prepareListViewRenderer : utile uniquement pour Drupal et Symfony (pour le moment)
	 * Prépare l'affichage d'une liste dans une page (non ajax)
	 * Attache les scripts et CSS et retourne un tableau associatif contenant toutes les données qui seront utilisées dans les vues
	 * $data Objet data tel que renvoyé par BridgeUtils::prepareListdataForRender
	 * $theme nom du fichier twig initial à charger
	 * @return array tableau associatif contenant toutes les données qui seront utilisées dans les vues
	 */
	public function prepareListViewRenderer($data, $theme = 'liste_sit') {
		return $data;
	}

	/**
	 * prepareFicheViewRenderer : utile uniquement pour Drupal et Symfony (pour le moment)
	 * Prépare l'affichage d'une fiche dans une page (non ajax)
	 * Attache les scripts et CSS et retourne un tableau associatif contenant toutes les données qui seront utilisées dans les vues
	 * @param $dataFiche Objet data tel que renvoyé par BridgeUtils::prepareFichedataForRender
	 * @param $theme nom du fichier twig initial à charger
	 * @return array tableau associatif contenant toutes les données qui seront utilisées dans les vues
	 */
	public function prepareFicheViewRenderer($datafiche, $theme = 'fiche_sit_ajax') {
		return $datafiche;
	}

}
