<?php

namespace Drupal\pylot_bridge\Commands;

use Drush\Commands\DrushCommands;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Drupal\Console\Core\Command\ContainerAwareCommand;
use Drupal\pylot_bridge\Services\BridgeUtils;
use Drupal\pylot_bridge\Services\BridgeShortCodeParser;
use Symfony\Component\DependencyInjection\ContainerInterface;
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\SMTP;

class Drush9BridgeCommands extends DrushCommands {

    /**
     * Dcommande drush qui importe les fiches et taxonomies SIT depuis l'applicaiton Bridge
     *
     * @command pylot_bridge:import
     * @aliases bridge-import
     * @usage pylot_bridge:import
     */
    public function import() {
        $this->output()->writeln('Début de l\'import Bridge complet');

        $credentials = BridgeUtils::getBridgeCredentials();
        $importService = \Drupal::service('pylot_bridge.import');


        $res = $importService->doImport(true);

        if($res === false) {
            $this->output()->writeln("\nImport arrêté en erreur\n");
            $this->output()->writeln($importService->getErrorMsg());

            // Envoi du mail d'erreur
            $address = BridgeUtils::getAdminMailAddress($credentials->siteID);
            if(!$address) {
                $address = \Drupal::config('system.site')->get('mail');
            }

            $this->output()->writeln("Envoi d'un email d'erreur à l'adresse '$address'");

            $mail = new PHPMailer(true);

            $config = \Drupal::config('smtp.settings');

            // Si jamais le plugin drupal/smtp n'est pas installé, on utilise les paramètres par défaut de phpmailer
            if(empty($config->get('smtp_host'))) {
                $config = \Drupal::config('phpmailer_smtp.settings');
            }

            // et si php_mailer n'est pas installé/configuré, on ressort une erreur
            // Quid de l'utilisation de la fonction native mail() de php ? Pas secure du tout mais peut dépanner
            if(empty($config->get('smtp_host'))) {
                $this->output()->writeln('Aucun serveur SMTP configuré. Aucun mail d\'erreur envoyé.');
                die;
            }

            try {
                $mail->isSMTP();
                $mail->Host       = $config->get('smtp_host');
                $mail->SMTPAuth   = true;
                $mail->Username   = $config->get('smtp_username');
                $mail->Password   = $config->get('smtp_password');
                $mail->Port       = $config->get('smtp_port');

                $mail->setFrom($config->get('smtp_from') , $config->get('smtp_fromname'));
                $mail->addAddress($address);

                $siteName = \Drupal::config('system.site')->get('mail');

                $mail->Subject = "[DRUPAL] $siteName - Erreur lors de l'import Bridge";

                $mail->isHTML(false);

                $emailBody = "Erreur dans l'import Bridge sur le site $siteName\n\n";
                $emailBody .= "Erreur : \n" . $importService->getErrorMsg() . "\n\n";
                $emailBody .= "Logs : \n" . $importService->getCurrentLog() . "\n\n";

                $mail->Body = $emailBody;

                if(!$mail->send()) {
                    $this->output()->writeln('Erreur : ' . $mail->ErrorInfo);
                } else {
                    $this->output()->writeln('Message envoyé');
                }
            }
            catch (Exception $e) {
                $this->output()->writeln('Erreur : ' . $mail->ErrorInfo);
            }
            die('');
        } else {
            $warnings = $importService->getWarnings();
            if(count($warnings) > 0) {
                $this->output()->writeln ("\nImport terminé avec avertissements\n");
                echo implode("\n", $warnings);
                die('');
            } else {
                $this->output()->writeln("\nImport terminé avec succès\n");
            }
        }

        $this->output()->writeln('Import terminé');
    }
    /**
     * commande Drush qui supprime les nodes et termes de taxonomie SIT issus de Bridge
     *
     * @command pylot_bridge:delete
     * @aliases bridge-delete
     * @usage pylot_bridge:delete
     */
    public function delete() {
        $this->output()->writeln('Début de la suppression');

        $credentials = BridgeUtils::getBridgeCredentials();
        $importService = \Drupal::service('pylot_bridge.import');
        $res = $importService->deleteAllData(true);
        if($res === false) {
            $this->output()->writeln("\nSuppression arrêtée en erreur\n");
            $this->output()->writeln($importService->getErrorMsg());
            die('');
        } else {
            $this->output()->writeln("\nSuppression terminée avec succès\n");
        }
    }
    /**
     * commande Drush qui met uniquement à jour les termes de taxonomie SIT depuis l'application  Bridge
     *
     * @command pylot_bridge:import_taxonomies
     * @aliases bridge-import-taxonomies
     * @usage pylot_bridge:import_taxonomies
     */
    public function import_taxonomies() {
        $this->output()->writeln('Début de l\'import des termes');

        $importService = \Drupal::service('pylot_bridge.import');
        $res = $importService->update_sit_terms_from_bridge();
        if($res === false) {
            $this->output()->writeln("\nImport arrêté en erreur - erreur à la mise à jour initiale des termes\n");
            $this->output()->writeln($importService->getErrorMsg());
            die('');
        } else {
            $this->output()->writeln("\nImport terminé avec succès\n");
        }
    }

    /**
     * commande Drush qui met uniquement à jour l'affectation des termes aux posts SIT
     *
     * @command pylot_bridge:update_post_categories
     * @aliases bridge-update-post-categories
     * @usage pylot_bridge:update_post_categories
     */
    public function update_post_categories() {
        $this->output()->writeln('Début de l\'import des termes');

        $importService = \Drupal::service('pylot_bridge.import');
        $res = $importService->updatePostCategories();
        if($res === false) {
            $this->output()->writeln("\nImport arrêté en erreur - erreur à la mise à jour des catégories de posts\n");
            $this->output()->writeln($importService->getErrorMsg());
            die('');
        } else {
            $this->output()->writeln("\nImport terminé avec succès\n");
        }
    }

    /**
     * commande Drush qui met uniquement à jour les termes de taxonomie SIT depuis l'application  Bridge
     *
     * @command pylot_bridge:test
     * @aliases bridge-test
     * @usage pylot_bridge:test
     */
    public function test_shortcode() {
        // $this->output()->writeln('Début de l\'import des termes');
        print_r(BridgeShortCodeParser::parse_shortcodes('uisqudoisq oisq [include file="header.html"] dsiodisqopd idoq idoq disqo d'));

    }

    /**
     * Commande qui vérifie si un import est nécessaire, puis le lance si besoin
     *
     * @command pylot_bridge:check_import
     * @aliases bridge:ci
     * @usage pylot_bridge:check_import
     */
    public function check_import() {
        $this->output()->writeln('Vérification de la nécessité de lancer un import...');

        $config = \Drupal::service('config.factory')->getEditable('pylot_bridge.bridgeconfig');

        if(!$config->get('need_to_import')) {
            $this->output()->writeln('Pas besoin de lancer un import');
            return;
        }

        $config->set('need_to_import', false)->save();

        $this->output()->writeln('Lancement de l\'import...');
        $this->import();
    }
}
