<?php

namespace Drupal\pylot_bridge\Hook;
use Drupal\pylot_bridge\Utils\BridgeCmsAbstractLayer;
use Drupal\pylot_bridge\Utils\BridgeBlockService;
use Drupal\pylot_bridge\Utils\BridgeUtils;
use Bridge\Weblibs\BridgeDataGetter;
use Bridge\Weblibs\BridgeRequestServiceInterface;
use Bridge\Weblibs\BridgeClientParamsInterface;
use Bridge\Weblibs\BridgeClientRendererInterface;
use Bridge\Weblibs\BridgeClientContentInterface;

use Drupal\Core\Entity\Display\EntityViewDisplayInterface;
use Drupal\node\NodeInterface;
use Drupal\Core\PageCache\ResponsePolicy\KillSwitch;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Class NodeViewHandler
 *
 * @package Drupal\pylot_bridge\Hook
 *
 * @author Pylot
 *
 * @license GNU/GPL v3 or later
 *
 * @version 3.2.6
 *
 */

final class NodeViewHandler
{

    public function __construct(
        private BridgeDataGetter $bridgeDataGetter,
        private BridgeRequestServiceInterface $bridgeRequestService,
        private BridgeClientParamsInterface $bridgeClientParams,
        private BridgeClientRendererInterface $bridgeClientRenderer,
        private BridgeClientContentInterface $bridgeClientContent,
        private readonly RequestStack $requestStack,
        private readonly KillSwitch $pageCacheKillSwitch,          // service: page_cache_kill_switch
        private readonly KillSwitch $dynamicPageCacheKillSwitch,   // service: dynamic_page_cache_kill_switch
    )
    {
    }

    public function onNodeView(array &$build, \Drupal\node\Entity\Node $entity, \Drupal\Core\Entity\Display\EntityViewDisplayInterface $display, $view_mode): void {
        // Example: Add a custom message to the node view.
        // Affichage d'une fiche SIT
        $request = $this->requestStack->getCurrentRequest();
        if ($entity->getType() == 'fiche_sit' && $view_mode == "full") {
            $noCache = $request->query->getBoolean('noCache');
            if($noCache) {
                // Désactivation du cache pour cette page
                $build['#cache']['max-age'] = 0;

                // Désactive aussi Page Cache + Dynamic Page Cache pour être sûr.
                $this->pageCacheKillSwitch->trigger();
                $this->dynamicPageCacheKillSwitch->trigger();
            }
            // On injecte les scripts et feuilles de style nécessaires
            $libraires = $this->bridgeClientRenderer->getFicheLibraries();
            foreach( $libraires as $lib)
                $build['#attached']['library'][] = $lib;

            try {
                $nodeId = $entity->id();
                $productCode = $entity->get('field_code_sit')->value;
                $rubriques = $entity->get('field_rubrique_sit_canonique')->getValue();
                $pageId = '';
                // On recherche le modèle de fiche défini au niveau de la rubrique principale => doit normalement toujours être renseigné
                if (is_array($rubriques) && count($rubriques) > 0) {
                    $rubrique = reset($rubriques);
                    $pageId = $this->bridgeClientContent->getWebPageIdFromTermId($rubrique['target_id']);
                }

                // Si jamais on en a pas, on recherche le premier modèle de fiche défini dans les autres listes associées à la fiche
                if(empty($pageId)) {
                    $rubriques = $entity->get('field_rubriques_sit')->getValue();
                    if (is_array($rubriques) && count($rubriques) > 0) {
                        $rubrique = reset($rubriques);
                        $pageId = $this->bridgeClientContent->getWebPageIdFromTermId($rubrique['target_id']);
                    }
                }

                // Récupération des données de la fiche SIT
                // $datafiche = BridgeUtils::getDataFiche($productCode, $pageId);
                $datafiche = $this->bridgeDataGetter->getDataFiche($productCode, $pageId);

                // Si erreur, on injecte le message dans la variable dédiée
                if(isset($datafiche->error) && !empty($datafiche->error)) {
                    $build['body'] = array(
                        'error' => $datafiche->error
                    );
                } else {
                    // On récupère les paramètres de Bridge pour les passer au Twig
                    $bridgeParameters = $this->bridgeClientParams->getBridgeParameters(); // BridgeUtils::getBridgeParameters();

                    // $test = BridgeUtils::getElevationData($datafiche);

                    $build['body'] = array(
                        'fiche' => $datafiche,
                        'bridgeParameters' => $bridgeParameters,
                    );
                }
            } catch (\Exception $e) {
                $stack = $e->getTraceAsString();

                $build['body'] = array(
                    'error' => 'Erreur 01 : ' . $e->getMessage() . ' - Trace : ' . $stack,
                );
            }
        }
    }
}
