<?php

namespace Drupal\pylot_bridge\Services;

use Bridge\Weblibs\BridgeClientParamsInterface;
use Bridge\Weblibs\BridgeCmsAbstractLayerInterface;
use Bridge\Weblibs\BridgeRequestServiceInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Url;
use Drupal\Core\Database\Connection;
use Drupal\Core\Extension\ExtensionPathResolver;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use function \Drupal\base_path;


class BridgeClientParamsDrupal implements BridgeClientParamsInterface
{
    private BridgeRequestServiceInterface $requestService;
	private BridgeCmsAbstractLayerInterface $bridgeCmsAbstractLayer;
    protected $configFactory;
    protected $database;
    protected $extensionPathResolver;
    protected $languageManager;
    protected $requestStack;

    public function __construct(
        BridgeRequestServiceInterface $requestService,
        BridgeCmsAbstractLayerInterface $bridgeCmsAbstractLayer,
        ConfigFactoryInterface $configFactory,
        Connection $database,
        ExtensionPathResolver $extensionPathResolver,
        LanguageManagerInterface $languageManager,
        RequestStack $requestStack
    ) {
		$this->requestService = $requestService;
		$this->bridgeCmsAbstractLayer = $bridgeCmsAbstractLayer;
        $this->configFactory = $configFactory;
        $this->database = $database;
        $this->extensionPathResolver = $extensionPathResolver;
        $this->languageManager = $languageManager;
        $this->requestStack = $requestStack;
    }

    /**
     * Retourne un tableau de Groupes d'options de markup HTML pour les listes (titres, conteneurs, etc.)
     * @return array Tableau de Groupes d'options de markup HTML pour les listes
     */
    public function getMarkupOptionGroups() {
        return array(
            'Listes et carrousels' => array (
                'tag_list_container' => 'Conteneur de liste',
                'tag_list_pagination' => 'Conteneur de pagination',
                'tag_listitem_container' => 'Conteneur d\'élément de liste',
                // 'list_titre' => 'Titre de liste', // kept commented like original
            ),
            'Eléments des listes et carrousels' => array(
                'tag_listitem_surtitre' => 'Surtitre',
                'tag_listitem_titre' => 'Titre',
                'tag_listitem_soustitre' => 'Sous-titre',
                'tag_listitem_commentaire' => 'Commentaire',
                'tag_listitem_pictogrammes' => 'Pictogrammes',
                'tag_listitem_info1' => 'Info1',
                'tag_listitem_info2' => 'Info2',
                'tag_listitem_info3' => 'Info3',
                'tag_listitem_info4' => 'Info4',
            ),
            'Moteur' => array(
                'tag_moteur_titre_section' => 'Titre de section',
                'tag_moteur_titre_subsection' => 'Titre de sous-section',
                'tag_moteur_titre_filtre' => 'Titre de filtre',
            ),
            'Fiche' => array(
                'tag_fiche_titre_infosupplementaires1' => 'Titre de bloc infos supplémentaires 1',
                'tag_fiche_titre_infosupplementaires2' => 'Titre de bloc infos supplémentaires 2',
                'tag_fiche_titre_infosupplementaires3' => 'Titre de bloc infos supplémentaires 3',
                'tag_fiche_titre_infosupplementaires4' => 'Titre de bloc infos supplémentaires 4',
                'tag_fiche_titre_infosupplementaires5' => 'Titre de bloc infos supplémentaires 5',
                'tag_fiche_titre_groupe_blocperso' => 'Titre de groupe de blocs perso',
                'tag_fiche_titre_blocperso' => 'Titre de bloc perso',
            ),
        );
    }

    /**
     * Retourne un tableau des options de balises HTML recommandées pour les listes (titres, conteneurs, etc.)
     * @return array Tableau des options de balises HTML recommandées pour les listes
     */
    public function getRecommendedMarkupOptions() {
        return array(
            'tag_list_container' => 'div',
            'tag_list_pagination' => 'nav',
            'tag_listitem_container' => 'div',
            'tag_listitem_titre' => 'h2',
            'tag_listitem_commentaire' => 'p',
            'tag_moteur_titre_section' => 'h2',
            'tag_moteur_titre_subsection' => 'h2',
            'tag_moteur_titre_filtre' => 'span',
            'tag_fiche_titre_infosupplementaires1' => 'h2',
            'tag_fiche_titre_infosupplementaires2' => 'h2',
            'tag_fiche_titre_infosupplementaires3' => 'h2',
            'tag_fiche_titre_infosupplementaires4' => 'h2',
            'tag_fiche_titre_infosupplementaires5' => 'h2',
            'tag_fiche_titre_groupe_blocperso' => 'h2',
            'tag_fiche_titre_blocperso' => 'h2',
        );
    }
    /**
     * @return array[] Tableau des options de balises HTML pour les listes déroulantes de sélection dans l'administration
     */
    public function getHtmlTagsForMarkupOptions() {
        return array(
            array('label' => '-- Sélectionnez --', 'value' => ''),
            array('label' => 'H1', 'value' => 'h1'),
            array('label' => 'H2', 'value' => 'h2'),
            array('label' => 'H3', 'value' => 'h3'),
            array('label' => 'H4', 'value' => 'h4'),
            array('label' => 'DIV', 'value' => 'div'),
            array('label' => 'SPAN', 'value' => 'span'),
            array('label' => 'P', 'value' => 'p'),
            array('label' => 'MAIN', 'value' => 'main'),
            array('label' => 'NAV', 'value' => 'nav'),
        );
    }
    /**
     * Retourne l'URL relative pour les appels AJAX des templates front-end qui lancent la récupération des données des listes
     */
    public function getAjaxURL() {
        $lang = $this->bridgeCmsAbstractLayer->getCurrentLanguage();
        if($lang == 'fr')
            return '/pylot_bridge/list_products_json';

        return'/' . $lang . '/pylot_bridge/list_products_json';
    }


    /**
     * Met à jour un réglage de Bridge en base de données
     *
     * @param $optionName
     * @param $optionValue
     */
    public function bridgeUpdateOption($optionName, $optionValue) {
        // Non testé sous Drupal
        $config = $this->configFactory->getEditable('pylot_bridge.bridgeconfig');
        $config->set($optionName, $optionValue)->save();
        return true;
    }

    /**
     * @return string code de la langue courante sur 2 lettres
     */
    public function getLanguage()
    {
        return $this->bridgeCmsAbstractLayer->getCurrentLanguage();
    }

    /**
     * @return string code de la langue courante sous la forme fr-FR ou en-GB (langue - pays)
     */
    public function getLanguageLocale()
    {
        return $this->bridgeCmsAbstractLayer->getCurrentLanguageLocale();
    }

    /**
     * DRUPAL : Fonction indiquant le préfixe de langue à ajouter aux liens URL - sans objet dans Wordpress
     */
    public function getLanguagePrefix() {
        if($prefixes = \Drupal::config('language.negotiation')->get('url.prefixes')) {
            $language = \Drupal::languageManager()->getCurrentLanguage()->getId();

            if(!empty($prefixes[$language])) {
                return "/" . $prefixes[$language];
            } else {
                return '';
            }

        }
        return '';
    }

    /**
     * @return mixed|string URL du serveur Pylot Bridge (application distante)
     */
    public function getBridgeUrl()
    {
        $config = \Drupal::config('pylot_bridge.bridgeconfig');
        return $config->get('url_bridge');
    }
    /**
     * Retounre l'URL de Pylot Bridge telle qu'enregistrée dans les paramètres
     * @return array|mixed|null URL de Pylot Bridge
     */
    public function getUrlBridge() {
        return $this->getBridgeUrl();
    }

    /**
     * @return mixed|string URL de la feuille de styles personnalisée définie dans le paramètres de l'extension
     */
    public function getCustomCssURl()
    {
        return '';
    }

    /**
     * @return mixed|string URL du script personnalisé défini dans le paramètres de l'extension
     */
    public function getCustomJsURl()
    {
        return '';
    }

    /**
     * @return object Paramètres de l'extension relatifs à l'affichage des cartes
     */
    public function getMapsDefaultSettings()
    {

        $obj = new \stdClass();
        $obj->latitude = '48.8534';
        $obj->longitude = '2.3488';
        $assetsBaseUrl = $this->bridgeDirUrl() . '/assets/img/';
        $obj->marker = $assetsBaseUrl . 'map-marker-icon.png';
        $obj->activeMarker = $assetsBaseUrl . 'map-marker-icon.png';
        $obj->activeGeoloc = false;
        $obj->defaultGeolocIconUrl = $assetsBaseUrl . 'geoloc-icon.svg';

        return $obj;
    }

    /**
     * @return false|mixed|string Chemin de base relatif à la réécriture des URL de taxonomies (hors permalien personnalisé)
     */
    public function getTaxonomyRootUrl()
    {
        $config = \Drupal::config('pylot_bridge.bridgeconfig');
        $url =  $config->get('fiche_root_url');
        if (!empty($url)) {
            $temp = $url;
            if (substr($temp, 0, 1) == '/')
                $temp = substr($temp, 1);
            if (substr($temp, 0, 1) == '/')
                $temp = substr($temp, 1);
            if (substr($temp, 0, 1) == '/')
                $temp = substr($temp, 1);
            return '/' . $temp;
        } else {
            return '/sit';
        }
    }

    /**
     * @return false|mixed|string Chemin de base relatif à la réécriture des URL des fiches (hors permalien personnalisé)
     */
    public function getFicheRootUrl()
    {
        $config = \Drupal::config('pylot_bridge.bridgeconfig');
        $url =  $config->get('fiche_root_url');
        if (!empty($url)) {
            $temp = $url;
            if (substr($temp, 0, 1) == '/')
                $temp = substr($temp, 1);
            if (substr($temp, 0, 1) == '/')
                $temp = substr($temp, 1);
            if (substr($temp, 0, 1) == '/')
                $temp = substr($temp, 1);
            return '/' . $temp;
        } else {
            return '/fiche-sit';
        }
    }

    /**
     * Retourne l'URL du dossier du plugin Pylot Bridge
     * @return string URL du dossier du plugin Pylot Bridge
     */
    public function bridgeDirUrl()
    {
        $rootUrl = \Drupal::request()->getSchemeAndHttpHost();
        $module_handler = \Drupal::service('module_handler');
        $bridgePath = $module_handler->getModule('pylot_bridge')->getPath();
        return $rootUrl . '/' . $bridgePath;
    }

    /**
     * Retournre le chemin physique du dossier du plugin Pylot Bridge
     * @return string Chemin du dossier du plugin Pylot Bridge
     */
    public function bridgeDirPath()
    {
        //return $this->extensionPathResolver->getPath('module', 'pylot_bridge');
        return realpath(dirname(__FILE__) . '');
    }

    /**
     * Retourne le chemin physique du dossier des telmplates d'affichage du plugin Pylot Bridge
     * @return string URL du dossier des telmplates d'affichage du plugin Pylot Bridge
     */
    public function bridgeTemplatesPath()
    {
        return $this->bridgeDirPath().'/templates/';

    }

    /**
     * Retourne l'URL du script php de resizer d'image Bridge
     * @return string URL du resizer d'image Bridge
     */
    public function getResizeImage() {
        return Url::fromRoute('pylot_bridge.image_resizer')->toString();
    }

    /**
     * getResizeImageLink
     * Génère un lien d'image redimensionnée selon les paramètres passés
     * @param $url URL de la photo originale
     * @param string $mode mode de redimensonnement :
     * - ajust pour ajuster aux dimensions passées (ne déborde pas)
     * - remplir : pour remplir le cadre de dimensions passées
     * - deform : pour déformer l'image et forcer les dimensions exactes
     * @param string $width largeur souhaitée
     * @param string $height hauteur souhaitée
     * @param string $quality qualité de compresion (sur 100 pour le jpg et de 0 à 9 pour le png)
     * @param int $timeToCache durée de cache navigateur à indiquer dans l'en-tête http
     * @return string l'URL du service qui affiche les données d'image redimensionnée
     */
    public function getResizeImageLink($url, $mode = "ajust", $width = "150", $height = "150", $quality = "60", $timeToCache = "1800") {
        $queryArgs = array(
            'lemode' => $mode,
            'selwidth' => $width,
            'selheight' => $height,
            'def' => $quality,
            'timeToCache' => $timeToCache,
            'file' => $url,
        );

        return Url::fromRoute('pylot_bridge.image_resizer', [], ['query' => $queryArgs])->toString();
    }

    /**
     * Retourne un objet contenant les paramètres de connexion au serveur Bridge (serveur et identifiants Bridge)
     *
     * @return object objets contenant les paramètres de connexion au serveur Bridge
     */
    public function getBridgeCredentials()
    {
        $config = \Drupal::config('pylot_bridge.bridgeconfig');

        $obj = new \stdClass();
        $obj->urlBridge = $config->get('url_bridge');

        $obj->loginBridge = $config->get('login_bridge');
        $obj->passwordBridge = $config->get('password_bridge');
        $obj->siteID = $config->get('bridge_site');


        return $obj;
    }

    /**
     * Permet de récupérer un token de connexion à l'API Bridge à partir d'identifiants
     *
     * @param string $url Url du serveur Bridge
     * @param string $login login Bridge
     * @param string $pw mot de passe Bridge
     * @throws \Exception
     * @return string token de connexion
     */
    public function getBridgeTokenFromCredentials(string $url, string $login, string $pw) {

        if ($url != '' && $login != '' && $pw != '') {
            $data = array(
                'username' => $login,
                'password' => $pw
            );
            $options = array(
                'http' => array(
                    'method' => 'POST',
                    'header' => "Content-Type: application/json",
                    'ignore_errors' => true,
                    'timeout' => 10,
                    'content' => json_encode($data),
                )
            );

            $context = stream_context_create($options);
            $res = json_decode(@file_get_contents($url . '/login_check', false, $context));

            if (isset($res) && isset($res->token)) {
                return $res->token;
            } else {
                throw new \Exception("Invalid credentials");
            }
        }
        return '';
    }

    /**
     * Permet de récupérer le token de connexion à l'API Bridge
     * @return string token de connexion à l'API Bridge
     * @throws \Exception
     */
    public function getBridgeToken(bool $useCache = true)
    {
        if($useCache) {
          $resCache = $this->requestService->getCachedData('bridgeApiToken', 120);
          if(!empty($resCache)) {
              return $resCache;
          }
        }

        $tmpOptions = $this->getBridgeCredentials();
        $urlBridge = $tmpOptions->urlBridge;
        $loginBridge = $tmpOptions->loginBridge;
        $passwordBridge = $tmpOptions->passwordBridge;

        $token = $this->getBridgeTokenFromCredentials($urlBridge, $loginBridge, $passwordBridge);
        if($useCache) {
            $this->requestService->storeCachedData('bridgeApiToken', $token);
        }
        return $token;
    }

    /**
     * Retounre l'id de site Pylot Bridge telle qu'enregistré dans les paramètres
     * @return array|mixed|null id de site Pylot Bridge
     */
    public function getSiteBridge() {
        $config = \Drupal::config('pylot_bridge.bridgeconfig');
        return $config->get('bridge_site');
    }

    /**
     * Retourne l'id du bock photos par défaut de Pylot Bridge tel qu'enregistré dans les paramètres
     * @return array|mixed|null id de bloc photos
     */
    public function getIdBlockPhotos() {
        $config = \Drupal::config('pylot_bridge.bridgeconfig');
        return $config->get('product_media_block');
    }

    /**
     * Retourne un tableau contenant les paramètres généraux du plugin Pylot Bridge ainsi que des données d'environnement
     * @return array tablea de paramètres
     */
    public function getBridgeParameters() {

        // On récupère les paramètres de Bridge pour les passer au Twig
        $bridgeParameters = array();
        $config = \Drupal::config('pylot_bridge.bridgeconfig');
        $bridgeParameters['url_bridge'] = $config->get('url_bridge');
        $bridgeParameters['custom_css_sit'] = $config->get('custom_css_sit');
        $bridgeParameters['custom_js_sit'] = $config->get('custom_js_sit');
        $bridgeParameters['bridge_site'] = $config->get('bridge_site');
        $bridgeParameters['marker_map'] = $config->get('marker_map');
        $bridgeParameters['marker_map_actif'] = $config->get('marker_map_actif');
        $bridgeParameters['product_media_block'] = $config->get('product_media_block');

        $rootUrl = \Drupal::request()->getSchemeAndHttpHost();
        // $rootUrl = Url::fromRoute('<front>', [], ['absolute' => TRUE])->toString();
        $module_handler = \Drupal::service('module_handler');
        $bridgePath = $module_handler->getModule('pylot_bridge')->getPath();
        $bridgeParameters['url_root'] = $rootUrl;
        $bridgeParameters['url_module_bridge'] = $bridgePath;
        $uri_parts = explode('?', $_SERVER['REQUEST_URI'], 2);
        $bridgeParameters['currentUrl'] = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://$_SERVER[HTTP_HOST]$uri_parts[0]" ;

        // Paramètres d'URL pour la pagination des listes
        $paginationUrlParams = $_SERVER['QUERY_STRING'] ;
        if(!empty($paginationUrlParams)) {
            $tabUrlParams = array();
            parse_str($paginationUrlParams, $tabUrlParams);
            // On retire le paraetre brpa de l'url courante pour construire les liens de pagination
            if(isset($tabUrlParams['brpa'])) {
                unset($tabUrlParams['brpa']);
                $paginationUrlParams = http_build_query($tabUrlParams);
            }
            if(!empty($paginationUrlParams))
                $paginationUrlParams = '?' . $paginationUrlParams;
        }
        $bridgeParameters['paginationUrlParams'] = $paginationUrlParams;

        if(empty($bridgeParameters['marker_map']))
            $bridgeParameters['marker_map'] = $bridgeParameters['url_root']  . '/'  . $bridgeParameters['url_module_bridge'] . '/assets/img/map-marker-icon.png';

        if(empty($bridgeParameters['marker_map_actif']))
            $bridgeParameters['marker_map_actif'] = $bridgeParameters['marker_map'] ;

        $bridgeParameters['marker_home'] = $bridgeParameters['url_root']  . '/'  . $bridgeParameters['url_module_bridge'] . '/assets/img/map-marker-home.png';

        // On donne des dimensions proportionnelles au marker avec en visée 50 px de haut
// On donne des dimensions proportionnelles au marker avec en visée 50 px de haut
        if(strtolower(substr($bridgeParameters['marker_map'], -3)) == 'svg') {
            $markerImage = $this->requestService->cachedFileGetContent($bridgeParameters['marker_map']);
            if(!empty($markerImage)) {
                try {
                    $xml = simplexml_load_string($markerImage);
                    if ($xml === false) {
                        throw new \Exception('Cannot load SVG from string');
                    }

                    $attr = $xml->attributes();
                    if ( is_object( $attr ) && isset( $attr->width ) && isset( $attr->height ) ) {
                        $ratio                                 = 50 / (int) $attr->height;
                        $bridgeParameters['marker_map_width']  = round( (int) $attr->width * $ratio );
                        $bridgeParameters['marker_map_height'] = round( (int) $attr->height * $ratio );
                    } else {
                        $bridgeParameters['marker_map_width']  = 35;
                        $bridgeParameters['marker_map_height'] = 50;
                    }
                } catch ( \Exception $e ) {
                    $bridgeParameters['marker_map_width']  = 35;
                    $bridgeParameters['marker_map_height'] = 50;
                }
            }
        } else {
            $markerPath = $bridgeParameters['marker_map'];
            $image_info = @getimagesize($markerPath);
            if($image_info) {
                $ratio = 50 / $image_info[1];
                $bridgeParameters['marker_map_width'] = round($image_info[0] * $ratio);
                $bridgeParameters['marker_map_height'] = round($image_info[1] * $ratio);
            } else {
                $bridgeParameters['marker_map_width'] = 35;
                $bridgeParameters['marker_map_height'] = 50;
            }
        }

        if(strtolower(substr($bridgeParameters['marker_map_actif'], -3)) == 'svg') {
            $markerActifImage = $this->requestService->cachedFileGetContent($bridgeParameters['marker_map_actif']);
            if(!empty($markerActifImage)) {
                try {
                    $xml = simplexml_load_string($markerActifImage);
                    if ($xml === false) {
                        throw new \Exception('Cannot load SVG from string');
                    }

                    $attr = $xml->attributes();
                    if ( is_object( $attr ) && isset( $attr->width ) && isset( $attr->height ) ) {
                        $ratio                                       = 50 / (int) $attr->height;
                        $bridgeParameters['marker_map_actif_width']  = round( (int) $attr->width * $ratio );
                        $bridgeParameters['marker_map_actif_height'] = round( (int) $attr->height * $ratio );
                    } else {
                        $bridgeParameters['marker_map_actif_width']  = 35;
                        $bridgeParameters['marker_map_actif_height'] = 50;
                    }
                } catch ( \Exception $e ) {
                    $bridgeParameters['marker_map_actif_width']  = 35;
                    $bridgeParameters['marker_map_actif_height'] = 50;
                }
            }
        } else {
            $markerActifPath = $bridgeParameters['marker_map_actif'];
            $image_info = @getimagesize($markerActifPath);
            if($image_info) {
                $ratio = 50 / $image_info[1];
                $bridgeParameters['marker_map_actif_width'] = round($image_info[0] * $ratio);
                $bridgeParameters['marker_map_actif_height'] = round($image_info[1] * $ratio);
            } else {
                $bridgeParameters['marker_map_actif_width'] = 35;
                $bridgeParameters['marker_map_actif_height'] = 50;
            }
        }



        /* $image_info = getImageSize($bridgeParameters['marker_map']);

        // Protection contre les erreurs de paramétrage
        if(is_array($image_info) && count($image_info) > 1 && $image_info[1] != 0 && $image_info[0] != 0) {
            $ratio = 50 / $image_info[1];
            $bridgeParameters['marker_map_width'] = $image_info[0] * $ratio;
            $bridgeParameters['marker_map_height'] = $image_info[1] * $ratio;

            $image_info = getImageSize($bridgeParameters['marker_map_actif']);
            $ratio = 50 / $image_info[1];
            $bridgeParameters['marker_map_actif_width'] = $image_info[0] * $ratio;
            $bridgeParameters['marker_map_actif_height'] = $image_info[1] * $ratio;
        } else {
            $bridgeParameters['marker_map_width'] = 30;
            $bridgeParameters['marker_map_height'] = 40;
            $bridgeParameters['marker_map_actif_width'] = 30;
            $bridgeParameters['marker_map_actif_height'] = 40;
        } */


        $bridgeParameters['maps_center_lat'] = $config->get('maps_center_lat');
        if(empty($bridgeParameters['maps_center_lat']))
            $bridgeParameters['maps_center_lat'] = 48.856614;
        $bridgeParameters['maps_center_lon'] = $config->get('maps_center_lon');
        if(empty($bridgeParameters['maps_center_lon']))
            $bridgeParameters['maps_center_lon'] = 2.3522219;

        // On ajoute aux paramètres d'application les constantes système de manière à tout avoir sous la main et ne pas avoir à faire trop d'appels
        $lang = $this->bridgeCmsAbstractLayer->getCurrentLanguage();
        $bridgeParameters['url_ajax_liste'] = $rootUrl . '/pylot_bridge/list_products_json' ;
        $bridgeParameters['url_ajax_liste_json_poi'] = $rootUrl . '/pylot_bridge/list_products_json_for_map_poi' ;
        $bridgeParameters['url_ajax_send_email_recaptcha'] = $rootUrl . '/pylot_bridge/send_email_recaptcha' ;
        if($lang != 'fr') {
            $bridgeParameters['url_ajax_liste'] = $rootUrl . '/' . $lang . '/pylot_bridge/list_products_json' ;
            $bridgeParameters['url_ajax_liste_json_poi'] = $rootUrl . '/' .  $lang . '/pylot_bridge/list_products_json_for_map_poi' ;
            $bridgeParameters['url_ajax_send_email_recaptcha'] = $rootUrl . '/' .  $lang . '/pylot_bridge/send_email_recaptcha' ;
        }

        // Et la langue courante tant qu'on y est
        $bridgeParameters['lang'] = $lang;

        // Facebook App Id pour la social box
        $bridgeParameters['facebookAppId'] = '495036869019149';
        $bridgeParameters['facebookAppNonce'] = 'awkTZqCe';

        return $bridgeParameters;
    }

	/**
	 * @return mixed|string Pour la demande d'import en tâche de fond
	 */
	public function getNeedToImport()
	{
        return $this->configFactory->get('pylot_bridge.state')->get('need_to_import');
	}

	/**
	 * @return mixed|string Pour la demande d'import en tâche de fond
	 */
	public function setNeedToImport($val)
	{
        $this->configFactory->getEditable('pylot_bridge.state')->set('need_to_import', $val)->save();
        return 1;
	}

	public function getCvPreviewLink() {
        $config = \Drupal::config('pylot_bridge.bridgeconfig');
        return $config->get('cv_preview_page_url');
	}
	public function getCvDownloadLink() {
		return Url::fromRoute('pylot_bridge.download_carnet')->toString();
	}

	public function getLastImportLog() {

	}
	public function setLastImportLog($txt) {

	}

    /**
     * @return string URL du dossier des images de l'application
     */
    public function getBridgeImgUrl() {
        return $this->bridgeDirUrl() . '/public/img';
    }

    public function cachedFileGetContent(string $url, bool $use_include_path = false, $options = null, int $cacheTimeMinutes = 120) {
        return $this->requestService->cachedFileGetContent($url, $use_include_path, $options, $cacheTimeMinutes);
    }
}
