<?php

namespace Drupal\pylot_bridge\Services;

use Bridge\Weblibs\BridgeCmsAbstractLayerInterface;
use Drupal\Core\Url;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;



/**
 * Couche d'abstraction au CMS
 *
 *
 * @package    Plugin_SIT
 * @subpackage Plugin_SIT/includes
 * @author     DN Consultants <info@dnconsultants.fr>
 */

class BridgeCmsAbstractLayerDrupal implements BridgeCmsAbstractLayerInterface {

    /**
     * Stores assets to be attached to the page.
     * @var array
     */
    private $attached = ['library' => []];

    /**
     * Retourne vrai si le script a été enregistré / chargé
     * @param $handle : clé texte du script
     * @param $list : statut à vérifier : enqueued, registered
     * @return mixed
     */
    public function scriptIs($handle, $list ) {
        return true;
    }
    /**
     * Ajouter un script JS à la page courante
     * Si src est absent, le handle du script doit faire référence à un script préalablement déclaré avec registerScript
     * @param $handle : id texte du script
     * @param string $src : url du script
     * @param array $deps : tableau de dépendances (liste d'id texte)
     * @param false $ver : version
     * @param false $in_footer : charger dans le footer
     * @return mixed
     */
    public function registerScript($handle, $src = "", $deps = array(), $ver = false, $in_footer = false ) {
        // In Drupal, scripts are registered as part of libraries in .libraries.yml files.
        // This function can be a no-op, or used to dynamically define a library if needed,
        // but that's a more complex scenario. We assume libraries are predefined.
		return true;
    }

    /**
     * Ajouter un script JS à la page courante
     * Si src est absent, le handle du script doit faire référence à un script préalablement déclaré avec registerScript
     * @param $handle : id texte du script
     * @param string $src : url du script
     * @param array $deps : tableau de dépendances (liste d'id texte)
     * @param false $ver : version
     * @param false $in_footer : charger dans le footer
     * @return mixed
     */
    public function enqueueScript($handle, $src = "", $deps = array(), $ver = false, $in_footer = false ) {
		return true;
    }

    /**
     * Référencer une feuille de style CSS : associer un id texte à une URL
     * Si src est absent, le handle du css doit faire référence à un fichier préalablement déclaré avec registerStyle
     * @param $handle : id texte du css
     * @param string $src : url du css
     * @param array $deps : tableau de dépendances (liste d'id texte)
     * @param false $ver : version
     * @param string $media
     * @return mixed
     */
    public function registerStyle($handle, $src = "", $deps = array(), $ver = false, $media = 'all' ) {
        // In Drupal, styles are registered as part of libraries in .libraries.yml files.
        // This function can be a no-op.
		return true;
    }

    /**
     * Ajouter une feuille de style CSS à la page courante
     * Si src est absent, le handle du css doit faire référence à un fichier préalablement déclaré avec registerStyle
     * @param $handle : id texte du css
     * @param string $src : url du css
     * @param array $deps : tableau de dépendances (liste d'id texte)
     * @param false $ver : version
     * @param string $media
     * @return mixed
     */
    public function enqueueStyle($handle, $src = "", $deps = array(), $ver = false, $media = 'all' ) {
		return true;
	}

    /**
     * Returns the attached assets.
     * To be used in a controller to attach assets to a render array.
     * e.g., $build['#attached'] = $bridge->getAttachedAssets();
     * @return array
     */
    public function getAttachedAssets() {
        return $this->attached;
    }

    /**
     * @return string Retourne le code de la langue courante (2 lettres en minuscules / fr par défaut)
     */
    public function getCurrentLanguage()
    {
	    return \Drupal::languageManager()->getCurrentLanguage()->getId();
    }

    /**
     * @return string Retourne le code de la langue courante sous la forme fr-FR ou en-GB (langue - pays)
     */
    public function getCurrentLanguageLocale()
    {
        // Get current language code
        return \Drupal::languageManager()->getCurrentLanguage()->getId();
    }

    /**
     * Retourne le code de langue d'un post donné (wordpress)
     * @param int $postId numéro de post
     * @return string
     */
    public function getPostLanguage($postId)
    {
        // Aucun sens sous Drupal
        return $this->getCurrentLanguage();
    }

    /**
     * Définit le code de langue d'un post donné (surtout pour WordPress qui ne gère pas nativement le multilingue
     * @param $postId : id du post
     * @return string code de langue sur 2 lettres
     */
    public function setPostLanguage($postId, $lang)
    {
	    // In Drupal, this is handled during node creation/translation.
        // This function might not be directly applicable unless creating a new translation.
        return true;
    }

    /**
     * Sauvegarde les traductions d'un produit (WordPress)
     * @param array $translationsArray tableau associatif $lang => $postId ('fr' => 4)
     */
    public function savePostTranslations($translationsArray)
    {
	    // In Drupal, this is handled by the entity translation system.
        return true;
    }

    /**
     * Définit le code de langue d'un post donné (WordPress)
     * @param $termId : id du terme
     * @return bool
     */
    public function setTermLanguage($termId, $lang)
    {
	    // In Drupal, this is handled during term creation/translation.
        return true;
    }

    /**
     * Retourne la langue d'un terme (WordPress)
     * @param $termId : id du terme
     * @return string code de langue
     */
    public function getTermLanguage($termId)
    {
        // Aucun sens sous Drupal
        return $this->getCurrentLanguage();
    }

    /**
     * Retourne la langue d'un terme de taxonomie (WordPress)
     * ND 12/09/2022 : après recherche dans le code cette fonction n'est jamais utilisée
     * @param $termId : id du terme
     * @return array taleau associatif lang => id de terme
     */
    public function getTermTranslations($termId)
    {
        // Aucun sens en Drupal
        $trans = array();
        return $trans;
    }

    /**
     * Sauvegarde les traductions d'un terme (Wordpress)
     * @param array $translationsArray tableau associatif $lang => $termId ('fr' => 4)
     */
    public function saveTermTranslations($translationsArray)
    {
	    // In Drupal, this is handled by the entity translation system.
        return true;
    }

    /**
     * Retourne la liste des langues actives
     * @return string[] tableau de codes de langues
     */
    public function getLanguagesList()
    {
        $langs = array('fr');

        $langcodes = \Drupal::languageManager()->getLanguages();
        $langcodesList = array_keys($langcodes);

        // Patch 16/05/2022 : on doit s'assurer que le français est en premier
        if(!empty($langcodesList)) {
            if($langcodesList[0] !== 'fr') {
                $res = array('fr');
                foreach($langcodesList as $lang) {
                    if($lang !== 'fr') {
                        $res[] = $lang;
                    }
                }
                $langcodesList = $res;
            }
        }


        if(empty($langcodesList))
            return $langs;
        else
            return $langcodesList;
    }

    /**
     * Retourne le permalien associé à une weblist
     * @param $term_id : id de term lié à la lweblist
     * @param string $lang : langue en cours
     * @return string : Début de l'URL pour la réécriture des liens vers les fiches de la liste'
     */
    public function getPermalinkFromTermId($term_id, $lang = '')
    {
	    if (empty($lang)) {
		    $lang = $this->getCurrentLanguage();
        }
        $term = \Drupal\taxonomy\Entity\Term::load($term_id);
        if ($term) {
            if ($term->hasTranslation($lang)) {
                $term = $term->getTranslation($lang);
            }
            return Url::fromRoute('entity.taxonomy_term.canonical', ['taxonomy_term' => $term->id()], ['absolute' => true])->toString();
        }
        return '';
    }

    public function get_site_url() {
        return Url::fromRoute('<front>', [], ['absolute' => true])->toString();
    }

    public function getLanguagePrefix() {
        if($prefixes = \Drupal::config('language.negotiation')->get('url.prefixes')) {
            $language = \Drupal::languageManager()->getCurrentLanguage()->getId();

            if(!empty($prefixes[$language])) {
                return "/" . $prefixes[$language];
            } else {
                return '';
            }
        }
        return '';
    }
    public function getPostPermalink($postId) {
        // $post = \Drupal\node\Entity\Node::load($postId);
        if(!empty($postId)) {
            $path = '/node/'.$postId ;
            $alias_manager = $permalink = \Drupal::service('path_alias.manager') ;
            $permalink = \Drupal::service('path_alias.manager')->getAliasByPath('/node/'.$postId);

            // On ajoute le prefixe de langue si nécessaire
            if(empty($permalink))
                $permalink = $this->getLanguagePrefix() . $path;
            else
                $permalink = $this->getLanguagePrefix() . $permalink;

            // Correction double /
            if(mb_substr($permalink, 0, 2) == '//') {
                $permalink = substr($permalink, 1);
            }
        } else {
            $permalink = "#ErrorPermalink#PostNotFoundForPost#" . $postId;
        }
        if(!empty($permalink))
            return $permalink;

        return "#ErrorPermalink#";
    }

    /**
     * Retourne le header du theme front-end du site pour affichage avant le code de la fichede détail (WordPress)
     * @return mixed
     */
    public function getHeaderBridge($arg = null) {
	    // Not applicable in Drupal. Content is rendered in blocks or main content area.
		return '';
    }

    /**
     * Retourne le footer du theme front-end du site pour affichage après le code de la fichede détail (WordPress)
     * @return mixed
     */
    public function getFooterBridge() {
	    // Not applicable in Drupal.
		return '';
    }



    /**
     * Renvoie une reponse http Json d'erreur
     * @param mixed|null $data données à renvoyer en Json
     * @param int|null $status_code code HTTP à renvoyer
     * @param int $options options d'après les specifications de la fonction wp_send_json_error
     */
    public function sendJsonError( $data = null, int $status_code = null, int $options = 0) {
	    $response_data = ['success' => false, 'data' => $data];
        $status = $status_code ?? 500;
        $response = new JsonResponse($response_data, $status);
        $response->send();
		exit();
    }

    /**
     * Renvoie une reponse http Json de succes
     * @param mixed|null $data données à renvoyer en Json
     * @param int|null $status_code code HTTP à renvoyer
     * @param int $options options d'après les specifications de la fonction wp_send_json_success
     */
    public function sendJsonSuccess( $data = null, int $status_code = null, int $options = 0) {
	    $response_data = ['success' => true, 'data' => $data];
        $status = $status_code ?? 200;
        $response = new JsonResponse($response_data, $status);
        $response->send();
	    exit();
    }

    /**
     * Renvoie une reponse http Json de succes
     * @param mixed|null $data données à renvoyer en Json
     * @param int|null $status_code code HTTP à renvoyer
     * @param int $options options d'après les specifications de la fonction wp_send_json_success
     */
    public function sendJson( $data = null, int $status_code = null, int $options = 0) {
	    $status = $status_code ?? 200;
        $response = new JsonResponse($data, $status);
        $response->send();
	    exit();
    }

    /**
     * Permet de déclarer une fonction de callback pour insérer du contenu dans les vues Bridge sans avoir à les overrider
     * éuivalent à wordpress add_action
     * @param $hookName : nom du hook
     * @param $callBack : fonction de callback
     * @param int $priority : priorité
     * @param int $acceptedArgs : nombre d'arguments acceptés
     */
    public function addDisplayHook($hookName, $callBack, $priority = 10, $acceptedArgs = 1) {
	    // This is not the Drupal way. Hooks are implemented in .module files.
        // This function could be a proxy to Drupal's hook system, but it's complex.
        // For now, it's a no-op. Developers should use Drupal's hook system directly.
    }

    /**
     * Permet d'insérer un hook d'affichage dans les templates
     * @param string $hookName
     * @param mixed $args
     */
    public function doDisplayHook($hookName, ...$arg) {
	    \Drupal::moduleHandler()->invokeAll($hookName, $arg);
    }

     /**
     * Permet de déclarer une fonction de callback pour retrvailler du contenu dans les vues Bridge sans avoir à les overrider
     * éuivalent à wordpress apply_filter
     * @param $hookName : nom du hook
     * @param $callBack : fonction de callback
     * @param int $priority : priorité
     * @param int $acceptedArgs : nombre d'arguments acceptés
     */
    public function applyFilters($hookName, $callBack, $priority = 10, $acceptedArgs = 1) {
	    // Not the Drupal way. Use Drupal's hook_alter system.
		// This is a wrapper around doDisplayHook for filter-like behavior.
        $value = $callBack;
        \Drupal::moduleHandler()->alter($hookName, $value);
        return $value;
    }

    /**
     * Permet de déclarer une fonction de callback pour retrvailler du contenu dans les vues Bridge sans avoir à les overrider
     * éuivalent à wordpress apply_filter
     * @param $hookName : nom du hook
     * @param $callBack : fonction de callback
     * @param int $priority : priorité
     * @param int $acceptedArgs : nombre d'arguments acceptés
     */
    public function addFilter($hookName, $callBack, $priority = 10, $acceptedArgs = 1) {
	    // Not the Drupal way.
	}

    /**
     * @return false|mixed|string Chemin de base relatif à la réécriture des URL des fiches (hors permalien personnalisé)
     */
    public function getFicheRewritePattern()
    {
	    // In Drupal, this is handled by the routing system and path aliases.
	    return '';
    }

    /**
     * Fonction pour WordPress : indique si on se trouve actuellement sur une page de taxonomie SIT
     * @return bool True si on est sur une page de taxonomie SIT
     *
     */
    public function IsFicheSitTaxonomy(){
	    $route_match = \Drupal::routeMatch();
        if ($route_match->getRouteName() == 'entity.taxonomy_term.canonical') {
            $term = $route_match->getParameter('taxonomy_term');
            // Add logic to check if it's a "SIT" taxonomy, e.g., by vocabulary machine name.
            // if ($term->bundle() === 'sit_vocabulary') { return true; }
            return true; // Assuming any taxonomy page is a "SIT" taxonomy page for now.
        }
        return false;
    }

    /**
     * @return bool true si on est sur une page de fiche SIT
     */
    public function IsFicheSit() {
	    $route_match = \Drupal::routeMatch();
        if ($route_match->getRouteName() == 'entity.node.canonical') {
            $node = $route_match->getParameter('node');
            // Add logic to check if it's a "SIT" node, e.g., by content type machine name.
            // if ($node->bundle() === 'sit_content_type') { return true; }
            return true; // Assuming any node page is a "SIT" page for now.
        }
        return false;
    }

    /**
     * fonction appelée depuis un modèle de fiche
     * Retourne le code HTML du fil d'Ariane affiché dans les fiches
     * @param string $delimiter délimiteur d'entrées
     * @param string $home texte pour l'entrée 'Accueil'
     * @param int $showCurrent mettre 1 pour afficher l'entre courante
     * @param string $before texte à afficher avant l'élément courant
     * @param string $after texte à afficher après l'élément courant
     *
     * @return string code HTML du fil d'Ariane affiché dans les fiches
     */
    public function BridgeBreadcrumbs($delimiter = '&raquo;',$home = '', $showCurrent = 1,$before = '<span class="current">',$after = '</span>',$showOnHome = true)
    {
	    // Drupal handles breadcrumbs via its breadcrumb system.
        // You can implement a custom BreadcrumbBuilder service.
        // This function could render a breadcrumb block if needed.
        return '';
    }

    /* Permet de trier un tableau */

    public function bridge_array_sort($array, $on, $order=SORT_ASC)
    {
        return $array;
    }


	public function _n($singular, $plural, $number, $domain = 'default') {
		return \Drupal::translation()->formatPlural($number, $singular, $plural);
	}
	public function _nmp($keyStr, $singular, $plural, $number, $domain = 'default') {
		return \Drupal::translation()->formatPlural($number, $singular, $plural);
	}

	public function __( $text, $domain = 'default' ) {
		return t($text);
	}

	public function __mp($keyStr, $text, $domain = 'default') {
		// On ignore $text et domain qui ne concernent que Wordpress
		return t($keyStr);
	}



	public function esc_attr($text) {
		return htmlspecialchars($text, ENT_QUOTES, 'UTF-8');
	}
	public function esc_html($text) {
		return htmlspecialchars($text, ENT_QUOTES, 'UTF-8');
	}

	public function esc_js($text) {
		return htmlspecialchars($text, ENT_QUOTES, 'UTF-8');
	}

	public function sendMail($to, $subject, $message, $headers = '', $attachments = array())
	{
		if(empty($to) || empty($subject) || empty($message))
			return false;

        $mailManager = \Drupal::service('plugin.manager.mail');
        $module = 'pylot_bridge'; // Or your module name
        $key = 'pylot_bridge_mail'; // A key to identify the mail
        $langcode = \Drupal::languageManager()->getCurrentLanguage()->getId();
        $params = [
            'message' => $message,
            'subject' => $subject,
            'headers' => $headers,
        ];
        $send = true;

        $result = $mailManager->mail($module, $key, $to, $langcode, $params, null, $send);
        return $result['result'] === true;
	}

	public function send404Error($text = '') {
		throw new NotFoundHttpException($text);
	}
}
