<?php
namespace Drupal\pylot_bridge\Twig;

use Twig\Extension\AbstractExtension;
use Twig\TwigFilter;
use Twig\TwigFunction;

use Drupal\Core\Template\TwigEnvironment;

// use Drupal\pylot_bridge\Services\BridgeBlockService;
// use Drupal\pylot_bridge\Services\BridgeShortCodeParser;
// use Drupal\pylot_bridge\Services\BridgeUtils;
// use \DateUtils;


use Bridge\Weblibs\DateUtils;
use Bridge\Weblibs\BridgeApiService;
use Bridge\Weblibs\BridgeBlockService;
use Bridge\Weblibs\BridgeShortCodeParser;
use Bridge\Weblibs\BridgeDataGetter;
use Bridge\Weblibs\BridgeCmsAbstractLayerInterface;
use Bridge\Weblibs\BridgeClientParamsInterface;
use Bridge\Weblibs\BridgeClientContentInterface;
use Bridge\Weblibs\BridgeClientRendererInterface;


/**
 * Custom twig functions.
 */
class BridgeTwig extends AbstractExtension {

    public function __construct(
        private readonly BridgeBlockService $bridgeBlockService,
        private readonly BridgeShortCodeParser $bridgeShortCodeParser,
        private readonly BridgeCmsAbstractLayerInterface $bridgeCmsAbstractLayer,
        private readonly BridgeClientParamsInterface $bridgeClientParams,
        private readonly BridgeClientContentInterface $bridgeClientContent,
        private readonly BridgeClientRendererInterface $bridgeClientRenderer,
        private readonly BridgeDataGetter $bridgeDataGetter,
        private readonly BridgeApiService $bridgeApiService
    ) {
    }
    public function getName() {
        return 'pylot_bridge.twig_extension';
    }
    public function getFilters() {
        return [
            new TwigFilter('bridge_render_block', [$this, 'renderBlock']),
            new TwigFilter('bridge_render_named_block', [$this, 'renderNamedBlock']),
            new TwigFilter('bridge_render_reseaux_sociaux', [$this, 'renderReseauxSociaux']),
            new TwigFilter('bridge_extract_photos', [$this, 'extractPhotos']),
            new TwigFilter('bridge_extract_photos_from_block', [$this, 'extractPhotosFromBlock']),
            new TwigFilter('bridge_get_elevation_data', [$this, 'getElevationData']),
            new TwigFilter('bridge_get_product_dispos', [$this, 'getProductDispos']),
            new TwigFilter('bridge_get_child_colums', [$this, 'getChildColumns']),
            new TwigFilter('bridge_get_block_size', [$this, 'getBlockSize']),
            new TwigFilter('bridge_get_item_width', [$this, 'getItemWidth']),
            new TwigFilter('bridge_force_int', [$this, 'forceInt']),
            new TwigFilter('bridge_force_float', [$this, 'forceFloat']),
            new TwigFilter('prepare_custom_blocks_display', [$this, 'prepareCustomBlocksDisplay']),
            new TwigFilter('extract_first_text_value', [$this, 'extractBlockFirstTextValue']),
            new TwigFilter('extract_block_data_first_text_value', [$this, 'extractBlockDataFirstTextValue']),
            new TwigFilter('extract_first_item_label', [$this, 'extractBlockFirstItemLabel']),
            new TwigFilter('bridge_resize_image', [$this, 'bridgeResizeImage']),
            new TwigFilter('bridge_render_icon', [$this, 'bridgeRenderIcon']),
            new TwigFilter('bridge_get_coupled_products', [$this, 'getCoupledProducts']),
            new TwigFilter('bridge_product_permalink', [$this, 'getPermalinkFromProduct']),
            new TwigFilter('bridge_product_gmap_link', [$this, 'getgmapLinkFromProduct']),
            new TwigFilter('get_min_value_for_filter_item', [$this, 'getMinValueForfilterItem']),
            new TwigFilter('get_max_value_for_filter_item', [$this, 'getMaxValueForfilterItem']),
            new TwigFilter('get_block_component', [$this, 'getBlockComponent']),
            new TwigFilter('json_decode', [$this, 'jsonDecode']),
            new TwigFilter('cast_to_array', [$this, 'castToArray']),
            new TwigFilter('filter_modalities', [$this, 'filterModalities']),
            new TwigFilter('scroll_to_id', [$this, 'scrollTo'])
        ];
    }
    public function getFunctions() {
        return [
            new TwigFunction('bridge_include', [$this, 'bridgeInclude'], ['needs_environment' => true, 'needs_context' => true, 'is_safe' => ['all']]),
            new TwigFunction('items_block_is_not_empty', [$this, 'itemsBlockIsNotEmpty']),
            new TwigFunction('block_is_not_empty', [$this, 'blockIsNotEmpty']),
            new TwigFunction('text_block_is_not_empty', [$this, 'textBlockIsNotEmpty']),
            new TwigFunction('text_block_data_is_not_empty', [$this, 'textBlockDataIsNotEmpty']),
            new TwigFunction('bridge_is_object', [$this, 'bridgeIsObject']),
            new TwigFunction('get_fiche_hours_as_array_for_display_as_calendar', [$this, 'getFicheHoursAsArrayForDisplayAsCalendar']),
            new TwigFunction('get_fiche_hours_as_array_for_display_as_array', [$this, 'getFicheHoursAsArrayForDisplayAsArray']),
            new TwigFunction('prepareHoursForDisplay', [$this, 'prepareHoursForDisplay']),
            new TwigFunction('affiche_intervalle_dates', [$this, 'affiche_intervalle_dates']),
            new TwigFunction('generate_random_string', [$this, 'generateRandomString']),
            new TwigFunction('render_engine_image', [$this, 'renderEngineImage']),
            new TwigFunction('bridge_render_shortcode', [$this, 'renderShortCode']),
        ];
    }

    /*          FILTRES          */


    /**
     * Utilitaire pour trouver un critère depuis la priopriété modalities d'un objet de données product issu de Bridge
     *
     * @param       $mods : passer product->modalities tel quel
     * @param array $filters : tableau associatif (ex: {criterionCode: 9000000, modalityCode:123456 }
     *
     * @return array Tableau d'objets product->modalities filtré sur les critères de recherche
     */
    public function filterModalities($mods, $filters = array()) {
        $filters = (array) $filters ;
        return $this->bridgeBlockService->filterModalities($mods, $filters);
    }

    /**
     * @param string $url : url
     * @param string $section : section jusqu'a laquelle scroller
     *
     * @return string lien amenant a la section du site
     */
    public function scrollTo($url, $section = "") {
        // On vire les query string qui empeche le scroll
        if(str_contains($url, "?")) {
            $url = substr($url, 0, strpos($url, "?"));
        }

        return $url . "#$section";
    }

    /**
     * jsonDecode
     * Decode une chaine json
     */
    public function jsonDecode($str) {
        return json_decode($str);
    }

    /**
     * castToArray
     * Cast Object to Array
     */
    public function castToArray($obj) {
        $response = (array)$obj;
        return $response;
    }

    /**
     * getgmapLinkFromProduct
     * Retoune un lien d'itinéraire google maps
     * @param $fiche : objet product tel que retourné par l'API Bridge
     * @return string : URL d'itineraire google Maps
     */
    public function getgmapLinkFromProduct($fiche) {
        return $this->bridgeBlockService->getGmapLink($fiche);
    }

    /**
     * getBlockSize
     * Retoune une chaine à insérer dans la propriété class d'une balise HTML pour lui donner la largeur définie dans Bridge
     * contient les noms de classes css du framework uikit à insérer pour obtenir la laerrgeur responsive souhaitée
     * @param $blockData : objet block tel que retourné par l'API Bridge
     * @return string : chaine de classe CSS
     */
    public function getBlockSize($blockData) {
        return $this->bridgeBlockService->renderSizeBlock($blockData);
    }

    /**
     * getItemWidth
     * Retoune une chaine à insérer dans la propriété class d'une balise HTML pour lui donner la largeur définie dans Bridge
     * contient les noms de classes css du framework uikit à insérer pour obtenir la laerrgeur responsive souhaitée
     * @param $widthData : propriété width d'un objet  retourné par l'API Bridge (objet avec propriété xl, l, m, s et default)
     * @return string : chaine de classe CSS
     */
    public function getItemWidth($widthData) {
        return $this->bridgeBlockService->renderWidth($widthData);
    }

    /**
     * getChildColumns
     * Retoune une chaine à insérer dans la propriété class d'une balise HTML contenant des sous balises
     * contient les noms de classes css du framework uikit à insérer pour obtenir le colonnage responsive paramétré dans Bridge
     * @param $columsObject : objet columns tel que retourné par l'API Bridge
     * @return string : chaine de classe CSS
     */
    public function getChildColumns($columsObject) {
        return $this->bridgeBlockService->getChildColumns($columsObject);
    }

    /**
     * renderReseauxSociaux
     * Retoune une chaine html de rendu du block réseaux sociaux de fiche
     * @param $fiche : objet product tel que retourné par l'API Bridge
     * @return string : chaine de classe CSS
     */
    public function renderReseauxSociaux($fiche, $blockData) {
        return $this->bridgeBlockService->renderReseauxSociaux($fiche, $blockData);
    }

    /**
     * getProductDispos
     * Interroge l'API Bridge temps réel pour obtenir les infos de disponibilités d'une fiche
     * @param $fiche : objet product tel que retourné par l'API Bridge
     * @return mixed|null : données de dispos sous la forme d'un tableau d'objets
     */
    public function getProductDispos($fiche) {
        return $this->bridgeDataGetter->getProductDispos($fiche);
    }

    /**
     * getElevationData
     * Interroge l'API Bridge pour obtenir la courbe d'élévation d'une fiche
     * @param $fiche : objet product tel que retourné par l'API Bridge
     * @return mixed|void|null  : données d'altitude sous la forme d'un tableau d'objets
     */
    public function getElevationData($fiche) {
        return $this->bridgeDataGetter->getElevationData($fiche);
    }

    /**
     * Retoune l'URL de lien vers une fiche à partir de son productCode - tient copte de la langue courante
     * @param $product : objet product  issu de l'API Brodge
     * @return string URL de restination vers la fiche détaillée
     */
    public function getPermalinkFromProduct($product) {
        return $this->bridgeClientContent->getPostLinkFromProductCode($product->productCode);
    }

    /**
     * Retourne la valeur mini à prendre en compte pour un item de filtre Bridge selon le contexte
     * @param object $item objet filter item bridge
     */
    public function getMinValueForfilterItem($item, $moteur, $filter) {

        if (isset($_GET['brflt_'.$moteur->id.'-'.$filter->id.'_min']) && $_GET['brflt_'.$moteur->id.'-'.$filter->id.'_min'] != '')
            $min = $_GET['brflt_'.$moteur->id.'-'.$filter->id.'_min'];
        else
            $min = $item->min;
        return $min;
    }

    /**
     * Retourne la valeur mini à prendre en compte pour un item de filtre Bridge selon le contexte
     * @param object $item objet filter item bridge
     */
    public function getMaxValueForfilterItem($item, $moteur, $filter) {
        if (isset($_GET['brflt_'.$moteur->id.'-'.$filter->id.'_max']) && $_GET['brflt_'.$moteur->id.'-'.$filter->id.'_max'] != '')
            $min = $_GET['brflt_'.$moteur->id.'-'.$filter->id.'_max'];
        else
            $min = $item->max;
        return $min;
    }

    /**
     * Retourne une balise <i> d'afichage d'icône font awesome à partir d'un nom d'icône
     * @param string $icon nom de l'icône
     * @return string balise <i> pour font-awesome
     */
    public function bridgeRenderIcon($icon) {
        return $this->bridgeBlockService->renderIcon($icon);
    }

    /**
     * getCoupledProducts
     * Retourne un tableau d'objets products des fiches associées à l'objet product passé, enrichis des prioriété photo et shortComment
     * @param $fiche Objet prodct issu de l'API Bridge
     * @return array|false|mixed
     */
    public function getCoupledProducts($fiche) {
        $res = array();
        $coupled = $this->bridgeDataGetter->getCoupledProducts($fiche);
        if (isset($coupled) && isset($coupled->products) && is_array($coupled->products) && count($coupled->products) > 0) {
            // On récupère les critères photos via un bloc dédié à l'import des photos qu'on définit dans les paramètres du plugin
            for( $i = 0 ; $i < count($coupled->products) ; $i++) {
                $coupled->products[$i]->photo = '';
            }
            $blockPhoto = $this->bridgeClientParams->getIdBlockPhotos();
            if (!empty($blockPhoto)) {
                $blockContent = $this->bridgeApiService->callBridgeApi('GET', '/api/blocks/' . $blockPhoto);
                $criteresPhotos = $this->bridgeBlockService->extractBlockCriterions(json_decode(json_encode($blockContent)));

                for( $i = 0 ; $i < count($coupled->products) ; $i++) {
                    if (isset($criteresPhotos) && is_array($criteresPhotos) && count($criteresPhotos) > 0) {
                        $testImage = $this->bridgeBlockService->filterModalities($coupled->products[$i]->modalities, $criteresPhotos[0]);
                        if (is_array($testImage) && count($testImage) > 0 && isset($testImage[0]->value) && trim($testImage[0]->value) != '') {
                            $photo = "http://" . $testImage[0]->value;
                            $photo = str_replace("http://http://", "http://", $photo);
                            $photo = str_replace("http://https://", "https://", $photo);
                            $coupled->products[$i]->photo = $photo;
                        }
                    }
                    $coupled->products[$i]->shortComment = '';
                    if(!empty($coupled->products[$i]->comment)) {
                        if(mb_strlen($coupled->products[$i]->comment) > 150) {
                            $coupled->products[$i]->shortComment = mb_substr($coupled->products[$i]->comment,0,150) . '...';
                        } else {
                            $coupled->products[$i]->shortComment = $coupled->products[$i]->comment ;
                        }
                    } else {
                        $coupled->products[$i]->shortComment = '';
                    }
                }
            }
            $res = $coupled->products;
        }


        return $res;
    }

    /**
     * extractBlockFirstTextValue
     * Retourne la valeur texte du premier élément trouvé. Sert particulièrement pour extraire les composants de coordonnées
     * @param $blockData Données du block issues de l'API Bridge
     * @return string valeur texte extraite
     */
    public function extractBlockFirstTextValue($blocks_object, $block_name) {
        if(is_object($blocks_object) && property_exists($blocks_object, $block_name)) {
            return $this->bridgeBlockService->extractBlockFirstTextValue($blocks_object->$block_name);
        }
        return '';
    }
    /**
     * extractBlockDataFirstTextValue
     * Retourne la valeur texte du premier élément trouvé. Sert particulièrement pour extraire les composants de coordonnées
     * @param $blockData Données du block issues de l'API Bridge
     * @return string valeur texte extraite
     */
    public function extractBlockDataFirstTextValue($blockData) {
        return $this->bridgeBlockService->extractBlockFirstTextValue($blockData);
    }

    /**
     * extractBlockFirstTextValue
     * Retourne la valeur texte du premier élément trouvé. Sert particulièrement pour extraire les composants de coordonnées
     * @param $blockData Données du block issues de l'API Bridge
     * @return string valeur texte extraite
     */
    public function extractBlockFirstItemLabel($blocks_object, $block_name) {
        if(is_object($blocks_object) && property_exists($blocks_object, $block_name)) {
            return $this->bridgeBlockService->extractBlockFirstItemLabel($blocks_object->$block_name);
        }
        return '';
    }
    /**
     * Retourne le composant attaché au bloc s'il y en a et chaine vide sinon
     * @param $fiche : objet product retourné par l'API Bridge
     * @param $blockData : donnée block ou tableau blocks (si prop est fournie)
     * @return string nom du composant attaché au bloc ou chaine vide
     */
    public function getBlockComponent($fiche, $blockData, $prop='') {
        return $this->bridgeBlockService->getBlockComponent($fiche, $blockData, $prop);
    }
    /**
     * @param $blocks_object : tableau de blocks retourné dans l'objet product retourné par l'API Bridge
     * @param $block_name : nom de la sous-propriété de l'objet blocks_object
     * @param string $sectionTitleTag Nom de Balise HTML pour les titres de sections
     * @param string $sectionsSeparator Chaine à insérer entre les sections
     * @param string $sectionCSSClass Nom de classe CSS pour les balises de sections
     * @param string $itemsSeparator Chaine à insérer entre les items
     * @param string $itemCSSclass Nom de classe CSS pour les balises d'items
     * @param false $insertItemProps Faut-il insérer les props dans les attributs HTML des champs (itemprop : micro données)
     * @return string Code HTML d'affichage
     */
    public function renderNamedBlock($blocks_object, $block_name, $sectionTitleTag = 'h4', $sectionsSeparator='', $sectionCSSClass = '', $itemsSeparator = '', $itemCSSclass = '', $insertItemProps = false) {
        if(is_object($blocks_object) && property_exists($blocks_object, $block_name)) {
            return $this->bridgeBlockService->renderBlock($blocks_object->$block_name, $sectionTitleTag, $sectionsSeparator, $sectionCSSClass, $itemsSeparator, $itemCSSclass, $insertItemProps);
        }
        return '';
    }

    /**
     * renderBlock
     * Retourne le code HTML d'affichage d'un block Bridge
     * @param $renderBlockData Données du block issues de l'API Bridge
     * @param string $sectionTitleTag Nom de Balise HTML pour les titres de sections
     * @param string $sectionsSeparator Chaine à insérer entre les sections
     * @param string $sectionCSSClass Nom de classe CSS pour les balises de sections
     * @param string $itemsSeparator Chaine à insérer entre les items
     * @param string $itemCSSclass Nom de classe CSS pour les balises d'items
     * @param false $insertItemProps Faut-il insérer les props dans les attributs HTML des champs (itemprop : micro données)
     * @return string Code HTML d'affichage
     */
    public function renderBlock($renderBlockData, $sectionTitleTag = 'h4', $sectionsSeparator='', $sectionCSSClass = '', $itemsSeparator = '', $itemCSSclass = '', $insertItemProps = false) {
        return $this->bridgeBlockService->renderBlock($renderBlockData, $sectionTitleTag, $sectionsSeparator, $sectionCSSClass, $itemsSeparator, $itemCSSclass, $insertItemProps);
    }

    /**
     * hierarchicalBlocks
     * Renvoit un objet structuré permettant l'affichage des blocks personnalisés
     * @param $fiche
     * @return array
     */
    public function prepareCustomBlocksDisplay($fiche) {
        if(isset($fiche) && is_object($fiche) && !empty($fiche->ficheBlocks) && isset($fiche->ficheBlocks->custom) && count($fiche->ficheBlocks->custom) > 0) {

            return $this->bridgeBlockService->prepareCustomBlocksDisplay($fiche, $fiche->ficheBlocks->custom);
        } else {
            return array();
        }
    }

    /**
     * extractPhotos
     * Extrait les infos de photos d'un bloc dédié aux photos
     * @param stdClass $blocks : proprété blocks issue de l'objet product renvoyé par l'API Bridge (parent)
     * @param string $blockName : nom du block
     * @param int $maxPhotos : nom maximum de photos à extraire
     * @return array|string|string[]|null tableau de chaines d'URL de photos
     */
    public function extractPhotos($blocks, $blockName, $maxPhotos = 999) {
        $images = array();
        if(isset($blocks) && is_object($blocks) && isset($blocks->$blockName) && isset($blocks->$blockName->content) && is_object($blocks->$blockName->content) && is_array($blocks->$blockName->content->fields) && !empty($blocks->$blockName->content->fields) && isset($blocks->$blockName->content->fields[0]->items) && is_array($blocks->$blockName->content->fields[0]->items) && !empty($blocks->$blockName->content->fields[0]->items)) {
            foreach($blocks->$blockName->content->fields as $field) {
                foreach ($field->items as $item) {
                    $image = new \stdClass();
                    if(isset($item->subItems) && count($item->subItems) > 0) {
                        foreach($item->subItems as $subItem) {
                            $image->url = $subItem;
                            if (substr(strtolower($image->url), 0, 4) !== 'http') {
                                $image->url = 'http://' . $image->url;
                            }
                            $image->caption = '';
                            if ( isset($item) && is_object($item) && isset($item->extraData) && ! empty($item->extraData) ) {
                                foreach ($item->extraData as $extra)
                                    $image->caption .= $extra->value . ' ';
                            }
                            $images[] = $image;
                        }
                    } else {
                        $image->url = $item->textValue;
                        if (substr(strtolower($image->url), 0, 4) !== 'http') {
                            $image->url = 'http://' . $image->url;
                        }
                        $image->caption = '';
                        if ( isset($item) && is_object($item) && isset($item->extraData) && ! empty($item->extraData) ) {
                            foreach ($item->extraData as $extra)
                                $image->caption .= $extra->value . ' ';
                        }
                        $images[] = $image;
                    }
                }
            }
        }
        return $images;
    }
    /** Extrait les infos de photos d'un bloc dédié aux photos
     * @param stdClass $blockData : objet block issu de l'objet product renvoyé par l'API Bridge (parent)
     * @return array|string|string[]|null tableau de chaines d'URL de photos
     */
    public function extractPhotosFromBlock($blockData) {
        return $this->bridgeBlockService->extractPhotosFromBlock($blockData);
    }

    /**
     * bridgeResizeImage
     * Renvoit l'URL du service qui redimensionne une image dont l'URL originale est passée en paramètre
     * @param $file URL de la photo originale
     * @param string $mode mode de redimensonnement :
     * - ajust pour ajuster aux dimensions passées (ne déborde pas)
     * - remplir : pour remplir le cadre de dimensions passées
     * - deform : pour déformer l'image et forcer les dimensions exactes
     * @param string $selwidth largeur souhaitée
     * @param string $selheight hauteur souhaitée
     * @param string $quality qualité de compresion (sur 100 pour le jpg et de 0 à 9 pour le png)
     * @param int $timeToCache durée de cache navigateur à indiquer dans l'en-tête http
     * @return string l'URL du service qui affiche les données d'image redimentsionnée
     */
    public function bridgeResizeImage($url, $mode = "ajust", $width = "150", $height = "150", $quality = "60", $timeToCache = "1800") {
        if(!empty($url) && strlen($url) >= 4 && strtolower(substr($url, 0, 4)) != 'http')
            $url = 'http://' . $url;
        $baseUrl = \Drupal::request()->getSchemeAndHttpHost();
        $baseUrl .= $this->bridgeClientParams->getLanguagePrefix();
        return $baseUrl . "/pylot_bridge/resize_image?mode=$mode&selwidth=$width&selheight=$height&quality=$quality&timeToCache=$timeToCache&file=" . urlencode($url);
    }

    /*          FONCTIONS */

    /**
     * Produit une balise image destinée à la décoration des élements de moteurs de recherche
     * @param $urlImg
     * @param $imageTitle
     * @return string
     */
    function renderEngineImage($urlImg, $imageTitle='') {
        return $this->bridgeBlockService->renderEngineImage($urlImg, $imageTitle='');
    }
    /**
     * Renders a template.
     *
     * @param array        $context
     * @param string|array $template      The template to render or an array of templates to try consecutively
     * @param array        $variables     The variables to pass to the template
     * @param bool         $withContext
     * @param bool         $ignoreMissing Whether to ignore missing templates or not
     * @param bool         $sandboxed     Whether to sandbox the template or not
     *
     * @return string The rendered template
     */
    function bridgeInclude(TwigEnvironment $env, $context, $template, $variables = [], $withContext = true, $ignoreMissing = false, $sandboxed = false)
    {
        $alreadySandboxed = false;
        $sandbox = null;
        if ($withContext) {
            $variables = array_merge($context, $variables);
        }

        // \Drupal::moduleHandler()->invokeAll('bridge_twig_variables_alter', $variables, $template);
        // \Drupal::moduleHandler()->invokeAll('bridge_variables', $variables, $template);
        \Drupal::moduleHandler()->alter('bridgetwigvariables', $variables, $template);
        \Drupal::theme()->alter('bridgetwigvariables', $variables, $template);

        if ($isSandboxed = $sandboxed && $env->hasExtension(SandboxExtension::class)) {
            $sandbox = $env->getExtension(SandboxExtension::class);
            if (!$alreadySandboxed = $sandbox->isSandboxed()) {
                $sandbox->enableSandbox();
            }

            foreach ((\is_array($template) ? $template : [$template]) as $name) {
                // if a Template instance is passed, it might have been instantiated outside of a sandbox, check security
                if ($name instanceof TemplateWrapper || $name instanceof Template) {
                    $name->unwrap()->checkSecurity();
                }
            }
        }

        try {
            $loaded = null;
            try {
                $loaded = $env->resolveTemplate($template);
            } catch (LoaderError $e) {
                if (!$ignoreMissing) {
                    throw $e;
                }
            }

            return $loaded ? $loaded->render($variables) : '';
        } finally {
            if ($isSandboxed && !$alreadySandboxed) {
                $sandbox->disableSandbox();
            }
        }
    }

    /**
     * Génère une suite de lettres et chiffres pouvant servir d'identifiant unique
     * @param int $length longueur de la chaine à générer
     * @return string
     */
    public function generateRandomString($length = 10) {
        $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $charactersLength = strlen($characters);
        $randomString = '';
        for ($i = 0; $i < $length; $i++) {
            $randomString .= $characters[rand(0, $charactersLength - 1)];
        }
        return $randomString;
    }

    public function bridgeIsObject($var) {
        return is_object($var);
    }
    /**
     * Retourne true si le block nommé ne contient aucun item renseigné
     * @param $fiche : objet product retourné par l'API Bridge
     * @param $blocks : proprété blocks de l'objet product (parent du block qui nous intéresse)
     * @param string $blockName : nom du block
     * @return bool true s'il y a des items dans le block false sinon
     */
    public function itemsBlockIsNotEmpty($fiche, $blocks, $blockName) {
        if(isset($blocks) && isset($blocks->$blockName)) {
            return $this->blockIsNotEmpty($fiche, $blocks->$blockName);
        } else {
            return false;
        }
    }

    /**
     * Retourne true si le block ne contient aucune donnée (item ou gabarit  renseigné)
     * @param $fiche : objet product retourné par l'API Bridge
     * @param $blockData : block à tester (parent du block qui nous intéresse)
     * @return bool true s'il y a des items dans le block false sinon
     */
    public function blockIsNotEmpty($fiche, $blockData) {
        return $this->bridgeBlockService->blockIsNotEmpty($fiche, $blockData);
    }

    /**
     * Retourne true si le block nommé ne contient aucun item texte renseigné
     * @param $fiche : objet product retourné par l'API Bridge
     * @param $blocks : proprété blocks de l'objet product (parent du block qui nous intéresse)
     * @param $blockName : nom du block
     * @return bool true s'il y a des items texte dans le block false sinon
     */
    public function textBlockIsNotEmpty($fiche, $blocks, $blockName) {
        if(isset($blocks->$blockName) && isset($blocks->$blockName->content) && is_object($blocks->$blockName->content) && is_array($blocks->$blockName->content->fields) && !empty($blocks->$blockName->content->fields))
            return true;
        else
            return false;
    }

    /**
     * Retourne true si le block nommé ne contient aucun item texte renseigné
     * @param $fiche : objet product retourné par l'API Bridge
     * @param $blocks : donnée bloc de l'objet product
     * @return bool true s'il y a des items texte dans le block false sinon
     */
    public function textBlockDataIsNotEmpty($fiche, $blockData) {
        return $this->bridgeBlockService->textBlockDataIsNotEmpty($fiche, $blockData);
    }

    /**
     * Retourne un tableau contenant toutes les dates au jour le jour avec un statut et les infos d'horaires correspondant - utilisé pour l'affichage en calendrier
     * @param $fiche : objet product retourné par l'API Bridge
     * @param string $lang : langue d'affichage
     */
    public function getFicheHoursAsArrayForDisplayAsCalendar($fiche, $lang) {
        return DateUtils::getFicheHoursAsArrayForDisplayAsCalendar($fiche, $lang);
    }

    /**
     * Retourne un tableau contenant toutes les périodes et pour chacune tous les jours de la semaine avec les horaires de la fiche
     * @param $fiche : objet product retourné par l'API Bridge
     * @param string $lang : langue d'affichage
     */
    public function getFicheHoursAsArrayForDisplayAsArray($fiche, $lang) {
        return DateUtils::getFicheHoursAsArrayForDisplayAsArray($fiche, $lang);
    }


    /**
     * Cette Methode s'appelle pour extraire les dates / heures d'un product en situation de front end
     * Retourne un tableau d'horaires hebdomadaires correspndant à la période du / au fournie
     * Si mode_strict : on n'affiche que les heures de la periode exacte (adapté à un mode calendrier)
     * Si mode strict = false : on recherche tous les horaires ayant lieu dans l'intervalle du au (mode horaires compilés sans doublonner la fiche)
     * Cumuler : si true : on retourne tous les horaires cumulés sans se préoccuper des periodes )
     * @param \stdClass $fiche objet product tel que retourné par l'API Bridge
     * @param string|null $lang - langue d'affichage
     * @param array|null $options - options d'affichage
     *
     * @return array|mixed tableau de péiodes avec horaires hebdomadaires
     */
    public function prepareHoursForDisplay(\stdClass $fiche, ?string $lang = 'fr', $options = array()) {
        return DateUtils::prepareHoursForDisplay($fiche, $lang, $options);
    }

    /**
     * retourne une expression texte correspondant à un intervalle de dates(du .. au ... ou le ...)
     * datedu et dateau sont des chaines au format JJ/MM/AAAA
     *
     * @param string $datedu date du au format d/m/Y
     * @param string $dateau date au au format d/m/Y
     * @param bool $afficher_du_au_le
     * @param bool $en_clair
     * @param bool $afficher_joursem
     * @param bool $afficher_annee_debut
     * @param bool $afficher_annee_fin
     * @param bool $nomjour_abrege
     * @param bool $nommois_abrege
     *
     * @return string expresion calculéée
     */
    public function affiche_intervalle_dates(string $datedu, string $dateau, ?string $lang = 'fr', ?array $optionsPerso = array()) {
        return DateUtils::affiche_intervalle_dates($datedu, $dateau, $lang, $optionsPerso);
    }

    /**
     * Retourne le html de rendu d'un shortcode
     * @param $text : texte contenant un shortcode à rendre
     */
    public function renderShortCode($text) {
        $lang = $this->bridgeCmsAbstractLayer->getCurrentLanguage();
        $res = '';
        $tmp = $this->bridgeShortCodeParser->process($text, $lang);
        if(is_object($tmp) && isset($tmp->text) && !empty($tmp->text)) {
            $res = $tmp->text ;
            return $res;
        }
        return '';
    }

    /**
     * Force la valeur passée en paramètre à être un entier
     * @param string|int $value
     * @return int
     */
    function forceInt($value) {
        return (int) $value;
    }

    /**
     * Force la valeur passée en paramètre à être un float
     * @param string|float $value
     * @return float
     */
    function forceFloat($value) {
        return (float) $value;
    }

}
