<?php

namespace Drupal\Tests\qr_generator\Functional;

use Drupal\qr_generator\Entity\QRCode;
use Drupal\Tests\BrowserTestBase;

/**
 * Functional tests for the QR code REST resource.
 *
 * This test verifies the behavior of the `/api/qr-code/{uuid}` endpoint
 * provided by the qr_generator module. The endpoint is responsible for
 * looking up a QR code entity by UUID, checking its expiry, and either:
 *   - Redirecting to the configured target URL if the QR code is valid.
 *   - Returning HTTP 410 Gone if the QR code has expired.
 *   - Returning HTTP 404 Not Found if no matching QR code exists.
 *
 * Scenarios covered:
 * - QR code without expiry: should successfully redirect (final status 200
 *   when followed by BrowserTestBase).
 * - QR code with a past expiry timestamp: should return HTTP 410.
 * - Invalid UUID: should return HTTP 404.
 *
 * The test uses BrowserTestBase to simulate HTTP requests in a functional
 * environment with the qr_generator module enabled.
 *
 * @group qr_generator
 */
class QRRestResourceFunctionalTest extends BrowserTestBase
{
    protected static $modules = [
        'qr_generator',
        'link',
        'text'
    ];
    protected $defaultTheme = 'olivero';
    private $_qrcode_infinite;
    private $_qrcode_expired;

    /**
     * {@inheritdoc}
     */
    public function setUp(): void
    {
        parent::setUp();
        $this->_qrcode_infinite = QRCode::create([
            'label' => 'QR Code Infinite',
            'status' => TRUE,
            'field_redirect_url' => ['uri' => $this->buildUrl('/')],
            'field_expires' => ''
        ]);
        $this->_qrcode_infinite->save();
        $this->_qrcode_expired = QRCode::create([
            'label' => 'QR Code Expired',
            'status' => TRUE,
            'field_redirect_url' => ['uri' => $this->buildUrl('/')],
            'field_expires' => '2025-05-05T15:30:00'
        ]);
        $this->_qrcode_expired->save();
    }

    /**
     * Test the QR code without an expiry date.
     */
    public function testQRCodeInfinite()
    {
        $path = 'api/qr-code/' . $this->_qrcode_infinite->uuid();
        $url = $this->buildUrl($path);
        $this->drupalGet($url);
        $this->assertEquals(200, $this->getSession()->getStatusCode());
    }

    /**
     * Test the QR code with an expiry date.
     */
    public function testQRCodeExpired()
    {
        $path = 'api/qr-code/' . $this->_qrcode_expired->uuid();
        $url = $this->buildUrl($path);
        $this->drupalGet($url);
        $this->assertEquals(410, $this->getSession()->getStatusCode());
    }

    /**
     *  Test the QR code with the wrong ID.
     */
    public function testQRCodeWrongID()
    {
        $path = 'api/qr-code/wrongid';
        $url = $this->buildUrl($path);
        $this->drupalGet($url);
        $this->assertEquals(404, $this->getSession()->getStatusCode());
    }
}
