<?php

declare(strict_types=1);

namespace Drupal\qr_generator;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Link;
use Drupal\Core\Url;

/**
 * Provides a list controller for the qr code entity type.
 */
final class QRCodeListBuilder extends EntityListBuilder {

  /**
   * {@inheritdoc}
   */
  public function buildHeader(): array {
    $header['id'] = $this->t('ID');
    $header['label'] = $this->t('Label');
    $header['status'] = $this->t('Status');
    $header['uid'] = $this->t('Author');
    $header['created'] = $this->t('Created');
    $header['changed'] = $this->t('Updated');
    $header['expires'] = $this->t('Expires');
    $header['qr_code_url'] = $this->t('QR Code URL');
    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity): array {
    /** @var \Drupal\qr_generator\QRCodeInterface $entity */
    $row['id'] = $entity->id();
    $row['label'] = $entity->toLink();
    $row['status'] = $entity->get('status')->value ? $this->t('Enabled') : $this->t('Disabled');
    $username_options = [
      'label' => 'hidden',
      'settings' => ['link' => $entity->get('uid')->entity->isAuthenticated()],
    ];
    $row['uid']['data'] = $entity->get('uid')->view($username_options);
    $row['created']['data'] = $entity->get('created')->view(['label' => 'hidden']);
    $row['changed']['data'] = $entity->get('changed')->view(['label' => 'hidden']);
    $expires = $entity->get('field_expires')->getString();
    if($expires){
      $row['expires'] = \Drupal::service('date.formatter')->format((int)$entity->get('field_expires')->getString(), 'custom', 'd/m/Y');
    } else {
      $row['expires'] = $this->t('Infinite');
    }
    $qrCodeURL = Url::fromUri('internal:/api/qr-code/' . $entity->uuid())->setAbsolute();
    $link = Link::fromTextAndUrl('Test URL', $qrCodeURL)->toRenderable();
    $row['qr_code_url']['data'] = $link;
    return $row + parent::buildRow($entity);
  }

}
