<?php

namespace Drupal\query_auth_params\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\PageCache\ResponsePolicy\KillSwitch;
use Drupal\Core\Routing\TrustedRedirectResponse;
use Drupal\Core\Url;
use Drupal\path_alias\AliasManagerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\ControllerEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Query auth params event subscriber.
 */
class QueryAuthParamsSubscriber implements EventSubscriberInterface {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The page cache kill switch.
   *
   * @var \Drupal\Core\PageCache\ResponsePolicy\KillSwitch
   */
  protected $pageCacheKillSwitch;

  /**
   * The path alias manager.
   *
   * @var \Drupal\path_alias\AliasManagerInterface
   */
  protected $aliasManager;

  /**
   * Constructs event subscriber.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\PageCache\ResponsePolicy\KillSwitch $page_cache_kill_switch
   *   The page cache kill switch.
   * @param \Drupal\path_alias\AliasManagerInterface $alias_manager
   *   The path alias manager.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    KillSwitch $page_cache_kill_switch,
    AliasManagerInterface $alias_manager,
  ) {
    $this->configFactory = $config_factory;
    $this->pageCacheKillSwitch = $page_cache_kill_switch;
    $this->aliasManager = $alias_manager;
  }

  /**
   * Kernel controller event handler.
   *
   * @param \Symfony\Component\HttpKernel\Event\ControllerEvent $event
   *   Response event.
   */
  public function onKernelController(ControllerEvent $event) {
    $config_settings = $this->configFactory->get('query_auth_params.settings');
    $protect_pages = $config_settings->get('protect_pages');

    if (empty($protect_pages) || !is_array($protect_pages)) {
      return;
    }

    $request = $event->getRequest();
    $protect_page_exists = NULL;
    $current_path = $request->getPathInfo();

    // Get the system path (e.g., /node/203) from the current path.
    // If current path is an alias like /mathematics, this gets /node/203.
    // If current path is already a system path, it returns the same path.
    $current_system_path = $this->aliasManager->getPathByAlias($current_path);

    foreach ($protect_pages as $key => $protect_page) {
      if (!isset($protect_page['restrict_url'])) {
        continue;
      }

      $protected_path = $protect_page['restrict_url'];

      // Get the system path for the configured protected path.
      $protected_system_path = $this->aliasManager->getPathByAlias($protected_path);

      // Check if current path matches protected path.
      // Compare both the original paths and system paths.
      if ($current_path === $protected_path ||
          $current_path === $protected_system_path ||
          $current_system_path === $protected_path ||
          $current_system_path === $protected_system_path) {
        $protect_page_exists = $key;
        break;
      }
    }

    if (!isset($protect_page_exists)) {
      return;
    }

    // Kill page cache for protected pages to ensure logic runs every time.
    $this->pageCacheKillSwitch->trigger();

    $protect_page = $protect_pages[$protect_page_exists];
    $params = $request->query->all();

    // Ensure param_name and param_value exist.
    if (!isset($protect_page['param_name']) || !isset($protect_page['param_value'])) {
      return;
    }

    $param_name = $protect_page['param_name'];
    $param_value = $protect_page['param_value'];

    // Check if page should be protected based on settings.
    // Special handling for "once" - if already shown, always redirect.
    if ($protect_page['settings'] === 'once' && isset($protect_page['shown']) && $protect_page['shown'] === 1) {
      // Page was already shown once, redirect unconditionally.
      $redirect_url = !empty($protect_page['redirect_url']) ? $protect_page['redirect_url'] : '/';
      $url = Url::fromUserInput($redirect_url)->toString();
      $response = new TrustedRedirectResponse($url);
      $event->setController(function () use ($response) {
        return $response;
      });
      return;
    }

    // Check if datetime_period has passed.
    if ($protect_page['settings'] === 'datetime_period') {
      if (isset($protect_page['datetime_period'])) {
        $datetime_period = $protect_page['datetime_period'];
        // If current time has passed the datetime, don't protect anymore.
        if (time() > $datetime_period) {
          return;
        }
      }
    }

    // Check if correct params are provided.
    if (isset($params[$param_name]) && $params[$param_name] === $param_value) {
      // Params are correct - allow access.
      if ($protect_page['settings'] === 'once') {
        $editable_config = $this->configFactory->getEditable('query_auth_params.settings');
        $protect_pages[$protect_page_exists]['shown'] = 1;
        $editable_config->set('protect_pages', $protect_pages)->save();
      }

      return;
    }

    // Params missing or incorrect - redirect.
    $redirect_url = !empty($protect_page['redirect_url'])
      ? $protect_page['redirect_url']
      : '/';

    $url = Url::fromUserInput($redirect_url)->toString();
    $response = new TrustedRedirectResponse($url);
    $event->setController(function () use ($response) {
      return $response;
    });
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      KernelEvents::CONTROLLER => ['onKernelController'],
    ];
  }

}
