<?php

namespace Drupal\queue_import\Drush\Commands;

use Drupal\queue_import\Controller\DataSourceConnection;
use Drush\Commands\DrushCommands;

/**
 * A Drush commandfile.
 *
 * In addition to this file, you need a drush.services.yml
 * in root of your module, and a composer.json file that provides the name
 * of the services file to use.
 */
class QueueImportCommands extends DrushCommands {

  /**
   * Importing Drupal content.
   *
   * @command queue-import:drupal
   * @aliases d7
   * @options content_type Choose queue worker to use.
   * @options limit Choose how many to import.
   * @usage queue-import:drupal article
   *   Imports Drupal content.
   */
  public function importDrupalContent($content_type = 'node', $limit = 1) {
    $queue = $content_type . '_queue_processor';
    $queue = \Drupal::queue($queue);
    $queue->createQueue();

    $query = "select nid, type from node where type = '$content_type' order by nid desc limit $limit";
    $output = DataSourceConnection::DataSourceQuery($query);

    foreach ($output as $node) {
      $data = DataSourceConnection::GetDrupalFields($node['nid']);
      $data['type'] = $node['type'];
      $data['nid'] = $node['nid'];
      $queue->createItem($data);
      $this->output()->writeln("Queued legacy nid: " . $node['nid']);
    }
  }

  /**
   * Create queue worker field mapping.
   *
   * @command queue-import:field-map
   * @aliases d7fm
   * @options content_type Choose D8/D9 content type to map D7 fields against.
   * @usage queue-import:field-map
   *    Outputs queue worker field mapping for content type.
   */
  public function fieldMap($content_type) {
    $queue_worker_builder = DataSourceConnection::queueWorkerGenerator($content_type);

    $module_handler = \Drupal::service('module_handler');
    $module_path = $module_handler->getModule('queue_import')->getPath();
    $file = "$module_path/src/Plugin/QueueWorker/Map" . ucfirst($content_type) . "QueueProcessor.php";

    $fp = fopen($file, "wb");
    fwrite($fp, $queue_worker_builder);
    fclose($fp);
    $this->output->writeln("Field map: " . print_r($queue_worker_builder));
  }

  /**
   * Test importing articles.
   *
   * @command queue-import:test-article
   * @aliases itest
   * @usage queue-import:test-article
   *   Imports a test article.
   */
  public function importTest() {
    $data = [
      'title' => 'Greatest Import Test Ever',
      'body' => '<p>This will import to the node__body table as the body_value.</p>',
      'main_image' => 'https://thebikeshed.cc/wp-content/uploads/2014/07/Pacific-MC-CX-z.jpg',
      'tags' => ['cx500', 'cafe racer', 'refurbish', 'build'],
      'migration_id' => 1,
    ];

    $queue = \Drupal::queue('node_queue_processor');
    $queue->createQueue();
    $queue->createItem($data);
  }

  /**
   * Deletes or simply counts all nodes of specified content type.
   *
   * Note: Does not work with content_lock module enabled.
   *
   * @command queue-import:count-content
   * @aliases qicount
   * @options content_type Choose between articles, authors, etc.
   * @usage queue-import:count-content
   *   Runs count of all entities of the bundle authors.
   * @usage queue-import:count-content --delete
   *   Returns deletes all author node entities.
   */
  public function deleteNodes($content_type, $options = ['delete' => FALSE]) {
    if ($content_type !== 'image') {
      $result = \Drupal::entityQuery('node')
        ->condition('type', $content_type)
        ->execute();

      if ($options['delete']) {
        // entity_delete_multiple('node', $result);.
        $result = array_slice($result, 0, 1000);

        $storage_handler = \Drupal::entityTypeManager()->getStorage("node");
        $entities = $storage_handler->loadMultiple($result);
        $storage_handler->delete($entities);

        $this->output()->writeln('Deleted ' . count($result) . ' ' . $content_type . '(s).');
      }
      else {
        $this->output()->writeln('There are ' . count($result) . ' ' . $content_type . '(s) on the site.');
      }
    }
    else {
      $result = \Drupal::entityQuery('media')
        ->condition('bundle', $content_type)
        ->execute();

      if ($options['delete']) {
        // entity_delete_multiple('node', $result);.
        $result = array_slice($result, 0, 1000);

        $storage_handler = \Drupal::entityTypeManager()->getStorage("media");
        $entities = $storage_handler->loadMultiple($result);
        $storage_handler->delete($entities);

        $this->output()->writeln('Deleted ' . count($result) . ' ' . $content_type . '(s).');
      }
      else {
        $this->output()->writeln('There are ' . count($result) . ' ' . $content_type . '(s) on the site.');
      }
    }
  }

}
