# Queue Processor Module

A Drupal module that processes queues at the end of page requests, similar to how core's automated cron works.

## Features

- **Non-blocking processing**: Queues are processed after the response is sent to the user
- **Priority ordering**: Configure which queues should be processed first (0-100 scale)
- **Per-queue time limits**: Set individual time limits for each queue
- **Global time limit**: Overall maximum execution time to prevent blocking
- **Configurable item limits**: Control how many items to process per queue per request
- **Admin route filtering**: Optionally skip processing on admin pages
- **Queue UI integration**: Enhanced interface when Queue UI module is installed
- **Comprehensive logging**: Track processing statistics and errors
- **Hook system**: Allow other modules to alter queue configurations

## Requirements

- Drupal 10 or 11
- PHP 8.3+

## Optional Dependencies

- [Queue UI](https://www.drupal.org/project/queue_ui) - Provides enhanced queue management interface

## Installation

1. Place this module in your `modules/custom/` directory
2. Enable the module:
   ```bash
   drush en queue_processor -y
   ```

## Configuration

Visit `/admin/config/system/queue-processor` to configure:

1. **Enable automatic queue processing**: Turn the feature on/off
2. **Items per queue per request**: How many items to process (default: 10)
3. **Global maximum execution time**: Total time budget in seconds (default: 5)
4. **Run on admin routes**: Whether to process queues on /admin/* pages
5. **Queue Configuration**: Per-queue settings:
   - **Enable processing**: Toggle queue processing
   - **Priority**: 0-100 (lower = higher priority, default: 50)
   - **Time limit**: Per-queue time budget in seconds (0 = use remaining global time)

## Usage Examples

### Basic Configuration

Enable processing for email and search queues:
- Email notifications: Priority 10, Time limit 3 seconds
- Search indexing: Priority 70, Time limit 0 (use remaining)

### E-commerce Setup

```yaml
enabled: true
max_execution_time: 8
items_per_queue: 15
queues:
  - id: order_confirmation_email
    priority: 5
    time_limit: 2
    enabled: true
  - id: product_price_sync
    priority: 40
    time_limit: 0
    enabled: true
  - id: search_api_indexing
    priority: 70
    time_limit: 0
    enabled: true
```

## Priority Guidelines

| Range | Category | Examples |
|-------|----------|----------|
| 0-10 | Critical | Order confirmations, security alerts |
| 11-30 | High | Notifications, payment processing |
| 31-50 | Medium | Data syncs, API integrations |
| 51-70 | Low | Search indexing, analytics |
| 71-100 | Very Low | Image optimization, cleanup tasks |

## Time Limit Strategy

### Use specific time limits when:
- Queue has expensive operations (API calls, file processing)
- You want to guarantee time for critical queues
- Queue might have bursts that could consume all time

### Use 0 (remaining time) when:
- Queue operations are fast and predictable
- You want flexible time allocation
- Queue is lower priority and should only run if time permits

## API for Developers

### Hook: hook_queue_processor_queues_alter()

Modify queue configurations before processing:

```php
/**
 * Implements hook_queue_processor_queues_alter().
 */
function mymodule_queue_processor_queues_alter(array &$queues): void {
  // Boost email queue priority during business hours.
  foreach ($queues as &$queue) {
    if ($queue['id'] === 'email_queue' && date('H') >= 9 && date('H') <= 17) {
      $queue['priority'] = max(0, $queue['priority'] - 20);
    }
  }
}
```

### Programmatic Configuration

```php
use Drupal\Core\Config\ConfigFactoryInterface;

public function configureQueues(ConfigFactoryInterface $configFactory): void {
  $config = $configFactory->getEditable('queue_processor.settings');

  $queues = [
    [
      'id' => 'my_queue',
      'enabled' => TRUE,
      'priority' => 25,
      'time_limit' => 3,
    ],
  ];

  $config->set('queues', $queues)->save();
}
```

## Performance Considerations

### Traffic Recommendations

| Site Traffic | Max Time | Reasoning |
|--------------|----------|-----------|
| High (1000+ req/min) | 2-3 sec | Many processing opportunities |
| Medium (100-1000 req/min) | 5-7 sec | Balanced approach |
| Low (<100 req/min) | 10-15 sec | Fewer opportunities |

### Important Notes

- Processing occurs on **every eligible page request** (no interval)
- Best for **active sites** with steady traffic
- For guaranteed processing or very high-volume queues, use **dedicated cron**
- Monitor queue depths and adjust settings accordingly

## Logging

The module logs to the `queue_processor` channel:

- **Info**: Successful processing statistics
- **Warning**: Queue suspensions
- **Error**: Worker failures, queue loading issues
- **Debug**: Time limit hits, detailed processing info

View logs at `/admin/reports/dblog` and filter by type "queue_processor".

## Troubleshooting

### Queues not processing
1. Check that the module is enabled
2. Verify `enabled` is TRUE in configuration
3. Ensure queues are enabled individually
4. Check that traffic is hitting non-admin routes (if admin filtering is on)
5. Review logs for errors

### Processing too slow
1. Increase `max_execution_time` (carefully)
2. Adjust per-queue time limits
3. Consider using dedicated cron for heavy queues

### High-priority queue not processing
1. Check priority values (lower = higher priority)
2. Verify queue is enabled
3. Check if earlier queues are consuming all time
4. Review per-queue time limits

## License

GPL-2.0-or-later

## Maintainers

Created for Drupal 10/11 with modern PHP practices.
