<?php

declare(strict_types=1);

namespace Drupal\queue_processor\Form;

use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Queue\QueueWorkerManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for Queue Processor.
 */
class QueueProcessorSettingsForm extends ConfigFormBase {

  /**
   * Constructs a QueueProcessorSettingsForm.
   */
  public function __construct(
    protected readonly QueueWorkerManagerInterface $queueWorkerManager,
    protected readonly ModuleHandlerInterface $moduleHandler,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('plugin.manager.queue_worker'),
      $container->get('module_handler'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['queue_processor.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'queue_processor_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('queue_processor.settings');

    $form['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable automatic queue processing'),
      '#default_value' => $config->get('enabled'),
      '#description' => $this->t('Process queues at the end of page requests.'),
    ];

    $form['logging'] = [
      '#type' => 'select',
      '#title' => $this->t('Logging'),
      '#options' => [
        'errors' => $this->t('Errors and warnings only'),
        'summary' => $this->t('Queue processing summary'),
        'all' => $this->t('All'),
      ],
      '#default_value' => $config->get('logging'),
      '#description' => $this->t('Set level of logging for queue processing.'),
    ];

    $form['max_execution_time'] = [
      '#type' => 'number',
      '#title' => $this->t('Global maximum execution time (seconds)'),
      '#default_value' => $config->get('max_execution_time'),
      '#min' => 1,
      '#max' => 60,
      '#required' => TRUE,
      '#description' => $this->t('Maximum total time to spend processing all queues per request. Recommended: 5-10 seconds.'),
    ];

    $form['run_on_admin_routes'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Run on admin routes'),
      '#default_value' => $config->get('run_on_admin_routes'),
      '#description' => $this->t('If unchecked, queues will not be processed on /admin/* routes.'),
    ];

    // Queue configuration section.
    $form['queues'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Queue Configuration'),
      '#description' => $this->t('Configure which queues to process, their priority (lower number = higher priority), and individual time limits.'),
      '#tree' => TRUE,
    ];

    // Check if Queue UI module is enabled.
    $queueUiEnabled = $this->moduleHandler->moduleExists('queue_ui');

    if ($queueUiEnabled) {
      $form['queue_ui_link'] = [
        '#type' => 'item',
        '#markup' => $this->t('Manage queues using <a href="@url">Queue UI</a>.', [
          '@url' => '/admin/config/system/queue-ui',
        ]),
      ];
    }
    else {
      $form['queue_ui_note'] = [
        '#type' => 'item',
        '#markup' => $this->t('Install the <a href="@url" target="_blank">Queue UI</a> module for enhanced queue management capabilities.', [
          '@url' => 'https://www.drupal.org/project/queue_ui',
        ]),
      ];
    }

    // Get available queues.
    $availableQueues = $this->getAvailableQueues();
    $queueConfigs = $config->get('queues') ?? [];

    // Create a keyed array for easier access.
    $queueConfigsKeyed = [];
    foreach ($queueConfigs as $queueConfig) {
      $queueConfigsKeyed[$queueConfig['id']] = $queueConfig;
    }

    // Build form elements for each available queue.
    foreach ($availableQueues as $queueId => $queueLabel) {
      $existingConfig = $queueConfigsKeyed[$queueId] ?? [];

      $form['queues'][$queueId] = [
        '#type' => 'details',
        '#title' => $queueLabel,
        '#open' => !empty($existingConfig['enabled']),
      ];

      $form['queues'][$queueId]['enabled'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Enable processing'),
        '#default_value' => $existingConfig['enabled'] ?? FALSE,
      ];

      $form['queues'][$queueId]['priority'] = [
        '#type' => 'number',
        '#title' => $this->t('Priority'),
        '#default_value' => $existingConfig['priority'] ?? 50,
        '#min' => 0,
        '#max' => 100,
        '#description' => $this->t('Lower numbers are processed first (0 = highest priority, 100 = lowest priority).'),
        '#states' => [
          'visible' => [
            ':input[name="queues[' . $queueId . '][enabled]"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $form['queues'][$queueId]['time_limit'] = [
        '#type' => 'number',
        '#title' => $this->t('Time limit (seconds)'),
        '#default_value' => $existingConfig['time_limit'] ?? 0,
        '#min' => 0,
        '#max' => 60,
        '#description' => $this->t('Maximum time to spend on this queue per request. Set to 0 to use remaining global time.'),
        '#states' => [
          'visible' => [
            ':input[name="queues[' . $queueId . '][enabled]"]' => ['checked' => TRUE],
          ],
        ],
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $queuesConfig = [];
    $queuesInput = $form_state->getValue('queues') ?? [];

    foreach ($queuesInput as $queueId => $queueSettings) {
      // Only save enabled queues or queues with non-default settings.
      if (!empty($queueSettings['enabled'])) {
        $queuesConfig[] = [
          'id' => $queueId,
          'enabled' => TRUE,
          'priority' => (int) ($queueSettings['priority'] ?? 50),
          'time_limit' => (int) ($queueSettings['time_limit'] ?? 0),
        ];
      }
    }

    $this->config('queue_processor.settings')
      ->set('enabled', (bool) $form_state->getValue('enabled'))
      ->set('logging', $form_state->getValue('logging'))
      ->set('queues', $queuesConfig)
      ->set('max_execution_time', (int) $form_state->getValue('max_execution_time'))
      ->set('run_on_admin_routes', (bool) $form_state->getValue('run_on_admin_routes'))
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Gets available queue workers.
   *
   * @return array
   *   Array of queue names keyed by queue ID.
   */
  protected function getAvailableQueues(): array {
    $queues = [];
    $definitions = $this->queueWorkerManager->getDefinitions();

    foreach ($definitions as $id => $definition) {
      $label = $definition['title'] ?? $id;
      $queues[$id] = $this->t('@label (@id)', [
        '@label' => $label,
        '@id' => $id,
      ]);
    }

    ksort($queues);
    return $queues;
  }

}
