<?php

declare(strict_types=1);

namespace Drupal\queue_processor\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Logger\RfcLogLevel;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Queue\QueueWorkerManagerInterface;
use Drupal\Core\Queue\SuspendQueueException;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpKernel\Event\TerminateEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Processes queues at the end of requests.
 */
class QueueProcessorSubscriber implements EventSubscriberInterface {

  /**
   * Constructs a QueueProcessorSubscriber.
   */
  public function __construct(
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly QueueFactory $queueFactory,
    protected readonly QueueWorkerManagerInterface $queueWorkerManager,
    protected readonly LoggerChannelInterface $logger,
    protected readonly RequestStack $requestStack,
    protected readonly ModuleHandlerInterface $moduleHandler,
  ) {
    // No op.
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      KernelEvents::TERMINATE => ['onKernelTerminate', -100],
    ];
  }

  /**
   * Processes queues after response is sent.
   */
  public function onKernelTerminate(TerminateEvent $event): void {
    $config = $this->configFactory->get('queue_processor.settings');

    // Check if enabled.
    if (!$config->get('enabled')) {
      return;
    }

    // Check if we should run on admin routes.
    if (!$config->get('run_on_admin_routes')) {
      $request = $this->requestStack->getCurrentRequest();
      if ($request && str_starts_with($request->getPathInfo(), '/admin')) {
        return;
      }
    }

    // Get queue configurations and sort by priority.
    $queueConfigs = $config->get('queues') ?? [];
    $queueConfigs = $this->sortQueuesByPriority($queueConfigs);

    // Allow other modules to alter queue configurations.
    $this->moduleHandler->alter('queue_processor_queues', $queueConfigs);

    $maxExecutionTime = (int) ($config->get('max_execution_time') ?? 5);

    $globalStartTime = microtime(TRUE);
    $totalProcessed = 0;

    foreach ($queueConfigs as $queueConfig) {
      // Skip disabled queues.
      if (empty($queueConfig['enabled'])) {
        continue;
      }

      $queueId = $queueConfig['id'];

      // Check if we've exceeded global max execution time.
      $globalElapsed = microtime(TRUE) - $globalStartTime;
      if ($globalElapsed >= $maxExecutionTime) {
        $this->log(RfcLogLevel::DEBUG, 'Queue processing stopped: global maximum execution time (@max seconds) reached. Processed @total items total.', [
          '@max' => $maxExecutionTime,
          '@total' => $totalProcessed,
        ]);
        break;
      }

      // Determine time limit for this queue.
      $queueTimeLimit = $queueConfig['time_limit'] ?? 0;
      if ($queueTimeLimit <= 0) {
        // Use remaining global time.
        $queueTimeLimit = $maxExecutionTime - $globalElapsed;
      }
      else {
        // Use the lesser of per-queue limit and remaining global time.
        $remainingGlobalTime = $maxExecutionTime - $globalElapsed;
        $queueTimeLimit = min($queueTimeLimit, $remainingGlobalTime);
      }

      $processed = $this->processQueue($queueId, $queueTimeLimit);
      $totalProcessed += $processed;
    }

    if ($totalProcessed > 0) {
      $totalDuration = round(microtime(TRUE) - $globalStartTime, 2);
      $this->log(RfcLogLevel::INFO, 'Queue processing completed: @total items processed in @duration seconds.', [
        '@total' => $totalProcessed,
        '@duration' => $totalDuration,
      ]);
    }
  }

  /**
   * Sorts queue configurations by priority.
   *
   * @param array $queueConfigs
   *   Array of queue configurations.
   *
   * @return array
   *   Sorted array of queue configurations.
   */
  protected function sortQueuesByPriority(array $queueConfigs): array {
    usort($queueConfigs, function ($a, $b) {
      $priorityA = $a['priority'] ?? 50;
      $priorityB = $b['priority'] ?? 50;
      return $priorityA <=> $priorityB;
    });

    return $queueConfigs;
  }

  /**
   * Processes a single queue.
   *
   * @param string $queueName
   *   The queue name.
   * @param float $maxTime
   *   Maximum time to spend processing this queue.
   *
   * @return int
   *   Number of items processed.
   */
  protected function processQueue(string $queueName, float $maxTime): int {
    try {
      $queue = $this->queueFactory->get($queueName);
      $worker = $this->queueWorkerManager->createInstance($queueName);
    }
    catch (\Exception $e) {
      $this->log(RfcLogLevel::ERROR, 'Failed to load queue or worker for @queue: @message', [
        '@queue' => $queueName,
        '@message' => $e->getMessage(),
      ]);
      return 0;
    }

    $processed = 0;
    $startTime = microtime(TRUE);

    while (TRUE) {
      // Check if we've exceeded max time for this queue.
      $elapsed = microtime(TRUE) - $startTime;
      if ($elapsed >= $maxTime) {
        $this->log(RfcLogLevel::DEBUG, 'Queue @queue stopped: time limit (@limit seconds) reached after processing @count items.', [
          '@queue' => $queueName,
          '@limit' => round($maxTime, 2),
          '@count' => $processed,
        ]);
        break;
      }

      $item = $queue->claimItem();
      if (!$item) {
        break;
      }

      try {
        $worker->processItem($item->data);
        $queue->deleteItem($item);
        $processed++;
      }
      catch (SuspendQueueException $e) {
        $queue->releaseItem($item);
        $this->log(RfcLogLevel::WARNING, 'Queue @queue suspended: @message', [
          '@queue' => $queueName,
          '@message' => $e->getMessage(),
        ]);
        break;
      }
      catch (\Exception $e) {
        $queue->deleteItem($item);
        $this->log(RfcLogLevel::ERROR, 'Error processing queue item in @queue: @message', [
          '@queue' => $queueName,
          '@message' => $e->getMessage(),
        ]);
      }
    }

    if ($processed > 0) {
      $duration = round(microtime(TRUE) - $startTime, 2);
      $this->log(RfcLogLevel::DEBUG, 'Processed @count items from queue @queue in @duration seconds.', [
        '@count' => $processed,
        '@queue' => $queueName,
        '@duration' => $duration,
      ]);
    }

    return $processed;
  }

  /**
   * Logs messages pertaining to queue processing.
   *
   * @param mixed $level
   *   Log severity level.
   * @param string|\Stringable $message
   *   The log message.
   * @param array $context
   *   The message context parameters.
   */
  protected function log(mixed $level, string|\Stringable $message, array $context = []): void {
    $config = $this->configFactory->get('queue_processor.settings');
    $configured_logging = $config->get('logging');

    $log = match ($configured_logging) {
      'all' => TRUE,
      'summary' => $level <= RfcLogLevel::INFO,
      default => $level <= RfcLogLevel::WARNING,
    };

    if ($log) {
      $this->logger->log($level, $message, $context);
    }
  }

}
