/**
 * @file
 * JavaScript for Quick Node Status Toggler.
 * Uses vanilla JavaScript (no jQuery) for Drupal 10+ compatibility.
 */

(function (Drupal, drupalSettings) {
  'use strict';

  Drupal.behaviors.quickNodeStatusToggler = {
    attach: function (context, settings) {
      // Find all toggle checkboxes in the current context
      const toggles = context.querySelectorAll('.quick-status-toggle .toggle-checkbox');

      toggles.forEach(function (checkbox) {
        // Check if already processed
        if (checkbox.hasAttribute('data-toggle-processed')) {
          return;
        }

        // Mark as processed
        checkbox.setAttribute('data-toggle-processed', 'true');

        const wrapper = checkbox.closest('.quick-status-toggle');
        const label = wrapper.querySelector('.toggle-label');

        // Add change event listener
        checkbox.addEventListener('change', function () {
          const nodeId = wrapper.dataset.nodeId;
          const csrfToken = wrapper.dataset.csrfToken;
          const newStatus = checkbox.checked;

          // Disable the toggle while processing
          checkbox.disabled = true;
          wrapper.classList.add('processing');

          // Build URL with CSRF token as query parameter
          const baseUrl = Drupal.url('admin/content/toggle-status/' + nodeId);
          const url = baseUrl + '?token=' + csrfToken;

          // Make AJAX request using Fetch API
          fetch(url, {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
              'X-Requested-With': 'XMLHttpRequest'
            },
            credentials: 'same-origin'
          })
            .then(response => {
              if (!response.ok) {
                return response.text().then(text => {
                  throw new Error('HTTP error ' + response.status);
                });
              }
              return response.json();
            })
            .then(data => {
              if (data.success) {
                // Update the label text and class
                if (data.status) {
                  label.textContent = Drupal.t('Published');
                  label.classList.remove('unpublished');
                  label.classList.add('published');
                } else {
                  label.textContent = Drupal.t('Unpublished');
                  label.classList.remove('published');
                  label.classList.add('unpublished');
                }

                // Show success message
                showMessage(data.message, 'status');
              } else {
                // Revert the toggle on error
                checkbox.checked = !newStatus;

                // Show error message
                showMessage(data.message || Drupal.t('An error occurred.'), 'error');
              }
            })
            .catch(error => {
              // Revert the toggle on error
              checkbox.checked = !newStatus;

              // Show error message
              const errorMsg = Drupal.t('Failed to update node status. Please try again.');
              showMessage(errorMsg, 'error');
            })
            .finally(() => {
              // Re-enable the toggle
              checkbox.disabled = false;
              wrapper.classList.remove('processing');
            });
        });
      });
    }
  };

  /**
   * Helper function to show messages and clear old ones.
   */
  function showMessage(message, type) {
    if (typeof Drupal.Message !== 'undefined') {
      const messages = new Drupal.Message();
      // Clear all existing messages to prevent piling up
      messages.clear();
      messages.add(message, { type: type });
    } else {
      // Fallback: create a simple message div
      const messagesContainer = document.querySelector('.region-highlighted') || document.querySelector('main');
      if (messagesContainer) {
        // Clear any previous fallback messages
        const oldMessages = messagesContainer.querySelectorAll('.quick-toggle-message');
        oldMessages.forEach(m => m.remove());

        const messageDiv = document.createElement('div');
        messageDiv.className = 'messages messages--' + type + ' quick-toggle-message';
        messageDiv.setAttribute('role', 'contentinfo');
        messageDiv.setAttribute('aria-label', type === 'status' ? 'Status message' : 'Error message');

        const wrapper = document.createElement('div');
        wrapper.className = 'messages__wrapper layout-container';

        const content = document.createElement('div');
        content.className = 'messages__content';
        content.textContent = message;

        wrapper.appendChild(content);
        messageDiv.appendChild(wrapper);
        messagesContainer.insertBefore(messageDiv, messagesContainer.firstChild);

        // Auto-remove after 5 seconds with fade out
        setTimeout(() => {
          messageDiv.style.opacity = '0';
          messageDiv.style.transition = 'opacity 0.5s';
          setTimeout(() => messageDiv.remove(), 500);
        }, 5000);
      } else {
        // Ultimate fallback
        alert(message);
      }
    }
  }

})(Drupal, drupalSettings);
