<?php

namespace Drupal\quick_node_status_toggler\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\node\NodeInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for toggling node status.
 */
class NodeStatusToggleController extends ControllerBase
{

  /**
   * Toggles the published status of a node.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node entity.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with the new status.
   */
  public function toggle(NodeInterface $node, Request $request)
  {
    // Check if user has permission to edit this node.
    if (!$node->access('update')) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => $this->t('You do not have permission to edit this node.'),
      ], 403);
    }

    // Manual CSRF Token Validation.
    $token = $request->query->get('token');
    $path = '/admin/content/toggle-status/' . $node->id();

    if (!$token || !\Drupal::csrfToken()->validate($token, $path)) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => $this->t('Invalid security token. Please refresh the page and try again.'),
      ], 403);
    }

    // Toggle the status.
    $new_status = !$node->isPublished();

    // Set the status using both methods to ensure it works.
    $node->setPublished($new_status);
    $node->set('status', $new_status);

    // Don't create a new revision.
    $node->setNewRevision(FALSE);

    try {
      // Save the node.
      $node->save();

      // Clear the entity cache to ensure fresh data on next load.
      \Drupal::entityTypeManager()->getStorage('node')->resetCache([$node->id()]);

      return new JsonResponse([
        'success' => TRUE,
        'status' => $new_status,
        'message' => $new_status
          ? $this->t('Node published successfully.')
          : $this->t('Node unpublished successfully.'),
      ]);
    } catch (\Exception $e) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => $this->t('An error occurred while updating the node.'),
      ], 500);
    }
  }

}
