<?php

namespace Drupal\quick_node_status_toggler\Plugin\views\field;

use Drupal\Core\Access\CsrfTokenGenerator;
use Drupal\Core\Url;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Field handler to display quick status toggle.
 *
 * @ingroup views_field_handlers
 *
 * @ViewsField("quick_node_status_toggle_field")
 */
class QuickNodeStatusToggleField extends FieldPluginBase
{

    /**
     * The CSRF token generator.
     *
     * @var \Drupal\Core\Access\CsrfTokenGenerator
     */
    protected $csrfToken;

    /**
     * Constructs a QuickNodeStatusToggleField object.
     *
     * @param array $configuration
     *   A configuration array containing information about the plugin instance.
     * @param string $plugin_id
     *   The plugin_id for the plugin instance.
     * @param mixed $plugin_definition
     *   The plugin implementation definition.
     * @param \Drupal\Core\Access\CsrfTokenGenerator $csrf_token
     *   The CSRF token generator.
     */
    public function __construct(array $configuration, $plugin_id, $plugin_definition, CsrfTokenGenerator $csrf_token)
    {
        parent::__construct($configuration, $plugin_id, $plugin_definition);
        $this->csrfToken = $csrf_token;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition)
    {
        return new static(
            $configuration,
            $plugin_id,
            $plugin_definition,
            $container->get('csrf_token')
        );
    }

    /**
     * {@inheritdoc}
     */
    public function query()
    {
        // Leave empty to avoid a query on this field.
    }

    /**
     * {@inheritdoc}
     */
    public function render(ResultRow $values)
    {
        $node = $this->getEntity($values);

        if (!$node || !$node->access('update')) {
            return [];
        }

        // Generate CSRF token for the specific route.
        // Explicitly using the leading slash which is required for _csrf_token validation.
        $path = '/admin/content/toggle-status/' . $node->id();
        $token = $this->csrfToken->get($path);

        $build = [
            '#theme' => 'quick_node_status_toggle',
            '#node_id' => $node->id(),
            '#status' => $node->isPublished(),
            '#csrf_token' => $token,
        ];

        // Attach the library at the top level of the render array.
        $build['#attached']['library'][] = 'quick_node_status_toggler/toggle';

        return $build;
    }

}
