<?php

namespace Drupal\quiz_commerce\EventSubscriber;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\Url;
use Drupal\quiz\Entity\Quiz;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Subscribes to the kernel request event to control access to quiz taking.
 */
class QuizTakeAccessSubscriber implements EventSubscriberInterface {

  use StringTranslationTrait;

  /**
   * The account manager.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * QuizTakeAccessSubscriber constructor.
   *
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   Cart provider.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger manager.
   * @param \Drupal\Core\StringTranslation\StringTranslationTrait $string_translation
   *   The translation string.
   */
  public function __construct(AccountProxyInterface $current_user, EntityTypeManagerInterface $entity_type_manager, MessengerInterface $messenger, TranslationInterface $string_translation) {
    $this->currentUser = $current_user;
    $this->entityTypeManager = $entity_type_manager;
    $this->messenger = $messenger;
    $this->stringTranslation = $string_translation;
  }

  /**
   * {@inheritdoc}
   *
   * @return array
   *   The event names to listen for, and the methods that should be executed.
   */
  public static function getSubscribedEvents() {
    return [
      KernelEvents::REQUEST => ['getAccessToQuiz'],
    ];
  }

  /**
   * Access to take quiz when request.
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   Our custom event object.
   */
  public function getAccessToQuiz(RequestEvent $event) {
    $request = $event->getRequest();
    $route_name = $request->attributes->get('_route');

    // Only trigger for specific route.
    if ($route_name !== 'entity.quiz.take') {
      return;
    }

    $quiz = $request->attributes->get('quiz');
    if (!$quiz || !$quiz instanceof Quiz) {
      return;
    }

    // Skip for admin user.
    if ($this->currentUser->hasPermission('administer quiz')) {
      return;
    }

    $uid = $this->currentUser->id();

    // Add your custom logic here: check if user has paid for the quiz.
    $is_bought = quiz_commerce_check_if_quiz_bought($quiz->id(), $uid);

    if (!$is_bought) {
      $this->messenger->addMessage($this->t('Quiz @quiz is paid. You are able to take quiz after purchase', [
        '@quiz' => $quiz->label(),
      ]));
      $url = new RedirectResponse(
        Url::fromRoute(
          'entity.quiz.canonical',
          ['quiz' => $quiz->id()]
        )->toString()
      );
      $event->setResponse($url);
      return;
    }
  }

}
