<?php

namespace Drupal\rail_ai_provider\Form;

use Drupal\ai\AiProviderPluginManager;
use Drupal\ai\Service\AiProviderFormHelper;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Rail access.
 */
class RailConfigForm extends ConfigFormBase {

  /**
   * The form helper.
   *
   * @var \Drupal\ai\Service\AiProviderFormHelper
   */
  protected $formHelper;

  /**
   * The AI Provider manager.
   *
   * @var \Drupal\ai\AiProviderPluginManager
   */
  protected $providerManager;

  /**
   * Constructs a new RailConfigForm object.
   */
  final public function __construct(AiProviderFormHelper $form_helper, AiProviderPluginManager $provider_manager) {
    $this->formHelper = $form_helper;
    $this->providerManager = $provider_manager;
  }

  /**
   * {@inheritdoc}
   */
  final public static function create(ContainerInterface $container) {
    return new static(
      $container->get('ai.form_helper'),
      $container->get('ai.provider'),
    );
  }

  /**
   * Config settings.
   */
  const CONFIG_NAME = 'rail_ai_provider.settings';

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'rail_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      static::CONFIG_NAME,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config(static::CONFIG_NAME);

    $form['api_key'] = [
      '#type' => 'key_select',
      '#title' => $this->t('Rail Access Token'),
      '#default_value' => $config->get('api_key'),
    ];

    $provider = $this->providerManager->createInstance('rail_ai');
    $form['models'] = $this->formHelper->getModelsTable($form, $form_state, $provider);

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $models = [];
    foreach ($form_state->getValues() as $key => $value) {
      if (substr($key, 0, 7) == 'model__' && $value) {
        $parts = explode('__', $key);
        $models[$parts[1]][] = $value;
      }
    }

    // Retrieve the configuration.
    $this->config(static::CONFIG_NAME)
      ->set('api_key', $form_state->getValue('api_key'))
      ->set('models', $models)
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Add more model.
   */
  public function addMoreModel(array &$form, FormStateInterface $form_state) {
    $models = [];
    foreach ($form_state->getValues() as $key => $value) {
      if (substr($key, 0, 7) == 'model__' && $value) {
        $parts = explode('__', $key);
        $models[$parts[1]][] = $value;
      }
    }
    $form_state->set('models', $models);
    $form_state->setRebuild();
  }

  /**
   * Add more model callback.
   */
  public function addMoreModelCallback(array &$form, FormStateInterface $form_state) {
    $trigger = $form_state->getTriggeringElement();
    $type = $trigger['#attributes']['data-type'];
    return $form[$type];
  }

}
