<?php

declare(strict_types=1);

namespace Drupal\rail_ai_provider\Plugin\AiGuardrail;

use Drupal\ai\Attribute\AiGuardrail;
use Drupal\ai\Guardrail\AiGuardrailPluginBase;
use Drupal\ai\Guardrail\Result\GuardrailResultInterface;
use Drupal\ai\Guardrail\Result\PassResult;
use Drupal\ai\Guardrail\Result\RewriteInputResult;
use Drupal\ai\Guardrail\Result\RewriteOutputResult;
use Drupal\ai\Guardrail\Result\StopResult;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\ai\OperationType\Chat\ChatOutput;
use Drupal\ai\OperationType\InputInterface;
use Drupal\ai\OperationType\OutputInterface;
use Drupal\Component\Plugin\ConfigurableInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\key\KeyRepositoryInterface;
use Drupal\rail_ai_provider\RailEthicalEvaluationService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the RAIL Ethical Dimensions guardrail.
 */
#[AiGuardrail(
  id: 'rail_ethical_dimensions',
  label: new TranslatableMarkup('RAIL Ethical Dimensions'),
  description: new TranslatableMarkup(
    "Evaluates content across RAIL's 8 dimensions of ethical AI with configurable weights and thresholds."
  ),
)]
class RailEthicalDimensionsGuardrail extends AiGuardrailPluginBase implements ConfigurableInterface, PluginFormInterface, ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  /**
   * The RAIL ethical evaluation service.
   *
   * @var \Drupal\rail_ai_provider\RailEthicalEvaluationService
   */
  protected RailEthicalEvaluationService $railEvaluationService;

  /**
   * The RAIL guardrail monitoring service.
   *
   * @var \Drupal\rail_ai_provider\RailGuardrailMonitoringService
   */
  protected $monitoringService;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * The key repository service.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected KeyRepositoryInterface $keyRepository;

  /**
   * The 8 ethical dimensions supported by RAIL.
   */
  public const ETHICAL_DIMENSIONS = [
    'fairness' => 'Fairness',
    'safety' => 'Safety',
    'reliability' => 'Reliability',
    'transparency' => 'Transparency',
    'privacy' => 'Privacy',
    'accountability' => 'Accountability',
    'inclusivity' => 'Inclusivity',
    'user_impact' => 'User Impact',
  ];

  /**
   * Critical dimensions that always stop processing on violation.
   */
  public const CRITICAL_DIMENSIONS = ['safety', 'privacy'];

  /**
   * Constructs a RailEthicalDimensionsGuardrail object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\rail_ai_provider\RailEthicalEvaluationService $rail_evaluation_service
   *   The RAIL ethical evaluation service.
   * @param \Drupal\rail_ai_provider\RailGuardrailMonitoringService $monitoring_service
   *   The RAIL guardrail monitoring service.
   * @param \Drupal\key\KeyRepositoryInterface $key_repository
   *   The key repository service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    RailEthicalEvaluationService $rail_evaluation_service,
    $monitoring_service = null,
    KeyRepositoryInterface $key_repository = null
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->railEvaluationService = $rail_evaluation_service;
    $this->monitoringService = $monitoring_service ?: \Drupal::service('rail_ai_provider.monitoring');
    $this->logger = \Drupal::logger('rail_ai_provider');
    $this->keyRepository = $key_repository ?: \Drupal::service('key.repository');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('rail_ai_provider.ethical_evaluation'),
      $container->get('rail_ai_provider.monitoring'),
      $container->get('key.repository')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function isAvailable(): bool {
    // Check if required services and dependencies are available
    // Don't check configuration here as that happens after plugin selection
    return $this->railEvaluationService !== NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function processInput(InputInterface $input): GuardrailResultInterface {
    // Only process if evaluation mode includes input
    $evaluation_mode = $this->configuration['evaluation_mode'] ?? 'both';
    if (!in_array($evaluation_mode, ['input', 'both'])) {
      return new PassResult('Input evaluation disabled in configuration.', $this);
    }

    $text = $this->extractTextFromInput($input);
    if (empty($text)) {
      return new PassResult('No text content found to evaluate.', $this);
    }

    return $this->evaluateContent($text, 'input');
  }

  /**
   * {@inheritdoc}
   */
  public function processOutput(OutputInterface $output): GuardrailResultInterface {
    // Only process if evaluation mode includes output
    $evaluation_mode = $this->configuration['evaluation_mode'] ?? 'both';
    if (!in_array($evaluation_mode, ['output', 'both'])) {
      return new PassResult('Output evaluation disabled in configuration.', $this);
    }

    $text = $this->extractTextFromOutput($output);
    if (empty($text)) {
      return new PassResult('No text content found to evaluate.', $this);
    }

    return $this->evaluateContent($text, 'output');
  }

  /**
   * Extracts text content from input with advanced content handling.
   *
   * This method implements comprehensive input processing logic that:
   * - Handles ChatInput messages appropriately by extracting the most recent user message
   * - Supports multi-message conversation evaluation with context awareness
   * - Gracefully handles various input types with appropriate logging
   * - Handles edge cases (empty content, malformed input) with robust error handling
   * - Prepares content for RAIL API evaluation by cleaning and validating text
   * - Supports different content types through extensible type detection
   * - Implements graceful degradation for unsupported inputs
   *
   * @param \Drupal\ai\OperationType\InputInterface $input
   *   The input to extract text from.
   *
   * @return string
   *   The extracted text content, cleaned and ready for evaluation.
   */
  protected function extractTextFromInput(InputInterface $input): string {
    // Handle different input types with graceful degradation
    if (!$input instanceof ChatInput) {
      return $this->handleUnsupportedInputType($input);
    }

    $messages = $input->getMessages();
    
    // Handle edge case: empty messages
    if (empty($messages)) {
      $this->logger->debug('RAIL guardrail: No messages found in ChatInput.');
      return '';
    }

    // Support multi-message conversation evaluation with context awareness
    $extracted_text = $this->extractTextFromConversation($messages);
    
    if (empty($extracted_text)) {
      $this->logger->debug('RAIL guardrail: No meaningful text content found after conversation processing.');
      return '';
    }

    // Prepare content for RAIL API evaluation by cleaning and validating
    $cleaned_text = $this->prepareContentForEvaluation($extracted_text);
    
    if (empty($cleaned_text)) {
      $this->logger->debug('RAIL guardrail: No meaningful text content found after cleaning.');
      return '';
    }

    $this->logger->debug('RAIL guardrail: Extracted @length characters from input for evaluation.', [
      '@length' => strlen($cleaned_text),
    ]);

    return $cleaned_text;
  }

  /**
   * Extracts text content from output with advanced content handling.
   *
   * This method implements comprehensive output processing logic that:
   * - Extracts text from AI-generated responses (ChatOutput)
   * - Handles streamed content reconstruction by collecting all text chunks
   * - Handles edge cases (empty content, malformed output) with robust error handling
   * - Applies same evaluation logic as input processing with content preparation
   * - Supports different evaluation modes through configuration checks
   * - Supports different content types through extensible type detection
   * - Gracefully handles various output types with appropriate logging
   * - Implements graceful degradation for unsupported outputs
   *
   * @param \Drupal\ai\OperationType\OutputInterface $output
   *   The output to extract text from.
   *
   * @return string
   *   The extracted text content, cleaned and ready for evaluation.
   */
  protected function extractTextFromOutput(OutputInterface $output): string {
    // Handle different output types with graceful degradation
    if (!$output instanceof ChatOutput) {
      return $this->handleUnsupportedOutputType($output);
    }

    // Handle edge cases and extract text with robust error handling
    $text = $this->extractTextFromChatOutput($output);
    
    if (empty($text)) {
      $this->logger->debug('RAIL guardrail: No meaningful text content found in output after extraction.');
      return '';
    }

    // Apply same evaluation logic as input processing with content preparation
    $cleaned_text = $this->prepareContentForEvaluation($text);
    
    if (empty($cleaned_text)) {
      $this->logger->debug('RAIL guardrail: No meaningful text content found in output after cleaning.');
      return '';
    }

    return $cleaned_text;
  }

  /**
   * Reconstructs complete content from streamed output.
   *
   * This method handles streamed content reconstruction by collecting
   * all available text chunks and combining them into complete content
   * for ethical evaluation.
   *
   * @param \Drupal\ai\OperationType\Chat\ChatOutput $output
   *   The streamed ChatOutput to reconstruct.
   *
   * @return string
   *   The reconstructed complete text content.
   */
  protected function reconstructStreamedContent(ChatOutput $output): string {
    $complete_text = '';
    
    try {
      // If the output has a method to get all chunks, use it
      if (method_exists($output, 'getAllChunks')) {
        $chunks = $output->getAllChunks();
        foreach ($chunks as $chunk) {
          if (is_string($chunk)) {
            $complete_text .= $chunk;
          } elseif (is_array($chunk) && isset($chunk['text'])) {
            $complete_text .= $chunk['text'];
          } elseif (method_exists($chunk, 'getText')) {
            $complete_text .= $chunk->getText();
          }
        }
      } else {
        // Fallback: try to get the complete text from normalized message
        $normalized = $output->getNormalized();
        if ($normalized instanceof ChatMessage) {
          $complete_text = $normalized->getText();
        }
      }
      
      $this->logger->debug('RAIL guardrail: Successfully reconstructed streamed content.');
      
    } catch (\Exception $e) {
      $this->logger->warning('RAIL guardrail: Failed to reconstruct streamed content: @message', [
        '@message' => $e->getMessage(),
      ]);
      
      // Fallback to whatever text is available
      try {
        $normalized = $output->getNormalized();
        if ($normalized instanceof ChatMessage) {
          $complete_text = $normalized->getText();
        }
      } catch (\Exception $fallback_e) {
        $this->logger->error('RAIL guardrail: Complete failure to extract text from streamed output: @message', [
          '@message' => $fallback_e->getMessage(),
        ]);
        return '';
      }
    }

    return $complete_text;
  }

  /**
   * Prepares content for RAIL API evaluation with enhanced security and validation.
   *
   * This method cleans and validates text content before sending it
   * to the RAIL API for ethical evaluation. It handles:
   * - Text normalization and cleaning
   * - Removal of excessive whitespace
   * - Basic content validation
   * - Length limits and truncation if necessary
   * - Security sanitization to prevent injection attacks
   * - Edge case handling (empty content, malformed input)
   * - Content type detection and appropriate processing
   *
   * @param string $text
   *   The raw text content to prepare.
   *
   * @return string
   *   The cleaned and prepared text content.
   */
  protected function prepareContentForEvaluation(string $text): string {
    if (empty($text)) {
      return '';
    }

    // Handle edge case: extremely long content
    if (strlen($text) > 200000) { // 200KB limit
      $this->logger->warning('RAIL guardrail: Content extremely large (@size bytes), truncating for evaluation.', [
        '@size' => strlen($text),
      ]);
      $text = substr($text, 0, 200000);
    }

    // Security: Remove null bytes and other dangerous characters
    $cleaned = str_replace(["\0", "\x0B", "\x0C"], '', $text);
    
    // Remove other control characters except common whitespace
    $cleaned = preg_replace('/[\x00-\x08\x0E-\x1F\x7F]/', '', $cleaned);

    // Handle edge case: content with only control characters
    if (empty(trim($cleaned))) {
      return '';
    }

    // Normalize whitespace and remove excessive spacing
    $cleaned = preg_replace('/\s+/', ' ', trim($cleaned));
    
    // Normalize line endings for consistency
    $cleaned = str_replace(["\r\n", "\r"], "\n", $cleaned);
    
    // Security: Limit content length to prevent DoS attacks
    $max_length = 50000; // Reasonable limit for ethical evaluation
    if (strlen($cleaned) > $max_length) {
      $cleaned = substr($cleaned, 0, $max_length);
      $this->logger->info('RAIL guardrail: Content truncated to @length characters for security and evaluation limits.', [
        '@length' => $max_length,
      ]);
    }

    // Security: Check for suspicious patterns that might indicate injection attempts
    $suspicious_patterns = [
      '/<script[^>]*>/i',
      '/javascript:/i',
      '/data:text\/html/i',
      '/vbscript:/i',
      '/<iframe[^>]*>/i',
      '/<object[^>]*>/i',
      '/<embed[^>]*>/i',
    ];

    foreach ($suspicious_patterns as $pattern) {
      if (preg_match($pattern, $cleaned)) {
        $this->logger->warning('RAIL guardrail: Suspicious content pattern detected and sanitized.');
        $cleaned = preg_replace($pattern, '[REMOVED]', $cleaned);
      }
    }

    // Ensure we have meaningful content (not just whitespace or very short)
    if (strlen(trim($cleaned)) < 3) {
      return '';
    }

    // Final validation: ensure content is valid UTF-8
    if (!mb_check_encoding($cleaned, 'UTF-8')) {
      $this->logger->warning('RAIL guardrail: Invalid UTF-8 content detected, attempting to clean.');
      $cleaned = mb_convert_encoding($cleaned, 'UTF-8', 'UTF-8');
    }

    // Handle edge case: content became empty after cleaning
    if (empty(trim($cleaned))) {
      $this->logger->debug('RAIL guardrail: Content became empty after security cleaning.');
      return '';
    }

    return $cleaned;
  }

  /**
   * Handles unsupported input types with graceful degradation.
   *
   * @param \Drupal\ai\OperationType\InputInterface $input
   *   The unsupported input.
   *
   * @return string
   *   Empty string for unsupported types.
   */
  protected function handleUnsupportedInputType(InputInterface $input): string {
    $input_type = get_class($input);
    
    // Log different levels based on input type
    if (strpos($input_type, 'Chat') !== FALSE) {
      // Chat-related but not ChatInput - might be a new type
      $this->logger->info('RAIL guardrail: Chat-related input type @type not fully supported, attempting basic text extraction.', [
        '@type' => $input_type,
      ]);
      
      // Try to extract text using common methods
      return $this->attemptGenericTextExtraction($input);
    } else {
      // Completely different input type
      $this->logger->info('RAIL guardrail: Unsupported input type @type, skipping evaluation.', [
        '@type' => $input_type,
      ]);
      return '';
    }
  }

  /**
   * Handles unsupported output types with graceful degradation.
   *
   * @param \Drupal\ai\OperationType\OutputInterface $output
   *   The unsupported output.
   *
   * @return string
   *   Empty string for unsupported types.
   */
  protected function handleUnsupportedOutputType(OutputInterface $output): string {
    $output_type = get_class($output);
    
    // Log different levels based on output type
    if (strpos($output_type, 'Chat') !== FALSE) {
      // Chat-related but not ChatOutput - might be a new type
      $this->logger->info('RAIL guardrail: Chat-related output type @type not fully supported, attempting basic text extraction.', [
        '@type' => $output_type,
      ]);
      
      // Try to extract text using common methods
      return $this->attemptGenericTextExtraction($output);
    } else {
      // Completely different output type
      $this->logger->info('RAIL guardrail: Unsupported output type @type, skipping evaluation.', [
        '@type' => $output_type,
      ]);
      return '';
    }
  }

  /**
   * Attempts generic text extraction from unknown input/output types.
   *
   * @param mixed $object
   *   The input or output object.
   *
   * @return string
   *   Extracted text or empty string.
   */
  protected function attemptGenericTextExtraction($object): string {
    try {
      // Try common text extraction methods
      if (method_exists($object, 'getText')) {
        $text = $object->getText();
        if (is_string($text) && !empty(trim($text))) {
          $this->logger->debug('RAIL guardrail: Successfully extracted text using getText() method.');
          return $text;
        }
      }
      
      if (method_exists($object, 'getContent')) {
        $content = $object->getContent();
        if (is_string($content) && !empty(trim($content))) {
          $this->logger->debug('RAIL guardrail: Successfully extracted text using getContent() method.');
          return $content;
        }
      }
      
      if (method_exists($object, '__toString')) {
        $text = (string) $object;
        if (!empty(trim($text))) {
          $this->logger->debug('RAIL guardrail: Successfully extracted text using __toString() method.');
          return $text;
        }
      }
      
      // If object has messages property, try to extract from there
      if (method_exists($object, 'getMessages')) {
        $messages = $object->getMessages();
        if (is_array($messages) && !empty($messages)) {
          return $this->extractTextFromConversation($messages);
        }
      }
      
    } catch (\Exception $e) {
      $this->logger->warning('RAIL guardrail: Failed to extract text from unknown type: @message', [
        '@message' => $e->getMessage(),
      ]);
    }
    
    return '';
  }

  /**
   * Extracts text from multi-message conversations with context awareness.
   *
   * @param array $messages
   *   Array of chat messages.
   *
   * @return string
   *   Extracted conversation text.
   */
  protected function extractTextFromConversation(array $messages): string {
    if (empty($messages)) {
      return '';
    }

    $conversation_mode = $this->configuration['conversation_mode'] ?? 'recent_user';
    
    switch ($conversation_mode) {
      case 'full_context':
        return $this->extractFullConversationContext($messages);
        
      case 'recent_exchange':
        return $this->extractRecentExchange($messages);
        
      case 'user_messages_only':
        return $this->extractUserMessagesOnly($messages);
        
      case 'recent_user':
      default:
        return $this->extractMostRecentUserMessage($messages);
    }
  }

  /**
   * Extracts the most recent user message from conversation.
   *
   * @param array $messages
   *   Array of chat messages.
   *
   * @return string
   *   The most recent user message text.
   */
  protected function extractMostRecentUserMessage(array $messages): string {
    // Handle multi-message conversations by evaluating the most recent user message
    // Iterate backwards to find the last user message (not system/assistant)
    $user_messages = [];
    foreach (array_reverse($messages) as $message) {
      if (!$message instanceof ChatMessage) {
        continue;
      }
      
      // Prioritize user messages over system/assistant messages
      $role = $message->getRole() ?? 'user';
      if ($role === 'user') {
        $user_messages[] = $message;
      }
    }

    // If no user messages found, fall back to the last message
    if (empty($user_messages)) {
      $last_message = end($messages);
      if (!$last_message instanceof ChatMessage) {
        $this->logger->debug('RAIL guardrail: No valid ChatMessage found in conversation.');
        return '';
      }
      $target_message = $last_message;
    } else {
      // Use the most recent user message
      $target_message = $user_messages[0];
    }

    return $target_message->getText() ?? '';
  }

  /**
   * Extracts full conversation context for comprehensive evaluation.
   *
   * @param array $messages
   *   Array of chat messages.
   *
   * @return string
   *   Full conversation text with role indicators.
   */
  protected function extractFullConversationContext(array $messages): string {
    $conversation_parts = [];
    $max_messages = 10; // Limit to prevent excessive content
    $message_count = 0;
    
    // Take the most recent messages up to the limit
    $recent_messages = array_slice(array_reverse($messages), 0, $max_messages);
    $recent_messages = array_reverse($recent_messages); // Restore chronological order
    
    foreach ($recent_messages as $message) {
      if (!$message instanceof ChatMessage) {
        continue;
      }
      
      $role = $message->getRole() ?? 'user';
      $text = $message->getText() ?? '';
      
      if (!empty(trim($text))) {
        $conversation_parts[] = "[{$role}]: {$text}";
        $message_count++;
      }
    }
    
    if (empty($conversation_parts)) {
      return '';
    }
    
    $full_context = implode("\n", $conversation_parts);
    
    $this->logger->debug('RAIL guardrail: Extracted full conversation context with @count messages.', [
      '@count' => $message_count,
    ]);
    
    return $full_context;
  }

  /**
   * Extracts recent exchange (last user message + last assistant response).
   *
   * @param array $messages
   *   Array of chat messages.
   *
   * @return string
   *   Recent exchange text.
   */
  protected function extractRecentExchange(array $messages): string {
    $recent_user = '';
    $recent_assistant = '';
    
    // Find the most recent user and assistant messages
    foreach (array_reverse($messages) as $message) {
      if (!$message instanceof ChatMessage) {
        continue;
      }
      
      $role = $message->getRole() ?? 'user';
      $text = $message->getText() ?? '';
      
      if (empty(trim($text))) {
        continue;
      }
      
      if ($role === 'user' && empty($recent_user)) {
        $recent_user = $text;
      } elseif (in_array($role, ['assistant', 'system']) && empty($recent_assistant)) {
        $recent_assistant = $text;
      }
      
      // Stop when we have both
      if (!empty($recent_user) && !empty($recent_assistant)) {
        break;
      }
    }
    
    $exchange_parts = [];
    if (!empty($recent_assistant)) {
      $exchange_parts[] = "[assistant]: {$recent_assistant}";
    }
    if (!empty($recent_user)) {
      $exchange_parts[] = "[user]: {$recent_user}";
    }
    
    return implode("\n", $exchange_parts);
  }

  /**
   * Extracts only user messages from conversation.
   *
   * @param array $messages
   *   Array of chat messages.
   *
   * @return string
   *   Combined user messages text.
   */
  protected function extractUserMessagesOnly(array $messages): string {
    $user_texts = [];
    $max_messages = 5; // Limit user messages to prevent excessive content
    $message_count = 0;
    
    foreach (array_reverse($messages) as $message) {
      if (!$message instanceof ChatMessage || $message_count >= $max_messages) {
        break;
      }
      
      $role = $message->getRole() ?? 'user';
      if ($role === 'user') {
        $text = $message->getText() ?? '';
        if (!empty(trim($text))) {
          $user_texts[] = $text;
          $message_count++;
        }
      }
    }
    
    if (empty($user_texts)) {
      return '';
    }
    
    // Reverse to maintain chronological order
    $user_texts = array_reverse($user_texts);
    
    return implode("\n", $user_texts);
  }

  /**
   * Extracts text from ChatOutput with robust error handling.
   *
   * @param \Drupal\ai\OperationType\Chat\ChatOutput $output
   *   The ChatOutput to extract text from.
   *
   * @return string
   *   Extracted text content.
   */
  protected function extractTextFromChatOutput(ChatOutput $output): string {
    try {
      // Handle streamed content reconstruction
      if (method_exists($output, 'isStreamed') && $output->isStreamed()) {
        // For streamed responses, collect all text chunks
        $text = $this->reconstructStreamedContent($output);
        
        $this->logger->debug('RAIL guardrail: Reconstructed @length characters from streamed output.', [
          '@length' => strlen($text),
        ]);
        
        return $text;
      } else {
        // For regular responses, get the complete text from the normalized message
        $normalized = $output->getNormalized();
        if ($normalized instanceof ChatMessage) {
          $text = $normalized->getText();
        } else {
          // Handle streamed message iterator
          $text = $this->reconstructStreamedContent($output);
        }
        
        $this->logger->debug('RAIL guardrail: Extracted @length characters from output.', [
          '@length' => strlen($text),
        ]);
        
        return $text ?? '';
      }
    } catch (\Exception $e) {
      $this->logger->warning('RAIL guardrail: Failed to extract text from ChatOutput: @message', [
        '@message' => $e->getMessage(),
      ]);
      
      // Fallback: try basic text extraction
      try {
        $normalized = $output->getNormalized();
        if ($normalized instanceof ChatMessage) {
          return $normalized->getText() ?? '';
        }
        return '';
      } catch (\Exception $fallback_e) {
        $this->logger->error('RAIL guardrail: Complete failure to extract text from ChatOutput: @message', [
          '@message' => $fallback_e->getMessage(),
        ]);
        return '';
      }
    }
  }

  /**
   * Evaluates content using RAIL ethical dimensions.
   *
   * @param string $text
   *   The text content to evaluate.
   * @param string $context
   *   The evaluation context ('input' or 'output').
   *
   * @return \Drupal\ai\Guardrail\Result\GuardrailResultInterface
   *   The evaluation result.
   */
  protected function evaluateContent(string $text, string $context): GuardrailResultInterface {
    $start_time = microtime(true);
    $api_start_time = null;
    $api_end_time = null;
    
    try {
      $enabled_dimensions = $this->getEnabledDimensions();
      if (empty($enabled_dimensions)) {
        $this->logger->debug('RAIL guardrail: No dimensions enabled for evaluation, allowing content to proceed.');
        
        // Log evaluation result with monitoring
        $this->monitoringService->logEvaluationResult([
          'guardrail_id' => $this->getPluginId(),
          'content_length' => strlen($text),
          'enabled_dimensions' => [],
          'evaluation_context' => $context,
          'weighted_score' => 10.0,
          'overall_threshold' => $this->configuration['overall_threshold'] ?? 7.0,
          'result_type' => 'pass',
          'processing_time' => (microtime(true) - $start_time) * 1000,
          'api_response_time' => 0,
          'dimension_scores' => [],
          'violations' => [],
        ]);
        
        return new PassResult('No dimensions enabled for evaluation.', $this);
      }

      // Use guardrail-specific API configuration
      $endpoint_url = $this->configuration['endpoint_url'] ?? '';
      
      // Get API key from Key module
      $api_key_id = $this->configuration['api_key'] ?? '';
      $api_key = '';
      if (!empty($api_key_id)) {
        $key_entity = $this->keyRepository->getKey($api_key_id);
        if ($key_entity) {
          $api_key = $key_entity->getKeyValue();
        }
      }
      
      $timeout = $this->configuration['timeout'] ?? 60;
      //print_r($this->configuration);
      if (empty($endpoint_url) || empty($api_key)) {
        $this->logger->warning('RAIL guardrail: API not configured, allowing content to proceed (fail-open).');
        
        // Log configuration issue
        $this->monitoringService->logIntegrationEvent('configuration_error', 
          'RAIL API not configured for guardrail', [
          'guardrail_id' => $this->getPluginId(),
          'missing_endpoint' => empty($endpoint_url),
          'missing_api_key' => empty($api_key),
        ]);
        
        return new PassResult('RAIL API not configured for this guardrail, allowing content to proceed.', $this);
      }

      // Check if API is available before making request
      if (!$this->railEvaluationService->isApiAvailable()) {
        $this->logger->info('RAIL guardrail: API currently unavailable, allowing content to proceed (fail-open).');
        
        // Log API availability issue
        $this->monitoringService->logIntegrationEvent('system_error', 
          'RAIL API currently unavailable', [
          'guardrail_id' => $this->getPluginId(),
          'endpoint_url' => $endpoint_url,
        ]);
        
        return new PassResult('RAIL API temporarily unavailable, allowing content to proceed.', $this);
      }

      // Check for recent authentication failures
      if ($this->railEvaluationService->hasRecentAuthFailure()) {
        $this->logger->warning('RAIL guardrail: Recent authentication failure detected, allowing content to proceed (fail-open).');
        
        // Log authentication issue
        $this->monitoringService->logIntegrationEvent('system_error', 
          'RAIL API authentication failure detected', [
          'guardrail_id' => $this->getPluginId(),
          'endpoint_url' => $endpoint_url,
        ]);
        
        return new PassResult('RAIL API authentication issue detected, allowing content to proceed.', $this);
      }

      $request_data = [
        'content' => $text,
        'dimensions' => $enabled_dimensions,
        'options' => [
          'include_details' => $this->configuration['include_details'] ?? TRUE,
          'format' => 'json',
          'timeout' => $timeout,
        ],
      ];

      // Log API usage before making request
      $this->monitoringService->logApiUsage([
        'endpoint' => $endpoint_url,
        'request_size' => strlen(json_encode($request_data)),
        'response_size' => 0, // Will be updated after response
        'dimensions_requested' => count($enabled_dimensions),
        'success' => false, // Will be updated after response
      ]);

      // Make API request with timing
      $api_start_time = microtime(true);
      $evaluation_result = $this->railEvaluationService->makeCustomApiRequest($endpoint_url, $api_key, $request_data, [
        'timeout' => $timeout,
      ]);
      $api_end_time = microtime(true);
      
      if (!$evaluation_result) {
        // Fail-open approach: allow content to proceed if evaluation fails
        $this->logger->info('RAIL guardrail: Evaluation service unavailable, allowing content to proceed (fail-open).');
        
        // Log API failure
        $this->monitoringService->logApiUsage([
          'endpoint' => $endpoint_url,
          'request_size' => strlen(json_encode($request_data)),
          'response_size' => 0,
          'dimensions_requested' => count($enabled_dimensions),
          'success' => false,
        ]);
        
        // Log evaluation result
        $this->monitoringService->logEvaluationResult([
          'guardrail_id' => $this->getPluginId(),
          'content_length' => strlen($text),
          'enabled_dimensions' => $enabled_dimensions,
          'evaluation_context' => $context,
          'weighted_score' => 10.0, // Default pass score for failed evaluation
          'overall_threshold' => $this->configuration['overall_threshold'] ?? 7.0,
          'result_type' => 'pass',
          'processing_time' => (microtime(true) - $start_time) * 1000,
          'api_response_time' => $api_end_time ? ($api_end_time - $api_start_time) * 1000 : 0,
          'dimension_scores' => [],
          'violations' => [],
        ]);
        
        return new PassResult('RAIL evaluation service unavailable, allowing content to proceed.', $this);
      }

      // Log successful API usage
      $response_size = strlen(json_encode($evaluation_result));
      $this->monitoringService->logApiUsage([
        'endpoint' => $endpoint_url,
        'request_size' => strlen(json_encode($request_data)),
        'response_size' => $response_size,
        'dimensions_requested' => count($enabled_dimensions),
        'success' => true,
      ]);

      return $this->processEvaluationResult($evaluation_result, $context, $start_time, $api_start_time, $api_end_time, $text, $enabled_dimensions);
    }
    catch (\Exception $e) {
      // Fail-open approach: log error and allow content to proceed
      $this->logger->error('RAIL guardrail: Evaluation failed with exception: @message. Allowing content to proceed (fail-open).', [
        '@message' => $e->getMessage(),
      ]);
      
      // Log additional context for debugging
      $this->logger->debug('RAIL guardrail: Exception details - Type: @type, File: @file, Line: @line', [
        '@type' => get_class($e),
        '@file' => $e->getFile(),
        '@line' => $e->getLine(),
      ]);
      
      // Log integration failure
      $this->monitoringService->logIntegrationEvent('integration_failure', 
        'RAIL guardrail evaluation failed with exception: ' . $e->getMessage(), [
        'guardrail_id' => $this->getPluginId(),
        'exception_type' => get_class($e),
        'exception_file' => $e->getFile(),
        'exception_line' => $e->getLine(),
      ]);
      
      // Log evaluation result for failed evaluation
      $this->monitoringService->logEvaluationResult([
        'guardrail_id' => $this->getPluginId(),
        'content_length' => strlen($text),
        'enabled_dimensions' => $enabled_dimensions ?? [],
        'evaluation_context' => $context,
        'weighted_score' => 10.0, // Default pass score for failed evaluation
        'overall_threshold' => $this->configuration['overall_threshold'] ?? 7.0,
        'result_type' => 'pass',
        'processing_time' => (microtime(true) - $start_time) * 1000,
        'api_response_time' => $api_end_time && $api_start_time ? ($api_end_time - $api_start_time) * 1000 : 0,
        'dimension_scores' => [],
        'violations' => [],
      ]);
      
      return new PassResult('RAIL evaluation failed due to technical error, allowing content to proceed.', $this);
    }
  }

  /**
   * Processes the evaluation result and determines the guardrail outcome.
   *
   * This method implements comprehensive violation detection logic that:
   * - Validates dimension scores for completeness and consistency
   * - Detects critical dimension failures (Safety, Privacy always stop)
   * - Applies weighted threshold to overall score
   * - Generates detailed violation reports with dimension breakdown
   * - Prioritizes violation messages by dimension weight and severity
   * - Generates appropriate result types based on configuration and violations
   *
   * @param array $evaluation_result
   *   The evaluation result from RAIL API.
   * @param string $context
   *   The evaluation context ('input' or 'output').
   * @param float $start_time
   *   The start time of the evaluation process.
   * @param float|null $api_start_time
   *   The start time of the API call.
   * @param float|null $api_end_time
   *   The end time of the API call.
   * @param string $text
   *   The original text content being evaluated.
   * @param array $enabled_dimensions
   *   The enabled dimensions for evaluation.
   *
   * @return \Drupal\ai\Guardrail\Result\GuardrailResultInterface
   *   The guardrail result.
   */
  protected function processEvaluationResult(array $evaluation_result, string $context, float $start_time = null, float $api_start_time = null, float $api_end_time = null, string $text = '', array $enabled_dimensions = []): GuardrailResultInterface {
    try {
      $dimension_scores = $evaluation_result['dimension_scores'] ?? [];
      
      // First, validate the dimension scores for completeness
      $validation_result = $this->validateDimensionScores($dimension_scores);
      
      if (!$validation_result['valid']) {
        // Log validation errors and fail-open
        $this->logger->error('RAIL guardrail: Dimension score validation failed: @errors. Allowing content to proceed (fail-open).', [
          '@errors' => implode(', ', $validation_result['errors'])
        ]);
        
        // Log evaluation result with monitoring
        $this->logEvaluationWithMonitoring($start_time, $api_start_time, $api_end_time, $text, $enabled_dimensions, $context, 10.0, 'pass', [], []);
        
        return $this->generatePassResult('RAIL evaluation failed validation, allowing content to proceed.', $context, [
          'validation_errors' => $validation_result['errors'],
          'context' => $context,
          'fail_reason' => 'validation_failure',
        ]);
      }

      // Log any validation warnings
      if (!empty($validation_result['warnings'])) {
        $this->logger->warning('RAIL guardrail: Evaluation warnings: @warnings', [
          '@warnings' => implode(', ', $validation_result['warnings'])
        ]);
      }

      // Check for critical dimension violations first (these always stop processing)
      $critical_check = $this->checkCriticalDimensionViolations($dimension_scores);
      
      if ($critical_check['has_critical_violations']) {
        $this->logger->warning('RAIL guardrail: Critical dimension violations detected in @context', [
          '@context' => $context,
        ]);
        
        // Log critical violations with monitoring
        $this->monitoringService->logViolationDetails($critical_check['critical_violations'], [
          'guardrail_id' => $this->getPluginId(),
          'evaluation_context' => $context,
          'critical' => true,
        ]);
        
        // Log evaluation result with monitoring
        $this->logEvaluationWithMonitoring($start_time, $api_start_time, $api_end_time, $text, $enabled_dimensions, $context, 0.0, 'stop', $dimension_scores, $critical_check['critical_violations']);
        
        return $this->generateStopResult($critical_check['critical_violations'], $context, [
          'violations' => $critical_check['critical_violations'],
          'context' => $context,
          'critical' => true,
          'validation_coverage' => $validation_result['coverage'],
        ]);
      }

      // Apply individual dimension thresholds for non-critical violations
      $threshold_results = $this->applyDimensionThresholds($dimension_scores);
      $violations = $threshold_results['violations'];
      
      // Calculate detailed scoring metrics
      $score_metrics = $this->calculateDetailedScoreMetrics($dimension_scores);
      $weighted_score = $score_metrics['weighted_score'];
      $overall_threshold = $this->configuration['overall_threshold'] ?? 7.0;

      // Check if overall weighted score violates threshold
      $overall_violation = $weighted_score < $overall_threshold;
      
      if ($overall_violation || !empty($violations)) {
        // Determine the action to take based on configuration
        $action = $this->configuration['action_on_violation'] ?? 'stop';
        
        // Create detailed violation context
        $all_violations = array_merge($violations, $threshold_results['critical_violations']);
        $violation_context = [
          'violations' => $violations,
          'weighted_score' => $weighted_score,
          'overall_threshold' => $overall_threshold,
          'score_metrics' => $score_metrics,
          'validation_coverage' => $validation_result['coverage'],
          'context' => $context,
          'action_taken' => $action,
          'improvement_suggestions' => $this->generateImprovementSuggestions($all_violations),
        ];

        $this->logger->info('RAIL guardrail: Ethical violations detected in @context (action: @action)', [
          '@context' => $context,
          '@action' => $action,
        ]);

        // Log violations with monitoring
        if (!empty($all_violations)) {
          $this->monitoringService->logViolationDetails($all_violations, [
            'guardrail_id' => $this->getPluginId(),
            'evaluation_context' => $context,
            'weighted_score' => $weighted_score,
            'overall_threshold' => $overall_threshold,
          ]);
        }

        // Log evaluation result with monitoring
        $result_type = ($action === 'warn') ? 'pass' : $action;
        $this->logEvaluationWithMonitoring($start_time, $api_start_time, $api_end_time, $text, $enabled_dimensions, $context, $weighted_score, $result_type, $dimension_scores, $all_violations);

        if ($action === 'stop') {
          return $this->generateStopResult($all_violations, $context, $violation_context);
        }
        elseif ($action === 'rewrite') {
          return $this->generateRewriteResult($all_violations, $context, $violation_context);
        }
        
        // For 'warn' action, log the violation and continue
        $this->logger->warning('RAIL guardrail: Ethical violation (action: @action): @message', [
          '@action' => $action,
          '@message' => $this->buildComprehensiveViolationMessage($all_violations, $weighted_score, $overall_threshold, $context),
        ]);
        
        return $this->generatePassResult('Content passed with warnings (logged violations).', $context, $violation_context);
      }

      // Content passed all evaluations
      $this->logger->debug('RAIL guardrail: Content passed ethical evaluation in @context', [
        '@context' => $context,
      ]);
      
      // Log successful evaluation with monitoring
      $this->logEvaluationWithMonitoring($start_time, $api_start_time, $api_end_time, $text, $enabled_dimensions, $context, $weighted_score, 'pass', $dimension_scores, []);
      
      return $this->generatePassResult('Content passed RAIL ethical evaluation.', $context, [
        'weighted_score' => $weighted_score,
        'overall_threshold' => $overall_threshold,
        'score_metrics' => $score_metrics,
        'passed_dimensions' => $threshold_results['passed_dimensions'],
        'validation_coverage' => $validation_result['coverage'],
        'context' => $context,
      ]);
    }
    catch (\Exception $e) {
      // Fail-open: if processing fails, allow content to proceed
      $this->logger->error('RAIL guardrail: Failed to process evaluation result: @message. Allowing content to proceed (fail-open).', [
        '@message' => $e->getMessage(),
      ]);
      
      // Log processing failure with monitoring
      $this->logEvaluationWithMonitoring($start_time, $api_start_time, $api_end_time, $text, $enabled_dimensions, $context, 10.0, 'pass', [], []);
      
      return $this->generatePassResult('RAIL evaluation processing failed, allowing content to proceed.', $context, [
        'context' => $context,
        'fail_reason' => 'processing_failure',
        'error_message' => $e->getMessage(),
      ]);
    }
  }

  /**
   * Calculates the weighted score across enabled dimensions.
   *
   * This method implements a comprehensive weighted scoring algorithm that:
   * - Only considers enabled dimensions in the calculation
   * - Handles missing dimension scores gracefully by using default values
   * - Applies individual dimension weights to calculate overall score
   * - Returns 0 if no valid dimensions are available for scoring
   *
   * @param array $dimension_scores
   *   The dimension scores from RAIL API in format:
   *   ['dimension_name' => ['score' => float, 'details' => string], ...]
   *
   * @return float
   *   The calculated weighted score (0.0-10.0 scale).
   */
  protected function calculateWeightedScore(array $dimension_scores): float {
    $enabled_dimensions = $this->getEnabledDimensions();
    
    if (empty($enabled_dimensions)) {
      return 0.0;
    }

    $total_weighted_score = 0.0;
    $total_weight = 0.0;
    $processed_dimensions = 0;

    foreach ($enabled_dimensions as $dimension) {
      // Get the score for this dimension, handling missing scores gracefully
      $score = $this->extractDimensionScore($dimension_scores, $dimension);
      
      if ($score !== null) {
        $weight = $this->getDimensionWeight($dimension);
        
        // Validate weight is positive
        if ($weight > 0) {
          $total_weighted_score += $score * $weight;
          $total_weight += $weight;
          $processed_dimensions++;
        }
      }
    }

    // Return weighted average if we have valid dimensions, otherwise 0
    if ($total_weight > 0 && $processed_dimensions > 0) {
      return $total_weighted_score / $total_weight;
    }

    return 0.0;
  }

  /**
   * Extracts and validates a dimension score from API response.
   *
   * Handles missing dimension scores gracefully by returning null for
   * dimensions that are not present in the API response.
   *
   * @param array $dimension_scores
   *   The dimension scores from RAIL API.
   * @param string $dimension
   *   The dimension name to extract.
   *
   * @return float|null
   *   The dimension score (0.0-10.0) or null if not available.
   */
  protected function extractDimensionScore(array $dimension_scores, string $dimension): ?float {
    if (!isset($dimension_scores[$dimension])) {
      return null;
    }

    $score_data = $dimension_scores[$dimension];
    
    // Handle different possible response formats
    if (is_array($score_data)) {
      $score = $score_data['score'] ?? null;
    } elseif (is_numeric($score_data)) {
      $score = $score_data;
    } else {
      return null;
    }

    // Validate score is numeric and within expected range
    if (!is_numeric($score)) {
      return null;
    }

    $score = (float) $score;
    
    // Clamp score to valid range (0.0-10.0)
    return max(0.0, min(10.0, $score));
  }

  /**
   * Calculates detailed scoring metrics for analysis and reporting.
   *
   * Provides comprehensive scoring information including individual dimension
   * contributions, missing dimensions, and overall statistics.
   *
   * @param array $dimension_scores
   *   The dimension scores from RAIL API.
   *
   * @return array
   *   Detailed scoring metrics with keys:
   *   - weighted_score: Overall weighted score
   *   - total_weight: Sum of all dimension weights
   *   - processed_dimensions: Count of dimensions with valid scores
   *   - missing_dimensions: Array of enabled dimensions without scores
   *   - dimension_contributions: Individual dimension score contributions
   */
  protected function calculateDetailedScoreMetrics(array $dimension_scores): array {
    $enabled_dimensions = $this->getEnabledDimensions();
    $total_weighted_score = 0.0;
    $total_weight = 0.0;
    $processed_dimensions = 0;
    $missing_dimensions = [];
    $dimension_contributions = [];

    foreach ($enabled_dimensions as $dimension) {
      $score = $this->extractDimensionScore($dimension_scores, $dimension);
      $weight = $this->getDimensionWeight($dimension);
      
      if ($score !== null && $weight > 0) {
        $contribution = $score * $weight;
        $total_weighted_score += $contribution;
        $total_weight += $weight;
        $processed_dimensions++;
        
        $dimension_contributions[$dimension] = [
          'score' => $score,
          'weight' => $weight,
          'contribution' => $contribution,
          'percentage' => 0, // Will be calculated after total_weight is known
        ];
      } else {
        $missing_dimensions[] = $dimension;
      }
    }

    // Calculate percentage contributions
    if ($total_weight > 0) {
      foreach ($dimension_contributions as $dimension => &$contribution) {
        $contribution['percentage'] = ($contribution['weight'] / $total_weight) * 100;
      }
    }

    $weighted_score = $total_weight > 0 ? $total_weighted_score / $total_weight : 0.0;

    return [
      'weighted_score' => $weighted_score,
      'total_weight' => $total_weight,
      'processed_dimensions' => $processed_dimensions,
      'missing_dimensions' => $missing_dimensions,
      'dimension_contributions' => $dimension_contributions,
      'enabled_dimensions_count' => count($enabled_dimensions),
    ];
  }

  /**
   * Gets the enabled dimensions from configuration.
   *
   * @return array
   *   Array of enabled dimension names.
   */
  protected function getEnabledDimensions(): array {
    return $this->configuration['enabled_dimensions'] ?? [];
  }

  /**
   * Gets the weight for a specific dimension.
   *
   * @param string $dimension
   *   The dimension name.
   *
   * @return float
   *   The dimension weight.
   */
  protected function getDimensionWeight(string $dimension): float {
    return $this->configuration['dimension_weights'][$dimension] ?? 1.0;
  }

  /**
   * Gets the threshold for a specific dimension.
   *
   * @param string $dimension
   *   The dimension name.
   *
   * @return float
   *   The dimension threshold.
   */
  protected function getDimensionThreshold(string $dimension): float {
    return $this->configuration['dimension_thresholds'][$dimension] ?? 7.0;
  }

  /**
   * Applies individual dimension thresholds to detect violations.
   *
   * Checks each enabled dimension against its configured threshold and
   * categorizes violations by severity (critical vs non-critical).
   *
   * @param array $dimension_scores
   *   The dimension scores from RAIL API.
   *
   * @return array
   *   Array with keys:
   *   - violations: Array of non-critical dimension violations
   *   - critical_violations: Array of critical dimension violations
   *   - passed_dimensions: Array of dimensions that passed thresholds
   */
  protected function applyDimensionThresholds(array $dimension_scores): array {
    $violations = [];
    $critical_violations = [];
    $passed_dimensions = [];
    $enabled_dimensions = $this->getEnabledDimensions();

    foreach ($enabled_dimensions as $dimension) {
      $score = $this->extractDimensionScore($dimension_scores, $dimension);
      
      if ($score === null) {
        // Skip dimensions without scores - they don't contribute to violations
        continue;
      }

      $threshold = $this->getDimensionThreshold($dimension);
      
      if ($score < $threshold) {
        $violation_info = [
          'dimension' => $dimension,
          'score' => $score,
          'threshold' => $threshold,
          'weight' => $this->getDimensionWeight($dimension),
          'message' => $this->getDimensionViolationMessage($dimension),
          'severity' => $score < ($threshold * 0.5) ? 'high' : 'medium',
        ];

        if (in_array($dimension, self::CRITICAL_DIMENSIONS)) {
          $critical_violations[] = $violation_info;
        } else {
          $violations[] = $violation_info;
        }
      } else {
        $passed_dimensions[] = [
          'dimension' => $dimension,
          'score' => $score,
          'threshold' => $threshold,
          'margin' => $score - $threshold,
        ];
      }
    }

    return [
      'violations' => $violations,
      'critical_violations' => $critical_violations,
      'passed_dimensions' => $passed_dimensions,
    ];
  }

  /**
   * Checks if any critical dimensions have violations.
   *
   * Critical dimensions (Safety, Privacy) always stop processing regardless
   * of overall weighted score or other configuration settings.
   *
   * @param array $dimension_scores
   *   The dimension scores from RAIL API.
   *
   * @return array
   *   Array with keys:
   *   - has_critical_violations: Boolean indicating if critical violations exist
   *   - critical_violations: Array of critical violation details
   */
  protected function checkCriticalDimensionViolations(array $dimension_scores): array {
    $critical_violations = [];
    
    foreach (self::CRITICAL_DIMENSIONS as $critical_dimension) {
      // Only check if the critical dimension is enabled
      if (!in_array($critical_dimension, $this->getEnabledDimensions())) {
        continue;
      }

      $score = $this->extractDimensionScore($dimension_scores, $critical_dimension);
      
      if ($score === null) {
        // Missing critical dimension scores are treated as violations for safety
        $critical_violations[] = [
          'dimension' => $critical_dimension,
          'score' => null,
          'threshold' => $this->getDimensionThreshold($critical_dimension),
          'weight' => $this->getDimensionWeight($critical_dimension),
          'message' => $this->t('Critical dimension @dimension could not be evaluated.', [
            '@dimension' => self::ETHICAL_DIMENSIONS[$critical_dimension]
          ]),
          'severity' => 'critical',
          'reason' => 'missing_score',
        ];
        continue;
      }

      $threshold = $this->getDimensionThreshold($critical_dimension);
      
      if ($score < $threshold) {
        $critical_violations[] = [
          'dimension' => $critical_dimension,
          'score' => $score,
          'threshold' => $threshold,
          'weight' => $this->getDimensionWeight($critical_dimension),
          'message' => $this->getDimensionViolationMessage($critical_dimension),
          'severity' => 'critical',
          'reason' => 'threshold_violation',
        ];
      }
    }

    return [
      'has_critical_violations' => !empty($critical_violations),
      'critical_violations' => $critical_violations,
    ];
  }

  /**
   * Validates dimension scores for completeness and consistency.
   *
   * Performs validation checks on the dimension scores received from the API
   * to ensure they are complete and consistent with configuration.
   *
   * @param array $dimension_scores
   *   The dimension scores from RAIL API.
   *
   * @return array
   *   Validation result with keys:
   *   - valid: Boolean indicating if scores are valid
   *   - warnings: Array of non-critical issues
   *   - errors: Array of critical issues
   *   - coverage: Percentage of enabled dimensions with scores
   */
  protected function validateDimensionScores(array $dimension_scores): array {
    $enabled_dimensions = $this->getEnabledDimensions();
    $warnings = [];
    $errors = [];
    $valid_scores = 0;

    if (empty($enabled_dimensions)) {
      $errors[] = 'No dimensions are enabled for evaluation';
      return [
        'valid' => false,
        'warnings' => $warnings,
        'errors' => $errors,
        'coverage' => 0,
      ];
    }

    foreach ($enabled_dimensions as $dimension) {
      $score = $this->extractDimensionScore($dimension_scores, $dimension);
      
      if ($score === null) {
        if (in_array($dimension, self::CRITICAL_DIMENSIONS)) {
          $errors[] = "Missing score for critical dimension: {$dimension}";
        } else {
          $warnings[] = "Missing score for dimension: {$dimension}";
        }
      } else {
        $valid_scores++;
        
        // Check for suspicious scores (exactly 0 or 10 might indicate API issues)
        if ($score === 0.0) {
          $warnings[] = "Dimension {$dimension} has score of 0.0 - verify API response";
        } elseif ($score === 10.0) {
          $warnings[] = "Dimension {$dimension} has perfect score of 10.0 - verify API response";
        }
      }
    }

    $coverage = count($enabled_dimensions) > 0 ? ($valid_scores / count($enabled_dimensions)) * 100 : 0;
    
    // Require at least 50% coverage for valid evaluation
    if ($coverage < 50) {
      $errors[] = "Insufficient dimension coverage: {$coverage}% (minimum 50% required)";
    }

    return [
      'valid' => empty($errors),
      'warnings' => $warnings,
      'errors' => $errors,
      'coverage' => $coverage,
    ];
  }

  /**
   * Gets the violation message for a specific dimension.
   *
   * Returns dimension-specific violation messages with actionable feedback.
   * Uses custom configured messages when available, otherwise provides
   * default actionable guidance for each ethical dimension.
   *
   * @param string $dimension
   *   The dimension name.
   *
   * @return string
   *   The violation message with actionable feedback.
   */
  protected function getDimensionViolationMessage(string $dimension): string {
    // Check for custom configured messages first
    $messages = $this->configuration['violation_messages'] ?? [];
    if (isset($messages[$dimension]) && !empty(trim($messages[$dimension]))) {
      return trim($messages[$dimension]);
    }
    
    // Provide default actionable messages for each dimension
    $default_messages = [
      'fairness' => (string) $this->t('Content shows bias or unfair treatment. Review for equitable representation and remove discriminatory language.'),
      'safety' => (string) $this->t('Content contains potentially harmful elements. Remove dangerous information and add appropriate safety warnings.'),
      'reliability' => (string) $this->t('Content accuracy concerns detected. Verify facts, add credible sources, and ensure consistency.'),
      'transparency' => (string) $this->t('Content lacks clarity or transparency. Provide clearer explanations and additional context.'),
      'privacy' => (string) $this->t('Content may compromise privacy. Remove personal information and ensure data protection compliance.'),
      'accountability' => (string) $this->t('Content accountability issues detected. Add clear attribution and take responsibility for claims.'),
      'inclusivity' => (string) $this->t('Content may exclude certain groups. Use inclusive language and consider diverse perspectives.'),
      'user_impact' => (string) $this->t('Content may negatively impact users. Consider broader implications and provide balanced viewpoints.'),
    ];
    
    return $default_messages[$dimension] ?? (string) $this->t('Content violates @dimension ethical standards. Please review and modify accordingly.', [
      '@dimension' => self::ETHICAL_DIMENSIONS[$dimension] ?? $dimension
    ]);
  }

  /**
   * Generates an aggregated violation message with improvement suggestions.
   *
   * Creates a coherent message that aggregates multiple violations and
   * provides actionable feedback to users. Prioritizes violations by
   * dimension weight and severity, and includes specific improvement
   * suggestions when possible.
   *
   * @param array $violations
   *   Array of violation information.
   * @param string $context
   *   The evaluation context ('input' or 'output').
   * @param bool $include_suggestions
   *   Whether to include improvement suggestions in the message.
   *
   * @return string
   *   The aggregated violation message with actionable feedback.
   */
  protected function generateAggregatedViolationMessage(array $violations, string $context, bool $include_suggestions = true): string {
    if (empty($violations)) {
      return $this->t('Content violates ethical standards.');
    }

    // Sort violations by priority (critical first, then by weight and severity)
    $sorted_violations = $this->prioritizeViolations($violations);
    
    $message_parts = [];
    
    // Main violation summary
    $violation_count = count($violations);
    $critical_count = count(array_filter($violations, function($v) {
      return in_array($v['dimension'], self::CRITICAL_DIMENSIONS);
    }));
    
    if ($critical_count > 0) {
      $message_parts[] = $this->t('Critical ethical violations detected (@critical_count critical, @total_count total)', [
        '@critical_count' => $critical_count,
        '@total_count' => $violation_count,
      ]);
    } else {
      $message_parts[] = $this->t('Ethical violations detected (@count dimensions affected)', [
        '@count' => $violation_count,
      ]);
    }
    
    // Primary violation details (top 2 most severe)
    $primary_violations = array_slice($sorted_violations, 0, 2);
    $violation_details = [];
    
    foreach ($primary_violations as $violation) {
      $dimension_name = self::ETHICAL_DIMENSIONS[$violation['dimension']] ?? $violation['dimension'];
      $score = $violation['score'] ?? 0;
      $threshold = $violation['threshold'];
      
      $violation_details[] = $this->t('@dimension: @score/@threshold', [
        '@dimension' => $dimension_name,
        '@score' => number_format($score, 1),
        '@threshold' => number_format($threshold, 1),
      ]);
    }
    
    if (!empty($violation_details)) {
      $message_parts[] = $this->t('Primary concerns: @details', [
        '@details' => implode(', ', $violation_details)
      ]);
    }
    
    // Add improvement suggestions if requested
    if ($include_suggestions) {
      $suggestions = $this->generateImprovementSuggestions($violations);
      if (!empty($suggestions)) {
        $top_suggestions = array_slice($suggestions, 0, 3); // Limit to top 3
        $message_parts[] = $this->t('Suggested improvements: @suggestions', [
          '@suggestions' => implode('; ', $top_suggestions)
        ]);
      }
    }
    
    return implode('. ', $message_parts);
  }

  /**
   * Prioritizes violations by severity and importance.
   *
   * Sorts violations to prioritize critical dimensions first, then by
   * severity level, and finally by dimension weight.
   *
   * @param array $violations
   *   Array of violation information.
   *
   * @return array
   *   Sorted array of violations by priority.
   */
  protected function prioritizeViolations(array $violations): array {
    $sorted = $violations;
    
    usort($sorted, function ($a, $b) {
      // Critical violations first
      $a_critical = in_array($a['dimension'], self::CRITICAL_DIMENSIONS);
      $b_critical = in_array($b['dimension'], self::CRITICAL_DIMENSIONS);
      
      if ($a_critical !== $b_critical) {
        return $b_critical <=> $a_critical;
      }
      
      // Then by severity level
      $severity_order = ['critical' => 4, 'high' => 3, 'medium' => 2, 'low' => 1];
      $a_severity = $severity_order[$a['severity'] ?? 'medium'];
      $b_severity = $severity_order[$b['severity'] ?? 'medium'];
      
      if ($a_severity !== $b_severity) {
        return $b_severity <=> $a_severity;
      }
      
      // Then by how far below threshold (more severe violations first)
      $a_gap = ($a['threshold'] - ($a['score'] ?? 0)) / $a['threshold'];
      $b_gap = ($b['threshold'] - ($b['score'] ?? 0)) / $b['threshold'];
      
      if (abs($a_gap - $b_gap) > 0.1) {
        return $b_gap <=> $a_gap;
      }
      
      // Finally by weight (higher weight = more important)
      return $b['weight'] <=> $a['weight'];
    });
    
    return $sorted;
  }

  /**
   * Provides actionable feedback to users based on violations.
   *
   * Generates specific, contextual feedback that helps users understand
   * what went wrong and how to fix it. Tailors feedback based on the
   * evaluation context (input vs output) and violation severity.
   *
   * @param array $violations
   *   Array of violation information.
   * @param string $context
   *   The evaluation context ('input' or 'output').
   *
   * @return array
   *   Array of actionable feedback messages.
   */
  protected function provideActionableFeedback(array $violations, string $context): array {
    $feedback = [];
    
    // Group violations by dimension for targeted feedback
    $violations_by_dimension = [];
    foreach ($violations as $violation) {
      $dimension = $violation['dimension'];
      $violations_by_dimension[$dimension][] = $violation;
    }
    
    // Generate context-specific feedback for each violated dimension
    foreach ($violations_by_dimension as $dimension => $dimension_violations) {
      $dimension_name = self::ETHICAL_DIMENSIONS[$dimension] ?? $dimension;
      $severity = $this->calculateDimensionSeverity($dimension_violations);
      
      // Get base feedback for the dimension
      $base_feedback = $this->getDimensionActionableFeedback($dimension, $context, $severity);
      
      // Add specific score information
      $worst_violation = $this->getWorstViolation($dimension_violations);
      $score_info = $this->t('Current score: @score (threshold: @threshold)', [
        '@score' => number_format($worst_violation['score'] ?? 0, 1),
        '@threshold' => number_format($worst_violation['threshold'], 1),
      ]);
      
      $feedback[] = $this->t('@dimension: @feedback (@score_info)', [
        '@dimension' => $dimension_name,
        '@feedback' => $base_feedback,
        '@score_info' => $score_info,
      ]);
    }
    
    return $feedback;
  }

  /**
   * Gets dimension-specific actionable feedback.
   *
   * Provides targeted feedback based on the specific ethical dimension,
   * evaluation context, and severity of the violation.
   *
   * @param string $dimension
   *   The violated dimension.
   * @param string $context
   *   The evaluation context ('input' or 'output').
   * @param string $severity
   *   The severity level ('low', 'medium', 'high', 'critical').
   *
   * @return string
   *   Actionable feedback for the dimension.
   */
  protected function getDimensionActionableFeedback(string $dimension, string $context, string $severity): string {
    $feedback_map = [
      'fairness' => [
        'input' => $this->t('Review your request for bias or discriminatory language'),
        'output' => $this->t('The response shows bias - request a more balanced perspective'),
      ],
      'safety' => [
        'input' => $this->t('Your request contains potentially harmful elements - please rephrase'),
        'output' => $this->t('The response contains unsafe content - request safer alternatives'),
      ],
      'reliability' => [
        'input' => $this->t('Your request may lead to unreliable information - be more specific'),
        'output' => $this->t('The response lacks reliability - ask for sources and verification'),
      ],
      'transparency' => [
        'input' => $this->t('Your request could be clearer - provide more context'),
        'output' => $this->t('The response lacks transparency - ask for clearer explanations'),
      ],
      'privacy' => [
        'input' => $this->t('Your request may involve private information - remove personal details'),
        'output' => $this->t('The response may compromise privacy - request anonymized information'),
      ],
      'accountability' => [
        'input' => $this->t('Your request should specify accountability requirements'),
        'output' => $this->t('The response lacks accountability - ask for sources and attribution'),
      ],
      'inclusivity' => [
        'input' => $this->t('Your request could be more inclusive - consider diverse perspectives'),
        'output' => $this->t('The response may exclude certain groups - request more inclusive content'),
      ],
      'user_impact' => [
        'input' => $this->t('Consider the broader impact of your request'),
        'output' => $this->t('The response may have negative impacts - request balanced alternatives'),
      ],
    ];
    
    $context_feedback = $feedback_map[$dimension][$context] ?? 
      $this->t('Address @dimension concerns in your @context', [
        '@dimension' => $dimension,
        '@context' => $context,
      ]);
    
    // Add severity-specific guidance
    if ($severity === 'critical' || $severity === 'high') {
      $context_feedback .= $this->t(' (immediate attention required)');
    }
    
    return $context_feedback;
  }

  /**
   * Calculates the overall severity for a dimension based on its violations.
   *
   * @param array $violations
   *   Array of violations for a single dimension.
   *
   * @return string
   *   The severity level ('low', 'medium', 'high', 'critical').
   */
  protected function calculateDimensionSeverity(array $violations): string {
    $max_severity = 'low';
    $severity_levels = ['low' => 1, 'medium' => 2, 'high' => 3, 'critical' => 4];
    
    foreach ($violations as $violation) {
      $severity = $violation['severity'] ?? 'medium';
      if ($severity_levels[$severity] > $severity_levels[$max_severity]) {
        $max_severity = $severity;
      }
    }
    
    return $max_severity;
  }

  /**
   * Gets the worst violation from a set of violations for a dimension.
   *
   * @param array $violations
   *   Array of violations for a single dimension.
   *
   * @return array
   *   The violation with the lowest score (worst performance).
   */
  protected function getWorstViolation(array $violations): array {
    $worst = $violations[0];
    
    foreach ($violations as $violation) {
      if (($violation['score'] ?? 0) < ($worst['score'] ?? 0)) {
        $worst = $violation;
      }
    }
    
    return $worst;
  }

  /**
   * Builds a comprehensive violation message.
   *
   * Creates dimension-specific violation messages with actionable feedback
   * and improvement suggestions. Aggregates multiple violations into a
   * coherent message prioritized by dimension weight and severity.
   *
   * @param array $violations
   *   Array of violation information.
   * @param string $context
   *   The evaluation context.
   * @param bool $is_critical
   *   Whether these are critical violations.
   *
   * @return string
   *   The formatted violation message with actionable feedback.
   */
  protected function buildViolationMessage(array $violations, string $context, bool $is_critical): string {
    if (empty($violations)) {
      return $this->t('Content violates ethical standards.');
    }

    // Sort violations by weight (highest first) for prioritization
    usort($violations, function ($a, $b) {
      // Critical violations first
      $a_critical = in_array($a['dimension'], self::CRITICAL_DIMENSIONS);
      $b_critical = in_array($b['dimension'], self::CRITICAL_DIMENSIONS);
      
      if ($a_critical !== $b_critical) {
        return $b_critical <=> $a_critical;
      }
      
      // Then by severity
      $severity_order = ['critical' => 3, 'high' => 2, 'medium' => 1, 'low' => 0];
      $a_severity = $severity_order[$a['severity'] ?? 'medium'];
      $b_severity = $severity_order[$b['severity'] ?? 'medium'];
      
      if ($a_severity !== $b_severity) {
        return $b_severity <=> $a_severity;
      }
      
      // Finally by weight (highest first)
      return $b['weight'] <=> $a['weight'];
    });

    $messages = [];
    $prefix = $is_critical ? $this->t('Critical ethical violation') : $this->t('Ethical violation');
    
    // Create dimension-specific violation messages with actionable feedback
    foreach ($violations as $violation) {
      $dimension_name = self::ETHICAL_DIMENSIONS[$violation['dimension']] ?? $violation['dimension'];
      
      // Get custom violation message if configured, otherwise use default
      $custom_message = $this->getDimensionViolationMessage($violation['dimension']);
      
      // Build detailed violation info with score and actionable guidance
      $violation_details = $this->t('@dimension (score: @score/@threshold): @message', [
        '@dimension' => $dimension_name,
        '@score' => number_format($violation['score'] ?? 0, 1),
        '@threshold' => number_format($violation['threshold'], 1),
        '@message' => $custom_message,
      ]);
      
      $messages[] = $violation_details;
    }

    // Limit to top 3 violations for readability, with count of additional
    $displayed_messages = array_slice($messages, 0, 3);
    $message_text = implode('; ', $displayed_messages);
    
    if (count($messages) > 3) {
      $additional_count = count($messages) - 3;
      $message_text .= $this->t(' (+@count more violations)', ['@count' => $additional_count]);
    }

    return $prefix . ' in ' . $context . ': ' . $message_text;
  }

  /**
   * Builds a comprehensive violation message with detailed breakdown.
   *
   * Creates detailed violation messages that include both individual dimension
   * violations and overall weighted score information, prioritized by dimension
   * weight and severity. Uses the new aggregated message generation for
   * improved user feedback.
   *
   * @param array $violations
   *   Array of violation information.
   * @param float $weighted_score
   *   The calculated weighted score.
   * @param float $overall_threshold
   *   The overall threshold that was violated.
   * @param string $context
   *   The evaluation context ('input' or 'output').
   *
   * @return string
   *   The comprehensive formatted violation message with actionable feedback.
   */
  protected function buildComprehensiveViolationMessage(array $violations, float $weighted_score, float $overall_threshold, string $context): string {
    $message_parts = [];
    
    // Overall score violation message
    if ($weighted_score < $overall_threshold) {
      $message_parts[] = $this->t('Overall ethical score (@score) below threshold (@threshold)', [
        '@score' => number_format($weighted_score, 1),
        '@threshold' => number_format($overall_threshold, 1),
      ]);
    }

    // Use the new aggregated violation message generation
    if (!empty($violations)) {
      $aggregated_message = $this->generateAggregatedViolationMessage($violations, $context, true);
      $message_parts[] = $aggregated_message;
    }

    $full_message = implode('. ', $message_parts);
    
    return $this->t('Ethical evaluation failed for @context: @details', [
      '@context' => $context,
      '@details' => $full_message,
    ]);
  }

  /**
   * Generates actionable improvement suggestions based on violations.
   *
   * Provides specific, actionable suggestions for addressing ethical violations
   * based on the dimensions that failed and their severity.
   *
   * @param array $violations
   *   Array of violation information.
   *
   * @return array
   *   Array of improvement suggestions.
   */
  protected function generateImprovementSuggestions(array $violations): array {
    $suggestions = [];
    
    $dimension_suggestions = [
      'fairness' => [
        'Review content for bias or discriminatory language',
        'Ensure balanced representation of different groups',
        'Consider diverse perspectives in content creation',
      ],
      'safety' => [
        'Remove potentially harmful or dangerous content',
        'Add appropriate safety warnings if needed',
        'Verify content does not promote unsafe behaviors',
      ],
      'reliability' => [
        'Fact-check all claims and statements',
        'Provide credible sources for information',
        'Ensure consistency in messaging',
      ],
      'transparency' => [
        'Clarify any ambiguous statements',
        'Provide context for complex topics',
        'Explain reasoning behind recommendations',
      ],
      'privacy' => [
        'Remove or anonymize personal information',
        'Review data handling practices',
        'Ensure compliance with privacy regulations',
      ],
      'accountability' => [
        'Take responsibility for content accuracy',
        'Provide clear attribution for sources',
        'Establish clear content ownership',
      ],
      'inclusivity' => [
        'Use inclusive language and examples',
        'Consider accessibility requirements',
        'Ensure content is welcoming to all users',
      ],
      'user_impact' => [
        'Consider broader implications of content',
        'Assess potential negative effects on users',
        'Provide balanced viewpoints on controversial topics',
      ],
    ];

    foreach ($violations as $violation) {
      $dimension = $violation['dimension'];
      if (isset($dimension_suggestions[$dimension])) {
        $suggestions = array_merge($suggestions, $dimension_suggestions[$dimension]);
      }
    }

    // Remove duplicates and limit to most relevant suggestions
    $suggestions = array_unique($suggestions);
    return array_slice($suggestions, 0, 5); // Limit to top 5 suggestions
  }

  /**
   * Generates a PassResult for content that meets thresholds.
   *
   * Creates a PassResult with detailed context including dimension scores,
   * weights, and evaluation metadata for successful evaluations.
   *
   * @param string $message
   *   The success message.
   * @param string $context
   *   The evaluation context ('input' or 'output').
   * @param array $additional_context
   *   Additional context data to include.
   *
   * @return \Drupal\ai\Guardrail\Result\PassResult
   *   The pass result with detailed context.
   */
  protected function generatePassResult(string $message, string $context, array $additional_context = []): PassResult {
    $result_context = array_merge([
      'guardrail_type' => 'rail_ethical_dimensions',
      'evaluation_context' => $context,
      'timestamp' => time(),
      'enabled_dimensions' => $this->getEnabledDimensions(),
    ], $additional_context);

    return new PassResult($message, $this, $result_context);
  }

  /**
   * Generates a StopResult for violations with detailed context.
   *
   * Creates a StopResult with comprehensive violation information including
   * dimension scores, weights, violation details, and improvement suggestions.
   *
   * @param array $violations
   *   Array of violation information.
   * @param string $context
   *   The evaluation context ('input' or 'output').
   * @param array $additional_context
   *   Additional context data to include.
   *
   * @return \Drupal\ai\Guardrail\Result\StopResult
   *   The stop result with detailed violation context.
   */
  protected function generateStopResult(array $violations, string $context, array $additional_context = []): StopResult {
    $message = $this->buildViolationMessage($violations, $context, true);
    
    // Calculate violation score for StopResult (lower score = more severe violation)
    $violation_score = $this->calculateViolationScore($violations);
    
    $result_context = array_merge([
      'guardrail_type' => 'rail_ethical_dimensions',
      'evaluation_context' => $context,
      'timestamp' => time(),
      'violation_count' => count($violations),
      'violation_score' => $violation_score,
      'enabled_dimensions' => $this->getEnabledDimensions(),
      'improvement_suggestions' => $this->generateImprovementSuggestions($violations),
    ], $additional_context);

    return new StopResult($message, $this, $result_context, $violation_score);
  }

  /**
   * Generates a RewriteResult when configured for content modification.
   *
   * Creates appropriate RewriteInputResult or RewriteOutputResult based on
   * the evaluation context, with detailed violation information and
   * improvement suggestions for content modification.
   *
   * @param array $violations
   *   Array of violation information.
   * @param string $context
   *   The evaluation context ('input' or 'output').
   * @param array $additional_context
   *   Additional context data to include.
   *
   * @return \Drupal\ai\Guardrail\Result\GuardrailResultInterface
   *   The appropriate rewrite result based on context.
   */
  protected function generateRewriteResult(array $violations, string $context, array $additional_context = []): GuardrailResultInterface {
    $message = $this->buildRewriteMessage($violations, $context);
    
    $result_context = array_merge([
      'guardrail_type' => 'rail_ethical_dimensions',
      'evaluation_context' => $context,
      'timestamp' => time(),
      'violation_count' => count($violations),
      'rewrite_reason' => 'ethical_violations',
      'enabled_dimensions' => $this->getEnabledDimensions(),
      'improvement_suggestions' => $this->generateImprovementSuggestions($violations),
    ], $additional_context);

    // Return appropriate rewrite result based on context
    if ($context === 'input') {
      return new RewriteInputResult($message, $this, $result_context);
    } else {
      return new RewriteOutputResult($message, $this, $result_context);
    }
  }

  /**
   * Calculates a violation score based on the severity and weight of violations.
   *
   * Lower scores indicate more severe violations. Critical violations always
   * result in very low scores.
   *
   * @param array $violations
   *   Array of violation information.
   *
   * @return float
   *   The violation score (0.0-1.0, where 0.0 is most severe).
   */
  protected function calculateViolationScore(array $violations): float {
    if (empty($violations)) {
      return 1.0; // No violations = perfect score
    }

    $total_severity = 0.0;
    $total_weight = 0.0;

    foreach ($violations as $violation) {
      $dimension = $violation['dimension'];
      $score = $violation['score'] ?? 0.0;
      $threshold = $violation['threshold'] ?? 7.0;
      $weight = $violation['weight'] ?? 1.0;
      
      // Calculate severity based on how far below threshold the score is
      $severity_ratio = $threshold > 0 ? max(0, $score / $threshold) : 0;
      
      // Critical dimensions get extra penalty
      if (in_array($dimension, self::CRITICAL_DIMENSIONS)) {
        $severity_ratio *= 0.5; // Double the penalty for critical dimensions
      }
      
      $total_severity += $severity_ratio * $weight;
      $total_weight += $weight;
    }

    // Return weighted average severity (inverted so lower = worse)
    return $total_weight > 0 ? $total_severity / $total_weight : 0.0;
  }

  /**
   * Builds a rewrite message with actionable guidance.
   *
   * Creates a message specifically for rewrite results that includes
   * clear guidance on what needs to be modified and why.
   *
   * @param array $violations
   *   Array of violation information.
   * @param string $context
   *   The evaluation context ('input' or 'output').
   *
   * @return string
   *   The formatted rewrite message with actionable guidance.
   */
  protected function buildRewriteMessage(array $violations, string $context): string {
    if (empty($violations)) {
      return $this->t('Content requires modification to meet ethical standards.');
    }

    // Sort violations by severity and weight for prioritization
    usort($violations, function ($a, $b) {
      // Critical violations first
      $a_critical = in_array($a['dimension'], self::CRITICAL_DIMENSIONS);
      $b_critical = in_array($b['dimension'], self::CRITICAL_DIMENSIONS);
      
      if ($a_critical !== $b_critical) {
        return $b_critical <=> $a_critical;
      }
      
      // Then by weight (highest first)
      return $b['weight'] <=> $a['weight'];
    });

    $primary_violation = $violations[0];
    $dimension_name = self::ETHICAL_DIMENSIONS[$primary_violation['dimension']] ?? $primary_violation['dimension'];
    
    $message_parts = [];
    
    // Primary violation message
    $message_parts[] = $this->t('Content modification required due to @dimension concerns', [
      '@dimension' => $dimension_name
    ]);
    
    // Add specific guidance based on the primary violation
    $guidance = $this->getRewriteGuidance($primary_violation['dimension']);
    if ($guidance) {
      $message_parts[] = $guidance;
    }
    
    // Add count of additional violations if any
    if (count($violations) > 1) {
      $additional_count = count($violations) - 1;
      $message_parts[] = $this->t('(@additional_count additional ethical concerns detected)', [
        '@additional_count' => $additional_count
      ]);
    }

    return implode('. ', $message_parts);
  }

  /**
   * Gets specific rewrite guidance for a dimension.
   *
   * Provides actionable guidance for content modification based on
   * the specific ethical dimension that was violated.
   *
   * @param string $dimension
   *   The dimension that was violated.
   *
   * @return string
   *   Specific guidance for addressing the violation.
   */
  protected function getRewriteGuidance(string $dimension): string {
    $guidance_map = [
      'fairness' => $this->t('Review for bias and ensure equitable treatment of all groups'),
      'safety' => $this->t('Remove harmful content and add appropriate safety considerations'),
      'reliability' => $this->t('Verify accuracy and provide credible sources'),
      'transparency' => $this->t('Clarify ambiguous statements and provide clear explanations'),
      'privacy' => $this->t('Remove personal information and ensure privacy compliance'),
      'accountability' => $this->t('Add clear attribution and take responsibility for claims'),
      'inclusivity' => $this->t('Use inclusive language and consider diverse perspectives'),
      'user_impact' => $this->t('Consider broader implications and potential negative effects'),
    ];

    return $guidance_map[$dimension] ?? $this->t('Review content to address ethical concerns');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfiguration(): array {
    return $this->configuration;
  }

  /**
   * {@inheritdoc}
   */
  public function setConfiguration(array $configuration): void {
    // Validate configuration before setting
    $validated_config = $this->validateAndSanitizeConfiguration($configuration);
    $this->configuration = $validated_config + $this->defaultConfiguration();
  }

  /**
   * Validates and sanitizes configuration values.
   *
   * @param array $configuration
   *   The configuration to validate.
   *
   * @return array
   *   The validated and sanitized configuration.
   */
  protected function validateAndSanitizeConfiguration(array $configuration): array {
    $validated = [];

    // Validate and sanitize API settings
    if (isset($configuration['endpoint_url'])) {
      $url = trim($configuration['endpoint_url']);
      
      // Ensure HTTPS-only URLs for security
      if (filter_var($url, FILTER_VALIDATE_URL) && preg_match('/^https:\/\//', $url)) {
        // Additional security validation
        $parsed_url = parse_url($url);
        if ($parsed_url && !empty($parsed_url['host'])) {
          // Prevent localhost/private IP access in production
          $host = $parsed_url['host'];
          if (!$this->isAllowedHost($host)) {
            $this->logger->warning('RAIL guardrail: Potentially unsafe endpoint host blocked: @host', ['@host' => $host]);
          } else {
            $validated['endpoint_url'] = $url;
          }
        }
      } else {
        $this->logger->error('RAIL guardrail: Invalid or non-HTTPS endpoint URL provided');
      }
    }

    if (isset($configuration['api_key'])) {
      $api_key_id = $configuration['api_key'];
      
      // For key_select, we store the key ID, not the actual key value
      // Just validate that it's not empty - the Key module handles the rest
      if (!empty($api_key_id)) {
        $validated['api_key'] = $api_key_id;
      } else {
        $this->logger->error('RAIL guardrail: No API key selected');
      }
    }

    if (isset($configuration['timeout'])) {
      $timeout = (int) $configuration['timeout'];
      $validated['timeout'] = max(10, min(300, $timeout)); // Clamp between 10-300 seconds
    }

    // Validate evaluation settings
    if (isset($configuration['evaluation_mode'])) {
      $valid_modes = ['input', 'output', 'both'];
      if (in_array($configuration['evaluation_mode'], $valid_modes)) {
        $validated['evaluation_mode'] = $configuration['evaluation_mode'];
      }
    }

    if (isset($configuration['action_on_violation'])) {
      $valid_actions = ['stop', 'rewrite', 'warn'];
      if (in_array($configuration['action_on_violation'], $valid_actions)) {
        $validated['action_on_violation'] = $configuration['action_on_violation'];
      }
    }

    if (isset($configuration['overall_threshold'])) {
      $threshold = (float) $configuration['overall_threshold'];
      $validated['overall_threshold'] = max(0.0, min(10.0, $threshold)); // Clamp between 0-10
    }

    if (isset($configuration['include_details'])) {
      $validated['include_details'] = (bool) $configuration['include_details'];
    }

    if (isset($configuration['conversation_mode'])) {
      $valid_modes = ['recent_user', 'recent_exchange', 'user_messages_only', 'full_context'];
      if (in_array($configuration['conversation_mode'], $valid_modes)) {
        $validated['conversation_mode'] = $configuration['conversation_mode'];
      }
    }

    // Validate dimension settings
    if (isset($configuration['enabled_dimensions']) && is_array($configuration['enabled_dimensions'])) {
      $valid_dimensions = array_keys(self::ETHICAL_DIMENSIONS);
      $validated['enabled_dimensions'] = array_intersect($configuration['enabled_dimensions'], $valid_dimensions);
    }

    if (isset($configuration['dimension_weights']) && is_array($configuration['dimension_weights'])) {
      $validated['dimension_weights'] = [];
      foreach ($configuration['dimension_weights'] as $dimension => $weight) {
        if (array_key_exists($dimension, self::ETHICAL_DIMENSIONS)) {
          $weight = (float) $weight;
          $validated['dimension_weights'][$dimension] = max(0.1, min(5.0, $weight)); // Clamp between 0.1-5.0
        }
      }
    }

    if (isset($configuration['dimension_thresholds']) && is_array($configuration['dimension_thresholds'])) {
      $validated['dimension_thresholds'] = [];
      foreach ($configuration['dimension_thresholds'] as $dimension => $threshold) {
        if (array_key_exists($dimension, self::ETHICAL_DIMENSIONS)) {
          $threshold = (float) $threshold;
          $validated['dimension_thresholds'][$dimension] = max(0.0, min(10.0, $threshold)); // Clamp between 0-10
        }
      }
    }

    // Sanitize violation messages
    if (isset($configuration['violation_messages']) && is_array($configuration['violation_messages'])) {
      $validated['violation_messages'] = [];
      foreach ($configuration['violation_messages'] as $dimension => $message) {
        if (array_key_exists($dimension, self::ETHICAL_DIMENSIONS) && is_string($message)) {
          // Sanitize message content
          $sanitized_message = $this->sanitizeViolationMessage($message);
          if (!empty($sanitized_message)) {
            $validated['violation_messages'][$dimension] = $sanitized_message;
          }
        }
      }
    }

    if (isset($configuration['critical_dimensions']) && is_array($configuration['critical_dimensions'])) {
      $valid_dimensions = array_keys(self::ETHICAL_DIMENSIONS);
      $validated['critical_dimensions'] = array_intersect($configuration['critical_dimensions'], $valid_dimensions);
    }

    return $validated;
  }

  /**
   * Checks if a host is allowed for API connections.
   *
   * @param string $host
   *   The hostname to check.
   *
   * @return bool
   *   TRUE if the host is allowed, FALSE otherwise.
   */
  protected function isAllowedHost(string $host): bool {
    // Allow legitimate RAIL API domains
    $allowed_patterns = [
      '/^api\.rail\.com$/',
      '/^.*\.rail\.com$/',
      '/^rail-api\..*\.com$/',
    ];

    foreach ($allowed_patterns as $pattern) {
      if (preg_match($pattern, $host)) {
        return TRUE;
      }
    }

    // Block localhost and private IP ranges in production
    if (filter_var($host, FILTER_VALIDATE_IP)) {
      // Allow public IPs, block private ranges
      if (filter_var($host, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
        return TRUE;
      }
      return FALSE;
    }

    // Block localhost variants
    $blocked_hosts = ['localhost', '127.0.0.1', '::1', '0.0.0.0'];
    if (in_array($host, $blocked_hosts)) {
      return FALSE;
    }

    // For development, allow more hosts
    if (\Drupal::config('system.performance')->get('cache.page.max_age') == 0) {
      return TRUE; // Development mode - allow all hosts
    }

    // In production, be more restrictive
    return FALSE;
  }

  /**
   * Sanitizes violation message content.
   *
   * @param string $message
   *   The violation message to sanitize.
   *
   * @return string
   *   The sanitized message.
   */
  protected function sanitizeViolationMessage(string $message): string {
    // Trim whitespace
    $message = trim($message);
    
    // Remove null bytes and control characters
    $message = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/', '', $message);
    
    // Limit message length
    if (strlen($message) > 500) {
      $message = substr($message, 0, 500);
    }
    
    // Basic HTML entity encoding for safety
    $message = htmlspecialchars($message, ENT_QUOTES | ENT_HTML5, 'UTF-8');
    
    return $message;
  }

  /**
   * Validates import data for security and format correctness.
   *
   * @param string $import_data
   *   The JSON import data to validate.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state to add errors to.
   */
  protected function validateImportData(string $import_data, FormStateInterface $form_state): void {
    // Check for reasonable size limits
    if (strlen($import_data) > 50000) {
      $form_state->setErrorByName('import_export][import_section][import_data', 
        $this->t('Import data is too large (maximum 50KB allowed).'));
      return;
    }

    // Validate JSON format
    $decoded_data = json_decode($import_data, TRUE);
    if (json_last_error() !== JSON_ERROR_NONE) {
      $form_state->setErrorByName('import_export][import_section][import_data', 
        $this->t('Invalid JSON format: @error', ['@error' => json_last_error_msg()]));
      return;
    }

    if (!is_array($decoded_data)) {
      $form_state->setErrorByName('import_export][import_section][import_data', 
        $this->t('Import data must be a JSON object.'));
      return;
    }

    // Validate required fields and structure
    $required_fields = ['version', 'evaluation_mode', 'enabled_dimensions'];
    foreach ($required_fields as $field) {
      if (!isset($decoded_data[$field])) {
        $form_state->setErrorByName('import_export][import_section][import_data', 
          $this->t('Missing required field in import data: @field', ['@field' => $field]));
        return;
      }
    }

    // Validate version compatibility
    if (!isset($decoded_data['version']) || $decoded_data['version'] !== '1.0') {
      $form_state->setErrorByName('import_export][import_section][import_data', 
        $this->t('Unsupported configuration version. Only version 1.0 is supported.'));
      return;
    }

    // Validate evaluation mode
    $valid_modes = ['input', 'output', 'both'];
    if (!in_array($decoded_data['evaluation_mode'], $valid_modes)) {
      $form_state->setErrorByName('import_export][import_section][import_data', 
        $this->t('Invalid evaluation mode in import data.'));
      return;
    }

    // Validate dimensions
    if (!is_array($decoded_data['enabled_dimensions'])) {
      $form_state->setErrorByName('import_export][import_section][import_data', 
        $this->t('Enabled dimensions must be an array.'));
      return;
    }

    $valid_dimensions = array_keys(self::ETHICAL_DIMENSIONS);
    foreach ($decoded_data['enabled_dimensions'] as $dimension) {
      if (!in_array($dimension, $valid_dimensions)) {
        $form_state->setErrorByName('import_export][import_section][import_data', 
          $this->t('Invalid dimension in import data: @dimension', ['@dimension' => $dimension]));
        return;
      }
    }

    // Validate weights if present
    if (isset($decoded_data['dimension_weights'])) {
      if (!is_array($decoded_data['dimension_weights'])) {
        $form_state->setErrorByName('import_export][import_section][import_data', 
          $this->t('Dimension weights must be an array.'));
        return;
      }

      foreach ($decoded_data['dimension_weights'] as $dimension => $weight) {
        if (!in_array($dimension, $valid_dimensions)) {
          $form_state->setErrorByName('import_export][import_section][import_data', 
            $this->t('Invalid dimension in weights: @dimension', ['@dimension' => $dimension]));
          return;
        }
        
        if (!is_numeric($weight) || $weight < 0.1 || $weight > 5.0) {
          $form_state->setErrorByName('import_export][import_section][import_data', 
            $this->t('Invalid weight for dimension @dimension: must be between 0.1 and 5.0', ['@dimension' => $dimension]));
          return;
        }
      }
    }

    // Validate thresholds if present
    if (isset($decoded_data['dimension_thresholds'])) {
      if (!is_array($decoded_data['dimension_thresholds'])) {
        $form_state->setErrorByName('import_export][import_section][import_data', 
          $this->t('Dimension thresholds must be an array.'));
        return;
      }

      foreach ($decoded_data['dimension_thresholds'] as $dimension => $threshold) {
        if (!in_array($dimension, $valid_dimensions)) {
          $form_state->setErrorByName('import_export][import_section][import_data', 
            $this->t('Invalid dimension in thresholds: @dimension', ['@dimension' => $dimension]));
          return;
        }
        
        if (!is_numeric($threshold) || $threshold < 0.0 || $threshold > 10.0) {
          $form_state->setErrorByName('import_export][import_section][import_data', 
            $this->t('Invalid threshold for dimension @dimension: must be between 0.0 and 10.0', ['@dimension' => $dimension]));
          return;
        }
      }
    }

    // Validate overall threshold if present
    if (isset($decoded_data['overall_threshold'])) {
      $overall_threshold = $decoded_data['overall_threshold'];
      if (!is_numeric($overall_threshold) || $overall_threshold < 0.0 || $overall_threshold > 10.0) {
        $form_state->setErrorByName('import_export][import_section][import_data', 
          $this->t('Invalid overall threshold: must be between 0.0 and 10.0'));
        return;
      }
    }
  }

  /**
   * Validates the current configuration for completeness and consistency.
   *
   * @return array
   *   Array with 'valid' boolean and array of 'errors' if invalid.
   */
  public function validateCurrentConfiguration(): array {
    $config = $this->getConfiguration();
    $errors = [];

    // Check API configuration
    if (empty($config['endpoint_url'])) {
      $errors[] = $this->t('API endpoint URL is required.');
    }
    elseif (!filter_var($config['endpoint_url'], FILTER_VALIDATE_URL)) {
      $errors[] = $this->t('API endpoint URL must be a valid URL.');
    }

    if (empty($config['api_key'])) {
      $errors[] = $this->t('API key is required.');
    }

    // Check dimension configuration
    $enabled_dimensions = $config['enabled_dimensions'] ?? [];
    if (empty($enabled_dimensions)) {
      $errors[] = $this->t('At least one dimension must be enabled.');
    }

    // Check that enabled dimensions have valid weights and thresholds
    $dimension_weights = $config['dimension_weights'] ?? [];
    $dimension_thresholds = $config['dimension_thresholds'] ?? [];

    foreach ($enabled_dimensions as $dimension) {
      if (!isset($dimension_weights[$dimension])) {
        $errors[] = $this->t('Weight missing for dimension: @dimension', ['@dimension' => $dimension]);
      }
      elseif ($dimension_weights[$dimension] < 0.1 || $dimension_weights[$dimension] > 5.0) {
        $errors[] = $this->t('Invalid weight for dimension @dimension: must be between 0.1 and 5.0', ['@dimension' => $dimension]);
      }

      if (!isset($dimension_thresholds[$dimension])) {
        $errors[] = $this->t('Threshold missing for dimension: @dimension', ['@dimension' => $dimension]);
      }
      elseif ($dimension_thresholds[$dimension] < 0.0 || $dimension_thresholds[$dimension] > 10.0) {
        $errors[] = $this->t('Invalid threshold for dimension @dimension: must be between 0.0 and 10.0', ['@dimension' => $dimension]);
      }
    }

    // Check overall threshold
    $overall_threshold = $config['overall_threshold'] ?? 7.0;
    if ($overall_threshold < 0.0 || $overall_threshold > 10.0) {
      $errors[] = $this->t('Overall threshold must be between 0.0 and 10.0');
    }

    return [
      'valid' => empty($errors),
      'errors' => $errors,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    $default_dimensions = array_keys(self::ETHICAL_DIMENSIONS);
    
    // Set balanced default weights with higher weights for critical dimensions
    $default_weights = [
      'fairness' => 1.0,
      'safety' => 1.5,      // Critical dimension - higher weight
      'reliability' => 1.0,
      'transparency' => 1.0,
      'privacy' => 1.5,     // Critical dimension - higher weight
      'accountability' => 1.0,
      'inclusivity' => 1.0,
      'user_impact' => 1.0,
    ];
    
    // Set reasonable default thresholds with stricter thresholds for critical dimensions
    $default_thresholds = [
      'fairness' => 7.0,
      'safety' => 8.0,      // Critical dimension - stricter threshold
      'reliability' => 7.0,
      'transparency' => 6.5,
      'privacy' => 8.0,     // Critical dimension - stricter threshold
      'accountability' => 6.5,
      'inclusivity' => 7.0,
      'user_impact' => 6.5,
    ];
    
    // Default violation messages with actionable guidance
    $default_messages = [
      'fairness' => $this->t('Content may contain bias or unfair treatment. Please review for equitable representation.'),
      'safety' => $this->t('Content contains potentially harmful or unsafe elements. Review and modify before proceeding.'),
      'reliability' => $this->t('Content reliability concerns detected. Verify accuracy and consistency.'),
      'transparency' => $this->t('Content lacks clarity or transparency. Consider providing more explanation.'),
      'privacy' => $this->t('Content may compromise privacy or contain sensitive information. Review data handling.'),
      'accountability' => $this->t('Content accountability concerns detected. Ensure responsible AI practices.'),
      'inclusivity' => $this->t('Content may not be inclusive or accessible. Review for broader accessibility.'),
      'user_impact' => $this->t('Content may have negative user impact. Consider broader implications.'),
    ];

    return [
      // API Configuration
      'endpoint_url' => '',
      'api_key' => '',
      'timeout' => 60,
      
      // Evaluation Settings
      'evaluation_mode' => 'both',
      'action_on_violation' => 'stop',
      'overall_threshold' => 7.0,
      'include_details' => TRUE,
      
      // Dimension Configuration
      'enabled_dimensions' => $default_dimensions,
      'dimension_weights' => $default_weights,
      'dimension_thresholds' => $default_thresholds,
      'violation_messages' => $default_messages,
      
      // System Configuration
      'critical_dimensions' => self::CRITICAL_DIMENSIONS,
      
      // Advanced Content Handling
      'conversation_mode' => 'recent_user',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['#attached']['library'][] = 'rail_ai_provider/guardrail_config';
    
    $form['api_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('API Settings'),
      '#open' => TRUE,
    ];

    $form['api_settings']['endpoint_url'] = [
      '#type' => 'url',
      '#title' => $this->t('RAIL API Endpoint URL'),
      '#description' => $this->t('The RAIL API endpoint URL for ethical evaluation. Must use HTTPS for security. <br><strong>Example:</strong> https://api.rail.com/v1/evaluate<br><strong>Need help?</strong> Contact RAIL support for your specific endpoint URL.'),
      '#default_value' => $this->configuration['endpoint_url'] ?? '',
      '#required' => TRUE,
      '#placeholder' => 'https://api.rail.com/v1/evaluate',
    ];
    $form['api_settings']['api_key'] = [
      '#type' => 'key_select',
      '#title' => $this->t('API Key'),
      '#description' => $this->t('Select the RAIL API key for authentication. Keys are managed securely through the Key module.<br><strong>Where to find:</strong> RAIL account dashboard → API Keys<br><strong>Key Management:</strong> <a href="/admin/config/system/keys">Manage Keys</a> to add your RAIL API key.'),
      '#default_value' => $this->configuration['api_key'] ?? 'rail',
      '#required' => TRUE,
    ];

    $form['api_settings']['timeout'] = [
      '#type' => 'number',
      '#title' => $this->t('API Timeout'),
      '#description' => $this->t('Maximum time to wait for API response (in seconds). <br><strong>Recommended:</strong> 60 seconds for most use cases<br><strong>Range:</strong> 10-300 seconds<br><strong>Note:</strong> Higher values allow for complex evaluations but may slow down content processing.'),
      '#min' => 10,
      '#max' => 300,
      '#default_value' => $this->configuration['timeout'] ?? 60,
    ];

    $form['api_settings']['test_connection'] = [
      '#type' => 'button',
      '#value' => $this->t('Test Connection'),
      '#ajax' => [
        'callback' => '::testConnectionCallback',
        'wrapper' => 'test-connection-result',
      ],
    ];

    $form['api_settings']['test_result'] = [
      '#type' => 'markup',
      '#markup' => '<div id="test-connection-result"></div>',
    ];

    $form['evaluation_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Evaluation Settings'),
      '#open' => TRUE,
    ];

    $form['evaluation_settings']['evaluation_mode'] = [
      '#type' => 'radios',
      '#title' => $this->t('Evaluation Mode'),
      '#description' => $this->t('Choose when to perform ethical evaluation of content.<br><strong>Input only:</strong> Evaluate user input before AI processing (prevents problematic prompts)<br><strong>Output only:</strong> Evaluate AI-generated responses (catches problematic outputs)<br><strong>Both:</strong> Comprehensive protection by evaluating both input and output (recommended for most use cases)'),
      '#options' => [
        'input' => $this->t('Input only - Evaluate user input before AI processing'),
        'output' => $this->t('Output only - Evaluate AI-generated responses'),
        'both' => $this->t('Both - Evaluate both input and output (recommended)'),
      ],
      '#default_value' => $this->configuration['evaluation_mode'] ?? 'both',
    ];

    $form['evaluation_settings']['action_on_violation'] = [
      '#type' => 'radios',
      '#title' => $this->t('Action on Violation'),
      '#description' => $this->t('What to do when ethical violations are detected.<br><strong>Stop processing:</strong> Block content that violates ethical standards (recommended for production)<br><strong>Request rewrite:</strong> Ask users to modify content to address violations<br><strong>Log warning:</strong> Allow content but log violations for review (useful for monitoring and development)'),
      '#options' => [
        'stop' => $this->t('Stop processing - Block content that violates ethical standards'),
        'rewrite' => $this->t('Request rewrite - Ask for content modification to address violations'),
        'warn' => $this->t('Log warning and continue - Allow content but log violations'),
      ],
      '#default_value' => $this->configuration['action_on_violation'] ?? 'stop',
    ];

    $form['evaluation_settings']['overall_threshold'] = [
      '#type' => 'range',
      '#title' => $this->t('Overall Weighted Threshold'),
      '#description' => $this->t('Minimum weighted score required to pass (0-10 scale). Lower values are more permissive.<br><strong>Strict:</strong> 8.0+ (high security)<br><strong>Balanced:</strong> 6.5-7.5 (recommended)<br><strong>Permissive:</strong> 5.0-6.5 (development/creative use)<br><strong>Note:</strong> Critical dimensions (Safety, Privacy) always stop processing on violation regardless of this threshold.'),
      '#min' => 0,
      '#max' => 10,
      '#step' => 0.1,
      '#default_value' => $this->configuration['overall_threshold'] ?? 7.0,
      '#field_suffix' => '<span class="range-value-display">' . ($this->configuration['overall_threshold'] ?? 7.0) . '</span>',
      '#attributes' => [
        'class' => ['overall-threshold-slider'],
      ],
    ];

    $form['evaluation_settings']['include_details'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include detailed evaluation results'),
      '#description' => $this->t('Include detailed dimension analysis in evaluation results for debugging.'),
      '#default_value' => $this->configuration['include_details'] ?? TRUE,
    ];

    $form['evaluation_settings']['conversation_mode'] = [
      '#type' => 'radios',
      '#title' => $this->t('Multi-message Conversation Handling'),
      '#description' => $this->t('Choose how to handle conversations with multiple messages for ethical evaluation.<br><strong>Recent user message:</strong> Most efficient, evaluates only the latest user input (recommended)<br><strong>Recent exchange:</strong> Evaluates last user message and assistant response together<br><strong>User messages only:</strong> Evaluates recent user messages (up to 5)<br><strong>Full context:</strong> Most comprehensive but slower, evaluates entire recent conversation (up to 10 messages)'),
      '#options' => [
        'recent_user' => $this->t('Recent user message only - Evaluate only the most recent user message (recommended)'),
        'recent_exchange' => $this->t('Recent exchange - Evaluate the last user message and assistant response together'),
        'user_messages_only' => $this->t('User messages only - Evaluate recent user messages (up to 5)'),
        'full_context' => $this->t('Full context - Evaluate the entire recent conversation (up to 10 messages)'),
      ],
      '#default_value' => $this->configuration['conversation_mode'] ?? 'recent_user',
    ];

    // Dimension configuration with improved UI
    $form['dimensions'] = [
      '#type' => 'details',
      '#title' => $this->t('Ethical Dimensions Configuration'),
      '#description' => $this->t('Configure which ethical dimensions to evaluate and their relative importance. Each dimension evaluates different aspects of ethical AI behavior.<br><strong>Tip:</strong> Start with Safety and Privacy for basic protection, then add other dimensions based on your specific needs.'),
      '#open' => TRUE,
    ];

    $form['dimensions']['dimension_help'] = [
      '#type' => 'markup',
      '#markup' => '<div class="dimension-help">' . 
        $this->t('<strong>How it works:</strong> Enable the dimensions you want to evaluate. Weights determine how much each dimension contributes to the overall score. Critical dimensions (marked with ⚠️) will always stop processing on violation.<br><strong>Quick start:</strong> Use the preset buttons below for common configurations, or hover over the ℹ️ icons for detailed information about each dimension.') . 
        '</div>',
    ];

    // Quick actions for dimension management
    $form['dimensions']['quick_actions'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['dimension-quick-actions']],
    ];

    $form['dimensions']['quick_actions']['enable_all'] = [
      '#type' => 'button',
      '#value' => $this->t('Enable All'),
      '#attributes' => ['class' => ['button--small', 'enable-all-dimensions']],
    ];

    $form['dimensions']['quick_actions']['disable_all'] = [
      '#type' => 'button',
      '#value' => $this->t('Disable All'),
      '#attributes' => ['class' => ['button--small', 'disable-all-dimensions']],
    ];

    $form['dimensions']['quick_actions']['reset_weights'] = [
      '#type' => 'button',
      '#value' => $this->t('Reset Weights'),
      '#attributes' => ['class' => ['button--small', 'reset-weights']],
    ];

    $enabled_dimensions = $this->configuration['enabled_dimensions'] ?? array_keys(self::ETHICAL_DIMENSIONS);
    $dimension_weights = $this->configuration['dimension_weights'] ?? [];
    $dimension_thresholds = $this->configuration['dimension_thresholds'] ?? [];

    // Create a table-like layout for better organization
    $form['dimensions']['table'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Dimension'),
        $this->t('Enable'),
        $this->t('Weight'),
        $this->t('Threshold'),
        $this->t('Description'),
      ],
      '#attributes' => ['class' => ['dimension-config-table']],
    ];

    $dimension_descriptions = [
      'fairness' => [
        'description' => $this->t('Ensures content treats all groups equitably without bias or discrimination.'),
        'tooltip' => $this->t('Evaluates whether content shows bias against protected groups, uses discriminatory language, or promotes unfair treatment. Higher scores indicate more equitable content.'),
        'examples' => $this->t('Examples: Gender-neutral job descriptions, culturally sensitive content, avoiding stereotypes'),
      ],
      'safety' => [
        'description' => $this->t('⚠️ Critical: Prevents harmful, dangerous, or unsafe content.'),
        'tooltip' => $this->t('Detects content that could cause physical, emotional, or psychological harm. This is a critical dimension that will always stop processing on violation.'),
        'examples' => $this->t('Examples: Violence prevention, suicide prevention, avoiding dangerous instructions'),
      ],
      'reliability' => [
        'description' => $this->t('Ensures content is accurate, consistent, and dependable.'),
        'tooltip' => $this->t('Assesses factual accuracy, consistency of information, and reliability of sources. Higher scores indicate more trustworthy content.'),
        'examples' => $this->t('Examples: Fact-checked information, consistent messaging, credible sources'),
      ],
      'transparency' => [
        'description' => $this->t('Promotes clear, understandable, and explainable content.'),
        'tooltip' => $this->t('Evaluates how clearly content explains its reasoning, sources, and limitations. Higher scores indicate more transparent communication.'),
        'examples' => $this->t('Examples: Clear explanations, citing sources, acknowledging uncertainty'),
      ],
      'privacy' => [
        'description' => $this->t('⚠️ Critical: Protects personal information and user privacy.'),
        'tooltip' => $this->t('Detects potential privacy violations, personal data exposure, or consent issues. This is a critical dimension that will always stop processing on violation.'),
        'examples' => $this->t('Examples: Avoiding personal data collection, respecting user consent, data anonymization'),
      ],
      'accountability' => [
        'description' => $this->t('Ensures content supports responsible and traceable AI decisions.'),
        'tooltip' => $this->t('Assesses whether content promotes responsible decision-making and clear attribution of responsibility.'),
        'examples' => $this->t('Examples: Clear attribution, taking responsibility for claims, audit trails'),
      ],
      'inclusivity' => [
        'description' => $this->t('Promotes content that is accessible and inclusive to all users.'),
        'tooltip' => $this->t('Evaluates whether content is accessible to diverse audiences and avoids exclusionary language or concepts.'),
        'examples' => $this->t('Examples: Accessible language, diverse representation, inclusive design'),
      ],
      'user_impact' => [
        'description' => $this->t('Considers the broader impact of content on users and society.'),
        'tooltip' => $this->t('Assesses potential positive or negative effects on individual users and broader society.'),
        'examples' => $this->t('Examples: Promoting wellbeing, considering social impact, avoiding manipulation'),
      ],
    ];

    foreach (self::ETHICAL_DIMENSIONS as $key => $label) {
      $is_critical = in_array($key, self::CRITICAL_DIMENSIONS);
      $dimension_info = $dimension_descriptions[$key];
      
      $form['dimensions']['table'][$key]['dimension'] = [
        '#markup' => '<div class="dimension-label">' .
          '<strong>' . $label . '</strong>' . 
          ($is_critical ? ' <span class="critical-badge">CRITICAL</span>' : '') .
          '<div class="dimension-tooltip" title="' . $dimension_info['tooltip'] . '">ℹ️</div>' .
          '</div>',
      ];

      $form['dimensions']['table'][$key]['enabled'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Enable @dimension', ['@dimension' => $label]),
        '#title_display' => 'invisible',
        '#default_value' => in_array($key, $enabled_dimensions),
        '#attributes' => ['class' => ['dimension-enable-checkbox']],
      ];

      $form['dimensions']['table'][$key]['weight'] = [
        '#type' => 'range',
        '#title' => $this->t('Weight for @dimension', ['@dimension' => $label]),
        '#title_display' => 'invisible',
        '#min' => 0.1,
        '#max' => 5.0,
        '#step' => 0.1,
        '#default_value' => $dimension_weights[$key] ?? 1.0,
        '#field_suffix' => '<span class="range-value-display">' . ($dimension_weights[$key] ?? 1.0) . '</span>',
        '#states' => [
          'visible' => [
            ':input[name="dimensions[table][' . $key . '][enabled]"]' => ['checked' => TRUE],
          ],
        ],
        '#attributes' => [
          'class' => ['dimension-weight-slider'],
          'title' => $this->t('Higher weights make this dimension more important in the overall score'),
        ],
      ];

      $form['dimensions']['table'][$key]['threshold'] = [
        '#type' => 'range',
        '#title' => $this->t('Threshold for @dimension', ['@dimension' => $label]),
        '#title_display' => 'invisible',
        '#min' => 0,
        '#max' => 10,
        '#step' => 0.1,
        '#default_value' => $dimension_thresholds[$key] ?? 7.0,
        '#field_suffix' => '<span class="range-value-display">' . ($dimension_thresholds[$key] ?? 7.0) . '</span>',
        '#states' => [
          'visible' => [
            ':input[name="dimensions[table][' . $key . '][enabled]"]' => ['checked' => TRUE],
          ],
        ],
        '#attributes' => [
          'class' => ['dimension-threshold-slider'],
          'title' => $this->t('Minimum score required to pass this dimension (0-10 scale)'),
        ],
      ];

      $form['dimensions']['table'][$key]['description'] = [
        '#markup' => '<div class="dimension-description">' . 
          '<div class="description-text">' . $dimension_info['description'] . '</div>' .
          '<div class="dimension-examples"><strong>' . $this->t('Examples:') . '</strong> ' . $dimension_info['examples'] . '</div>' .
          '</div>',
      ];
    }

    // Configuration presets
    $form['presets'] = [
      '#type' => 'details',
      '#title' => $this->t('Configuration Presets'),
      '#description' => $this->t('Quick configuration presets for common use cases. These will override your current dimension settings.<br><strong>Strict Mode:</strong> Maximum protection for high-security environments<br><strong>Balanced Mode:</strong> Recommended for most applications<br><strong>Permissive Mode:</strong> Relaxed settings for development or creative use'),
      '#open' => FALSE,
    ];

    $form['presets']['preset_buttons'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['preset-buttons']],
    ];

    $form['presets']['preset_buttons']['strict'] = [
      '#type' => 'button',
      '#value' => $this->t('Strict Mode'),
      '#attributes' => ['class' => ['button', 'apply-preset'], 'data-preset' => 'strict'],
    ];

    $form['presets']['preset_buttons']['balanced'] = [
      '#type' => 'button',
      '#value' => $this->t('Balanced Mode'),
      '#attributes' => ['class' => ['button', 'apply-preset'], 'data-preset' => 'balanced'],
    ];

    $form['presets']['preset_buttons']['permissive'] = [
      '#type' => 'button',
      '#value' => $this->t('Permissive Mode'),
      '#attributes' => ['class' => ['button', 'apply-preset'], 'data-preset' => 'permissive'],
    ];

    // Configuration import/export
    $form['import_export'] = [
      '#type' => 'details',
      '#title' => $this->t('Import/Export Configuration'),
      '#description' => $this->t('Import or export dimension configuration for backup or sharing between environments.<br><strong>Export:</strong> Download your current configuration as JSON<br><strong>Import:</strong> Load a previously exported configuration<br><strong>Note:</strong> API credentials are not included in exports for security reasons.'),
      '#open' => FALSE,
    ];

    $form['import_export']['export_section'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Export Configuration'),
    ];

    $form['import_export']['export_section']['export_button'] = [
      '#type' => 'button',
      '#value' => $this->t('Export Current Configuration'),
      '#ajax' => [
        'callback' => '::exportConfigurationCallback',
        'wrapper' => 'export-result',
      ],
    ];

    $form['import_export']['export_section']['export_result'] = [
      '#type' => 'markup',
      '#markup' => '<div id="export-result"></div>',
    ];

    $form['import_export']['import_section'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Import Configuration'),
    ];

    $form['import_export']['import_section']['import_data'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Configuration JSON'),
      '#description' => $this->t('Paste exported configuration JSON here to import settings.'),
      '#rows' => 8,
      '#placeholder' => $this->t('Paste exported JSON configuration here...'),
    ];

    $form['import_export']['import_section']['import_button'] = [
      '#type' => 'button',
      '#value' => $this->t('Import Configuration'),
      '#ajax' => [
        'callback' => '::importConfigurationCallback',
        'wrapper' => 'import-result',
      ],
    ];

    $form['import_export']['import_section']['import_result'] = [
      '#type' => 'markup',
      '#markup' => '<div id="import-result"></div>',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $values = $form_state->getValues();

    // Validate API settings with security checks and enhanced error messages
    if (empty($values['api_settings']['endpoint_url'])) {
      $form_state->setErrorByName('api_settings][endpoint_url', $this->t('API endpoint URL is required. Please enter the RAIL API endpoint URL provided by your RAIL account.'));
    }
    else {
      // Validate URL format and security
      $url = $values['api_settings']['endpoint_url'];
      if (!filter_var($url, FILTER_VALIDATE_URL)) {
        $form_state->setErrorByName('api_settings][endpoint_url', $this->t('Please enter a valid URL. Example: https://api.rail.com/v1/evaluate'));
      } elseif (!preg_match('/^https:\/\//', $url)) {
        $form_state->setErrorByName('api_settings][endpoint_url', $this->t('Endpoint URL must use HTTPS for security. HTTP connections are not allowed. Please use https:// instead of http://'));
      } else {
        // Additional security validation
        $parsed_url = parse_url($url);
        if (!$parsed_url || empty($parsed_url['host'])) {
          $form_state->setErrorByName('api_settings][endpoint_url', $this->t('Invalid URL structure. Please check the URL format and try again.'));
        } else {
          $host = $parsed_url['host'];
          if (!$this->isAllowedHost($host)) {
            $form_state->setErrorByName('api_settings][endpoint_url', $this->t('The specified host "@host" is not allowed for security reasons. Please use an official RAIL API endpoint.', ['@host' => $host]));
          }
        }
      }
    }

    if (empty($values['api_settings']['api_key'])) {
      $form_state->setErrorByName('api_settings][api_key', $this->t('API key is required. Please enter your RAIL API key from your account dashboard.'));
    }
    else {
      // Validate API key format with helpful messages
      $api_key = trim($values['api_settings']['api_key']);
      if (strlen($api_key) < 10) {
        $form_state->setErrorByName('api_settings][api_key', $this->t('API key appears to be too short (minimum 10 characters). Please check your API key and try again.'));
      } elseif (strlen($api_key) > 500) {
        $form_state->setErrorByName('api_settings][api_key', $this->t('API key appears to be too long (maximum 500 characters). Please check your API key and try again.'));
      } elseif (!preg_match('/^[a-zA-Z0-9\-_\.]+$/', $api_key)) {
        $form_state->setErrorByName('api_settings][api_key', $this->t('API key contains invalid characters. API keys should only contain letters, numbers, hyphens, underscores, and periods.'));
      }
    }

    // Validate timeout with security limits and helpful messages
    $timeout = $values['api_settings']['timeout'] ?? 60;
    if ($timeout < 10 || $timeout > 300) {
      $form_state->setErrorByName('api_settings][timeout', $this->t('Timeout must be between 10 and 300 seconds. Recommended: 60 seconds for most use cases.'));
    }

    // Validate overall threshold with helpful guidance
    $overall_threshold = $values['evaluation_settings']['overall_threshold'] ?? 7.0;
    if ($overall_threshold < 0 || $overall_threshold > 10) {
      $form_state->setErrorByName('evaluation_settings][overall_threshold', $this->t('Overall threshold must be between 0.0 and 10.0. Higher values are more strict (recommended: 7.0).'));
    }

    // Validate that at least one dimension is enabled with helpful guidance
    $enabled_count = 0;
    $dimension_table = $values['dimensions']['table'] ?? [];
    
    foreach (self::ETHICAL_DIMENSIONS as $key => $label) {
      if (!empty($dimension_table[$key]['enabled'])) {
        $enabled_count++;
        
        // Validate weight values with security limits and helpful messages
        $weight = $dimension_table[$key]['weight'] ?? 1.0;
        if ($weight < 0.1 || $weight > 5.0) {
          $form_state->setErrorByName("dimensions][table][$key][weight", 
            $this->t('Weight for @dimension must be between 0.1 and 5.0. Higher weights make this dimension more important (recommended: 1.0-2.0).', ['@dimension' => $label]));
        }
        
        // Validate threshold values with helpful guidance
        $threshold = $dimension_table[$key]['threshold'] ?? 7.0;
        if ($threshold < 0 || $threshold > 10) {
          $form_state->setErrorByName("dimensions][table][$key][threshold", 
            $this->t('Threshold for @dimension must be between 0.0 and 10.0. Higher values are more strict (recommended: 7.0-8.0).', ['@dimension' => $label]));
        }
      }
    }

    if ($enabled_count === 0) {
      $form_state->setErrorByName('dimensions][table', $this->t('At least one ethical dimension must be enabled. We recommend enabling Safety and Privacy at minimum for basic protection.'));
    }

    // Security check: Ensure critical dimensions are handled appropriately
    $critical_enabled = FALSE;
    foreach (self::CRITICAL_DIMENSIONS as $critical_dimension) {
      if (!empty($dimension_table[$critical_dimension]['enabled'])) {
        $critical_enabled = TRUE;
        break;
      }
    }

    if (!$critical_enabled && $enabled_count > 0) {
      \Drupal::messenger()->addWarning($this->t('No critical dimensions (Safety, Privacy) are enabled. Consider enabling at least one critical dimension for comprehensive protection.'));
    }

    // Validate import data if provided
    if (!empty($values['import_export']['import_section']['import_data'])) {
      $import_data = trim($values['import_export']['import_section']['import_data']);
      if (!empty($import_data)) {
        $this->validateImportData($import_data, $form_state);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $values = $form_state->getValues();
    // Process API settings
    $configuration = [
      'endpoint_url' => trim($values['api_settings']['endpoint_url']),
      'api_key' => $values['api_settings']['api_key'], // Don't trim key IDs
      'timeout' => (int) $values['api_settings']['timeout'],
      'evaluation_mode' => $values['evaluation_settings']['evaluation_mode'],
      'action_on_violation' => $values['evaluation_settings']['action_on_violation'],
      'overall_threshold' => (float) $values['evaluation_settings']['overall_threshold'],
      'include_details' => (bool) $values['evaluation_settings']['include_details'],
      'conversation_mode' => $values['evaluation_settings']['conversation_mode'] ?? 'recent_user',
    ];
    // Process dimension settings from table structure
    $enabled_dimensions = [];
    $dimension_weights = [];
    $dimension_thresholds = [];
    $dimension_table = $values['dimensions']['table'] ?? [];

    foreach (self::ETHICAL_DIMENSIONS as $key => $label) {
      if (!empty($dimension_table[$key]['enabled'])) {
        $enabled_dimensions[] = $key;
        $dimension_weights[$key] = (float) $dimension_table[$key]['weight'];
        $dimension_thresholds[$key] = (float) $dimension_table[$key]['threshold'];
      }
    }

    $configuration['enabled_dimensions'] = $enabled_dimensions;
    $configuration['dimension_weights'] = $dimension_weights;
    $configuration['dimension_thresholds'] = $dimension_thresholds;

    // Keep existing violation messages and critical dimensions from defaults
    $defaults = $this->defaultConfiguration();
    $configuration['violation_messages'] = $defaults['violation_messages'];
    $configuration['critical_dimensions'] = self::CRITICAL_DIMENSIONS;

    $this->setConfiguration($configuration);

    // Show success message with summary
    $enabled_count = count($enabled_dimensions);
    $critical_count = count(array_intersect($enabled_dimensions, self::CRITICAL_DIMENSIONS));
    
    \Drupal::messenger()->addStatus($this->t('RAIL Ethical Dimensions guardrail configured successfully. @enabled_count dimensions enabled (@critical_count critical).', [
      '@enabled_count' => $enabled_count,
      '@critical_count' => $critical_count,
    ]));
  }

  /**
   * AJAX callback for testing API connection.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The AJAX response.
   */
  public function testConnectionCallback(array &$form, FormStateInterface $form_state): array {
    $values = $form_state->getValues();
    $endpoint_url = $values['api_settings']['endpoint_url'] ?? '';
    $api_key = $values['api_settings']['api_key'] ?? '';

    if (empty($endpoint_url) || empty($api_key)) {
      $message = $this->t('Please enter both endpoint URL and API key before testing.');
      $class = 'messages--error';
    }
    else {
      $test_result = $this->railEvaluationService->testConnection($endpoint_url, $api_key);
      $message = $test_result['message'];
      $class = $test_result['success'] ? 'messages--status' : 'messages--error';
    }

    return [
      '#type' => 'markup',
      '#markup' => '<div class="messages ' . $class . '">' . $message . '</div>',
    ];
  }

  /**
   * AJAX callback for exporting configuration.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The AJAX response.
   */
  public function exportConfigurationCallback(array &$form, FormStateInterface $form_state): array {
    $current_config = $this->getConfiguration();
    
    // Create exportable configuration (exclude sensitive data)
    $exportable_config = [
      'version' => '1.0',
      'exported_at' => date('c'),
      'evaluation_mode' => $current_config['evaluation_mode'] ?? 'both',
      'action_on_violation' => $current_config['action_on_violation'] ?? 'stop',
      'overall_threshold' => $current_config['overall_threshold'] ?? 7.0,
      'include_details' => $current_config['include_details'] ?? TRUE,
      'enabled_dimensions' => $current_config['enabled_dimensions'] ?? [],
      'dimension_weights' => $current_config['dimension_weights'] ?? [],
      'dimension_thresholds' => $current_config['dimension_thresholds'] ?? [],
      'timeout' => $current_config['timeout'] ?? 60,
    ];

    $json_config = json_encode($exportable_config, JSON_PRETTY_PRINT);
    
    $download_link = '<a href="data:application/json;charset=utf-8,' . urlencode($json_config) . '" download="rail-guardrail-config.json" class="button">' . $this->t('Download Configuration File') . '</a>';
    
    $markup = '<div class="messages messages--status">' . 
      $this->t('Configuration exported successfully.') . 
      '</div>' .
      '<div style="margin-top: 1rem;">' . $download_link . '</div>' .
      '<details style="margin-top: 1rem;"><summary>' . $this->t('View JSON') . '</summary>' .
      '<textarea readonly style="width: 100%; height: 200px; font-family: monospace;">' . $json_config . '</textarea>' .
      '</details>';

    return [
      '#type' => 'markup',
      '#markup' => $markup,
    ];
  }

  /**
   * AJAX callback for importing configuration.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The AJAX response.
   */
  public function importConfigurationCallback(array &$form, FormStateInterface $form_state): array {
    $values = $form_state->getValues();
    $import_data = trim($values['import_export']['import_section']['import_data'] ?? '');

    if (empty($import_data)) {
      return [
        '#type' => 'markup',
        '#markup' => '<div class="messages messages--error">' . $this->t('Please paste configuration JSON to import.') . '</div>',
      ];
    }

    // Validate and parse JSON
    $imported_config = json_decode($import_data, TRUE);
    if (json_last_error() !== JSON_ERROR_NONE) {
      return [
        '#type' => 'markup',
        '#markup' => '<div class="messages messages--error">' . 
          $this->t('Invalid JSON format: @error', ['@error' => json_last_error_msg()]) . 
          '</div>',
      ];
    }

    // Validate configuration structure
    $validation_result = $this->validateImportedConfiguration($imported_config);
    if (!$validation_result['valid']) {
      return [
        '#type' => 'markup',
        '#markup' => '<div class="messages messages--error">' . 
          $this->t('Invalid configuration: @error', ['@error' => $validation_result['error']]) . 
          '</div>',
      ];
    }

    // Apply imported configuration to form
    $this->applyImportedConfiguration($form, $imported_config);

    $summary = $this->generateImportSummary($imported_config);
    
    return [
      '#type' => 'markup',
      '#markup' => '<div class="messages messages--status">' . 
        $this->t('Configuration imported successfully. Please review and save the form.') . 
        '</div>' .
        '<div class="import-summary" style="margin-top: 1rem;">' . $summary . '</div>',
    ];
  }

  /**
   * Validates imported configuration structure and values.
   *
   * @param array $config
   *   The imported configuration array.
   *
   * @return array
   *   Array with 'valid' boolean and 'error' message if invalid.
   */
  protected function validateImportedConfiguration(array $config): array {
    // Check required fields
    $required_fields = ['evaluation_mode', 'action_on_violation', 'overall_threshold'];
    foreach ($required_fields as $field) {
      if (!isset($config[$field])) {
        return ['valid' => FALSE, 'error' => $this->t('Missing required field: @field', ['@field' => $field])];
      }
    }

    // Validate evaluation mode
    $valid_modes = ['input', 'output', 'both'];
    if (!in_array($config['evaluation_mode'], $valid_modes)) {
      return ['valid' => FALSE, 'error' => $this->t('Invalid evaluation mode')];
    }

    // Validate action on violation
    $valid_actions = ['stop', 'rewrite', 'warn'];
    if (!in_array($config['action_on_violation'], $valid_actions)) {
      return ['valid' => FALSE, 'error' => $this->t('Invalid action on violation')];
    }

    // Validate overall threshold
    $threshold = $config['overall_threshold'];
    if (!is_numeric($threshold) || $threshold < 0 || $threshold > 10) {
      return ['valid' => FALSE, 'error' => $this->t('Invalid overall threshold')];
    }

    // Validate dimensions if present
    if (isset($config['enabled_dimensions'])) {
      $valid_dimensions = array_keys(self::ETHICAL_DIMENSIONS);
      foreach ($config['enabled_dimensions'] as $dimension) {
        if (!in_array($dimension, $valid_dimensions)) {
          return ['valid' => FALSE, 'error' => $this->t('Invalid dimension: @dimension', ['@dimension' => $dimension])];
        }
      }
    }

    // Validate weights if present
    if (isset($config['dimension_weights'])) {
      foreach ($config['dimension_weights'] as $dimension => $weight) {
        if (!is_numeric($weight) || $weight < 0.1 || $weight > 5.0) {
          return ['valid' => FALSE, 'error' => $this->t('Invalid weight for @dimension', ['@dimension' => $dimension])];
        }
      }
    }

    // Validate thresholds if present
    if (isset($config['dimension_thresholds'])) {
      foreach ($config['dimension_thresholds'] as $dimension => $threshold) {
        if (!is_numeric($threshold) || $threshold < 0 || $threshold > 10) {
          return ['valid' => FALSE, 'error' => $this->t('Invalid threshold for @dimension', ['@dimension' => $dimension])];
        }
      }
    }

    return ['valid' => TRUE, 'error' => ''];
  }

  /**
   * Applies imported configuration to the form.
   *
   * @param array $form
   *   The form array.
   * @param array $config
   *   The imported configuration.
   */
  protected function applyImportedConfiguration(array &$form, array $config): void {
    // Apply evaluation settings
    if (isset($config['evaluation_mode'])) {
      $form['evaluation_settings']['evaluation_mode']['#default_value'] = $config['evaluation_mode'];
    }
    if (isset($config['action_on_violation'])) {
      $form['evaluation_settings']['action_on_violation']['#default_value'] = $config['action_on_violation'];
    }
    if (isset($config['overall_threshold'])) {
      $form['evaluation_settings']['overall_threshold']['#default_value'] = $config['overall_threshold'];
    }
    if (isset($config['include_details'])) {
      $form['evaluation_settings']['include_details']['#default_value'] = $config['include_details'];
    }

    // Apply dimension settings
    $enabled_dimensions = $config['enabled_dimensions'] ?? [];
    $dimension_weights = $config['dimension_weights'] ?? [];
    $dimension_thresholds = $config['dimension_thresholds'] ?? [];

    foreach (self::ETHICAL_DIMENSIONS as $key => $label) {
      $is_enabled = in_array($key, $enabled_dimensions);
      $form['dimensions']['table'][$key]['enabled']['#default_value'] = $is_enabled;
      
      if (isset($dimension_weights[$key])) {
        $form['dimensions']['table'][$key]['weight']['#default_value'] = $dimension_weights[$key];
      }
      
      if (isset($dimension_thresholds[$key])) {
        $form['dimensions']['table'][$key]['threshold']['#default_value'] = $dimension_thresholds[$key];
      }
    }
  }

  /**
   * Generates a summary of imported configuration.
   *
   * @param array $config
   *   The imported configuration.
   *
   * @return string
   *   HTML summary of the imported configuration.
   */
  protected function generateImportSummary(array $config): string {
    $summary_items = [];
    
    $summary_items[] = $this->t('Evaluation Mode: @mode', ['@mode' => $config['evaluation_mode']]);
    $summary_items[] = $this->t('Action on Violation: @action', ['@action' => $config['action_on_violation']]);
    $summary_items[] = $this->t('Overall Threshold: @threshold', ['@threshold' => $config['overall_threshold']]);
    
    $enabled_count = count($config['enabled_dimensions'] ?? []);
    $summary_items[] = $this->t('Enabled Dimensions: @count', ['@count' => $enabled_count]);
    
    if (isset($config['exported_at'])) {
      $summary_items[] = $this->t('Exported: @date', ['@date' => $config['exported_at']]);
    }

    return '<ul><li>' . implode('</li><li>', $summary_items) . '</li></ul>';
  }

  /**
   * Helper method to log evaluation results with comprehensive monitoring.
   *
   * Implements comprehensive evaluation logging as required by requirement 7.1.
   *
   * @param float|null $start_time
   *   The start time of the evaluation process.
   * @param float|null $api_start_time
   *   The start time of the API call.
   * @param float|null $api_end_time
   *   The end time of the API call.
   * @param string $text
   *   The original text content being evaluated.
   * @param array $enabled_dimensions
   *   The enabled dimensions for evaluation.
   * @param string $context
   *   The evaluation context ('input' or 'output').
   * @param float $weighted_score
   *   The calculated weighted score.
   * @param string $result_type
   *   The result type ('pass', 'stop', 'rewrite').
   * @param array $dimension_scores
   *   The dimension scores from RAIL API.
   * @param array $violations
   *   Array of violations if any.
   */
  protected function logEvaluationWithMonitoring(?float $start_time, ?float $api_start_time, ?float $api_end_time, string $text, array $enabled_dimensions, string $context, float $weighted_score, string $result_type, array $dimension_scores, array $violations): void {
    $processing_time = $start_time ? (microtime(true) - $start_time) * 1000 : 0;
    $api_response_time = ($api_start_time && $api_end_time) ? ($api_end_time - $api_start_time) * 1000 : 0;

    $evaluation_data = [
      'guardrail_id' => $this->getPluginId(),
      'content_length' => strlen($text),
      'enabled_dimensions' => $enabled_dimensions,
      'evaluation_context' => $context,
      'weighted_score' => $weighted_score,
      'overall_threshold' => $this->configuration['overall_threshold'] ?? 7.0,
      'result_type' => $result_type,
      'processing_time' => $processing_time,
      'api_response_time' => $api_response_time,
      'dimension_scores' => $dimension_scores,
      'violations' => $violations,
    ];

    $this->monitoringService->logEvaluationResult($evaluation_data);
  }



}