<?php

namespace Drupal\rail_ai_provider\Plugin\AiProvider;

use Drupal\ai\AiProviderPluginManager;
use Drupal\ai\Attribute\AiProvider;
use Drupal\ai\Base\AiProviderClientBase;
use Drupal\ai\Exception\AiMissingFeatureException;
use Drupal\ai\Exception\AiRateLimitException;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatInterface;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\ai\OperationType\Chat\ChatOutput;
use Drupal\ai\OperationType\Embeddings\EmbeddingsInput;
use Drupal\ai\OperationType\Embeddings\EmbeddingsInterface;
use Drupal\ai\OperationType\Embeddings\EmbeddingsOutput;
use Drupal\ai\OperationType\TranslateText\TranslateTextInput;
use Drupal\ai\OperationType\TranslateText\TranslateTextInterface;
use Drupal\ai\OperationType\TranslateText\TranslateTextOutput;
use Drupal\ai\Plugin\ProviderProxy;
use Drupal\ai\Traits\OperationType\ChatTrait;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Messenger\MessengerTrait;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Template\TwigEnvironment;
use Drupal\language\Entity\ConfigurableLanguage;
use Drupal\rail_ai_provider\RailApi;
use GuzzleHttp\Exception\GuzzleException;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Plugin implementation of the 'Rail' provider.
 */
#[AiProvider(
  id: 'rail_ai',
  label: new TranslatableMarkup('Rail AI'),
)]
class RailAIProvider extends AiProviderClientBase implements
  ContainerFactoryPluginInterface,
  ChatInterface
 {

  use StringTranslationTrait;
  use ChatTrait;
  use MessengerTrait;

  /**
   * AI provider plugin manager.
   *
   * @var \Drupal\ai\AiProviderPluginManager
   */
  protected AiProviderPluginManager $manager;

  /**
   * The Rail Client.
   *
   * @var \Drupal\rail_ai_provider\RailApi
   */
  protected RailApi $client;

  /**
   * API Key.
   *
   * @var string
   */
  protected string $apiKey = '';

  /**
   * We want to add models to the provider dynamically.
   *
   * @var bool
   */
  protected bool $hasPredefinedModels = FALSE;

  /**
   * The Twig engine.
   *
   * @var \Drupal\Core\Template\TwigEnvironment
   */
  protected TwigEnvironment $twig;

  /**
   * Configuration of default chat provider.
   *
   * @var array
   */
  protected array $chatConfiguration;

  /**
   * Lazy-loaded provider that actually performs text translation.
   *
   * @var \Drupal\ai\Plugin\ProviderProxy
   */
  protected ProviderProxy $realTranslator;


  /**
   * Supported Types.
   *
   * @var array
   */
  protected $supportedTypes = [
    'chat' => [
      'label' => 'Chat',
      'filter' => 'text-generation',
    ]
  ];

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->client = $container->get('rail_ai_provider.api');
    $instance->manager = $container->get('ai.provider');
    $instance->twig = $container->get('twig');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function isUsable(?string $operation_type = NULL, array $capabilities = []): bool {
    // If its not configured, it is not usable.
    if (!$this->getConfig()->get('api_key')) {
      return FALSE;
    }
    // If its one of the bundles that Mistral supports its usable.
    if ($operation_type) {
      return in_array($operation_type, $this->getSupportedOperationTypes());
    }
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedOperationTypes(): array {
    return [
      'chat',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getConfig(): ImmutableConfig {
    return $this->configFactory->get('rail_ai_provider.settings');
  }

  /**
   * {@inheritdoc}
   */
  public function getApiDefinition(): array {
    // Load the configuration.
    $definition = Yaml::parseFile($this->moduleHandler->getModule('rail_ai_provider')->getPath() . '/definitions/api_defaults.yml');
    return $definition;
  }

  /**
   * {@inheritdoc}
   */
  public function getModelSettings(string $model_id, array $generalConfig = []): array {
    return $generalConfig;
  }

  /**
   * {@inheritdoc}
   */
  public function setAuthentication(mixed $authentication): void {
    // Set the new API key and reset the client.
    $this->apiKey = $authentication;
    $this->client->setApiToken($this->apiKey);
  }

  /**
   * Gets the raw client.
   *
   * This is the client for inference.
   *
   * @return \Drupal\rail_ai_provider\RailApi
   *   The Rail client.
   */
  public function getClient(): RailApi {
    $this->loadClient();
    return $this->client;
  }

  /**
   * Loads the Rail Client with authentication if not initialized.
   */
  protected function loadClient(): void {
    if (!$this->apiKey) {
      $this->setAuthentication($this->loadApiKey());
    }
    $this->client->setApiToken($this->apiKey);
  }

  /**
   * {@inheritdoc}
   */
  public function chat(array|string|ChatInput $input, string $model_id, array $tags = []): ChatOutput {
    $info = $this->getModelInfo('chat', $model_id);
    if (!$info['rail_endpoint']) {
      throw new AiMissingFeatureException('Rail endpoint is missing.');
    }
    $this->loadClient();
    // Normalize the input if needed.
    $chat_input = $input;
    if ($input instanceof ChatInput) {
      $chat_input = "";
      // Add a warning log if they set an system role.
      if ($this->chatSystemRole) {
        $this->loggerFactory->get('ai')->warning('A chat message with system role was sent with Rail provider. Rail does not support system roles and this was removed.');
      }
      foreach ($input->getMessages() as $message) {
        $chat_input .= $message->getText() . "\n";
        if (count($message->getImages())) {
          throw new AiMissingFeatureException('Images are not supported by Rail.');
        }
      }
    }
    try {
      $response = json_decode($this->client->textGeneration($info['rail_endpoint'], $chat_input), TRUE);
    }
    catch (\Exception $e) {
      // If the rate limit is reach, special error.
      if (strpos($e->getMessage(), 'Rate limit reached') !== FALSE) {
        throw new AiRateLimitException($e->getMessage());
      }
      throw $e;
    }
    $message = new ChatMessage('', json_encode($response, TRUE));
    return new ChatOutput($message, $response, []);
  }

}
