<?php

namespace Drupal\rail_ai_provider;

use GuzzleHttp\Client;

/**
 * Basic Rail API.
 */
class RailApi {

  /**
   * The http client.
   */
  protected Client $client;

  /**
   * API Token.
   */
  private string $apiToken;

  /**
   * The serverless base path.
   */
  private string $serverless = 'https://aes-translate-581568263471.europe-west1.run.app/translate';

  /**
   * Constructs a new Rail object.
   *
   * @param \GuzzleHttp\Client $client
   *   Http client.
   */
  public function __construct(Client $client) {
    $this->client = $client;
  }

  /**
   * Set the API token.
   *
   * @param string $apiToken
   *   The API token.
   */
  public function setApiToken($apiToken) {
    $this->apiToken = $apiToken;
  }

  /**
   * Checks if the api is set.
   *
   * @return bool
   *   If the api is set.
   */
  public function isApiSet() {
    return !empty($this->apiToken);
  }

  /**
   * Makes an Text Generation task call.
   *
   * @param string $endpoint
   *   The endpoint url or model name.
   * @param string $prompt
   *   The prompt to send.
   * @param array $parameters
   *   The parameters to send.
   * @param bool $useCache
   *   Use cache to speed up inference requests.
   * @param bool $waitForModel
   *   Wait for model instead of 503.
   *
   * @return string
   *   The return response undecoded.
   */
  public function textGeneration($endpoint, $prompt, array $parameters = [], $useCache = TRUE, $waitForModel = FALSE) {
    $apiEndPoint = $this->finalEndpoint($endpoint);
    return $this->makeRequest($apiEndPoint, [
      'content' => $prompt,
    ]);
  }

  /**
   * Makes an Translation task call.
   *
   * @param string $endpoint
   *   The endpoint url or model name.
   * @param string $source
   *   The text to translate.
   * @param bool $useCache
   *   Use cache to speed up inference requests.
   * @param bool $waitForModel
   *   Wait for model instead of 503.
   *
   * @return string
   *   The return response undecoded.
   */
  public function translation($endpoint, $source, $useCache = TRUE, $waitForModel = FALSE) {
    $apiEndPoint = $this->finalEndpoint($endpoint);
    return $this->makeRequest($apiEndPoint, [
      'inputs' => $source,
      'use_cache' => $useCache,
      'wait_for_model' => $waitForModel,
    ]);
  }

  /**
   * Is endpoint a serverless endpoint or a dedicated url.
   *
   * @param string $endpoint
   *   The endpoint url or model name.
   *
   * @return string
   *   The final endpoint.
   */
  protected function finalEndpoint($endpoint) {
    // If it has a protocol, it's a dedicated url.
    if (strpos($endpoint, 'https://') === 0 || strpos($endpoint, 'http://') === 0) {
      return $endpoint;
    }
    // Otherwise, it's a serverless endpoint.
    return $this->serverless . $endpoint;
  }

  /**
   * Make Rail API call.
   *
   * @param string $apiEndPoint
   *   The api endpoint.
   * @param string $json
   *   JSON params.
   * @param string $file
   *   A (real) filepath.
   * @param string $method
   *   The http method.
   *
   * @return string|object
   *   The return response.
   */
  protected function makeRequest($apiEndPoint, $json = NULL, $file = NULL, $method = 'POST') {
    if (empty($this->apiToken)) {
      throw new \Exception('No Rail API token found.');
    }

    // We can wait some.
    $options['connect_timeout'] = 120;
    $options['read_timeout'] = 120;
    // Set authorization header.
    $options['headers']['Authorization'] = 'Bearer ' . $this->apiToken;
    if ($json) {
      $options['body'] = json_encode($json);
      $options['headers']['Content-Type'] = 'application/json';
    }

    if ($file) {
      $options['body'] = fopen($file, 'r');
    }

    $res = $this->client->request($method, $apiEndPoint, $options);
    return $res->getBody();
  }

}
