<?php

declare(strict_types=1);

namespace Drupal\rail_ai_provider;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Exception\RequestException;

/**
 * Service for RAIL ethical evaluation API integration.
 */
class RailEthicalEvaluationService {

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $httpClient;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected CacheBackendInterface $cache;

  /**
   * Cache key for API unavailability status.
   */
  protected const UNAVAILABLE_CACHE_KEY = 'rail_api_unavailable';

  /**
   * Cache timeout for API unavailability (5 minutes).
   */
  protected const UNAVAILABLE_CACHE_TIMEOUT = 300;

  /**
   * Default API timeout in seconds.
   */
  protected const DEFAULT_TIMEOUT = 60;

  /**
   * Cache key for authentication failures.
   */
  protected const AUTH_FAILURE_CACHE_KEY = 'rail_api_auth_failure';

  /**
   * Cache timeout for authentication failures (15 minutes).
   */
  protected const AUTH_FAILURE_CACHE_TIMEOUT = 900;

  /**
   * Cache key prefix for evaluation results.
   */
  protected const EVALUATION_CACHE_PREFIX = 'rail_evaluation_';

  /**
   * Cache timeout for evaluation results (10 minutes).
   */
  protected const EVALUATION_CACHE_TIMEOUT = 600;

  /**
   * Cache key for API usage statistics.
   */
  protected const USAGE_STATS_CACHE_KEY = 'rail_api_usage_stats';

  /**
   * Maximum content length for caching (to avoid caching very large content).
   */
  protected const MAX_CACHEABLE_CONTENT_LENGTH = 5000;

  /**
   * Constructs a RailEthicalEvaluationService object.
   *
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger channel factory.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   */
  public function __construct(
    ClientInterface $http_client,
    ConfigFactoryInterface $config_factory,
    LoggerChannelFactoryInterface $logger_factory,
    CacheBackendInterface $cache
  ) {
    $this->httpClient = $http_client;
    $this->configFactory = $config_factory;
    $this->logger = $logger_factory->get('rail_ai_provider');
    $this->cache = $cache;
  }

  /**
   * Checks if the service is properly configured.
   *
   * @return bool
   *   TRUE if configured, FALSE otherwise.
   */
  public function isConfigured(): bool {
    $config = $this->configFactory->get('rail_ai_provider.settings');
    $api_key = $config->get('api_key');
    
    return !empty($api_key);
  }

  /**
   * Checks if the API is currently available.
   *
   * @return bool
   *   TRUE if available, FALSE if recently marked unavailable.
   */
  public function isApiAvailable(): bool {
    $cache_item = $this->cache->get(self::UNAVAILABLE_CACHE_KEY);
    return $cache_item === FALSE;
  }

  /**
   * Checks if authentication has recently failed.
   *
   * @return bool
   *   TRUE if authentication recently failed, FALSE otherwise.
   */
  public function hasRecentAuthFailure(): bool {
    $cache_item = $this->cache->get(self::AUTH_FAILURE_CACHE_KEY);
    return $cache_item !== FALSE;
  }

  /**
   * Marks the API as unavailable for a period of time.
   */
  protected function markApiUnavailable(): void {
    $this->cache->set(
      self::UNAVAILABLE_CACHE_KEY,
      TRUE,
      time() + self::UNAVAILABLE_CACHE_TIMEOUT
    );
    $this->logger->warning('RAIL API marked as unavailable for @timeout seconds due to server errors.', [
      '@timeout' => self::UNAVAILABLE_CACHE_TIMEOUT,
    ]);
  }

  /**
   * Marks authentication as failed for a period of time.
   */
  protected function markAuthFailure(): void {
    $this->cache->set(
      self::AUTH_FAILURE_CACHE_KEY,
      TRUE,
      time() + self::AUTH_FAILURE_CACHE_TIMEOUT
    );
    $this->logger->error('RAIL API authentication failed. Cached failure for @timeout seconds to avoid repeated attempts.', [
      '@timeout' => self::AUTH_FAILURE_CACHE_TIMEOUT,
    ]);
  }

  /**
   * Evaluates content using RAIL ethical dimensions with caching optimization.
   *
   * This method implements performance optimizations including:
   * - Content-based caching for repeated evaluations
   * - Request deduplication to minimize API calls
   * - Configurable timeout handling
   * - API usage tracking and monitoring
   *
   * @param string $content
   *   The content to evaluate.
   * @param array $enabled_dimensions
   *   Array of enabled dimension names.
   * @param array $options
   *   Additional options for the evaluation.
   *
   * @return array|null
   *   The evaluation result array, or NULL if evaluation failed.
   */
  public function evaluateContent(string $content, array $enabled_dimensions, array $options = []): ?array {
    if (!$this->isConfigured()) {
      $this->logger->error('RAIL API not configured. Please configure API credentials.');
      return NULL;
    }

    if (!$this->isApiAvailable()) {
      $this->logger->debug('RAIL API currently marked as unavailable, skipping evaluation (fail-open).');
      return NULL;
    }

    if ($this->hasRecentAuthFailure()) {
      $this->logger->debug('RAIL API authentication recently failed, skipping evaluation (fail-open).');
      return NULL;
    }

    if (empty($content)) {
      $this->logger->debug('Empty content provided for evaluation.');
      return NULL;
    }

    if (empty($enabled_dimensions)) {
      $this->logger->debug('No dimensions enabled for evaluation.');
      return NULL;
    }

    // Check cache for repeated evaluations (performance optimization)
    $cache_key = $this->generateEvaluationCacheKey($content, $enabled_dimensions, $options);
    $cached_result = $this->getCachedEvaluation($cache_key);
    
    if ($cached_result !== NULL) {
      $this->logger->debug('RAIL evaluation cache hit for content hash @hash', [
        '@hash' => substr($cache_key, -8),
      ]);
      $this->trackApiUsage('cache_hit');
      return $cached_result;
    }

    try {
      $start_time = microtime(TRUE);
      
      $request_data = $this->buildRequestData($content, $enabled_dimensions, $options);
      $response = $this->makeApiRequest($request_data, $options);
      
      $response_time = microtime(TRUE) - $start_time;
      
      if ($response) {
        // Cache successful evaluations for performance
        $this->cacheEvaluation($cache_key, $response, $content);
        
        $this->logger->info('RAIL evaluation completed successfully for @dimensions dimensions in @time seconds.', [
          '@dimensions' => count($enabled_dimensions),
          '@time' => number_format($response_time, 3),
        ]);
        
        $this->trackApiUsage('success', $response_time);
        return $response;
      }
      
      $this->trackApiUsage('failure', $response_time);
      return NULL;
    }
    catch (\Exception $e) {
      $this->logger->error('RAIL evaluation failed: @message', [
        '@message' => $e->getMessage(),
      ]);
      
      // Handle different types of failures for fail-open approach
      $this->handleEvaluationFailure($e);
      $this->trackApiUsage('error');
      
      return NULL;
    }
  }

  /**
   * Handles different types of evaluation failures for fail-open approach.
   *
   * @param \Exception $exception
   *   The exception that occurred during evaluation.
   */
  protected function handleEvaluationFailure(\Exception $exception): void {
    if ($exception instanceof RequestException) {
      $response = $exception->getResponse();
      
      if ($response) {
        $status_code = $response->getStatusCode();
        
        // Handle authentication failures
        if ($status_code === 401 || $status_code === 403) {
          $this->markAuthFailure();
          $this->logger->error('RAIL API authentication failed (HTTP @status). Please check your API credentials. Content will be allowed to proceed (fail-open).', [
            '@status' => $status_code,
          ]);
          return;
        }
        
        // Handle server errors
        if (in_array($status_code, [500, 502, 503, 504])) {
          $this->markApiUnavailable();
          $this->logger->warning('RAIL API server error (HTTP @status). API marked as unavailable. Content will be allowed to proceed (fail-open).', [
            '@status' => $status_code,
          ]);
          return;
        }
        
        // Handle rate limiting
        if ($status_code === 429) {
          $this->markApiUnavailable();
          $this->logger->warning('RAIL API rate limit exceeded (HTTP 429). API marked as unavailable temporarily. Content will be allowed to proceed (fail-open).');
          return;
        }
        
        // Handle client errors (4xx)
        if ($status_code >= 400 && $status_code < 500) {
          $this->logger->error('RAIL API client error (HTTP @status): @message. Content will be allowed to proceed (fail-open).', [
            '@status' => $status_code,
            '@message' => $exception->getMessage(),
          ]);
          return;
        }
      }
    }
    
    // Handle timeout exceptions
    if ($exception instanceof \GuzzleHttp\Exception\ConnectException || 
        $exception instanceof \GuzzleHttp\Exception\RequestException) {
      $message = $exception->getMessage();
      
      if (strpos($message, 'timeout') !== FALSE || strpos($message, 'timed out') !== FALSE) {
        $this->logger->warning('RAIL API request timed out: @message. Content will be allowed to proceed (fail-open).', [
          '@message' => $message,
        ]);
        // Don't mark as unavailable for timeouts - might be temporary network issue
        return;
      }
      
      if (strpos($message, 'Connection refused') !== FALSE || strpos($message, 'Could not resolve host') !== FALSE) {
        $this->markApiUnavailable();
        $this->logger->error('RAIL API connection failed: @message. API marked as unavailable. Content will be allowed to proceed (fail-open).', [
          '@message' => $message,
        ]);
        return;
      }
    }
    
    // Handle all other exceptions
    $this->logger->error('RAIL API evaluation failed with unexpected error: @message. Content will be allowed to proceed (fail-open).', [
      '@message' => $exception->getMessage(),
    ]);
  }

  /**
   * Builds the request data for RAIL API call.
   *
   * @param string $content
   *   The content to evaluate.
   * @param array $enabled_dimensions
   *   Array of enabled dimension names.
   * @param array $options
   *   Additional options.
   *
   * @return array
   *   The request data array.
   */
  protected function buildRequestData(string $content, array $enabled_dimensions, array $options = []): array {
    return [
      'content' => $content,
      'dimensions' => $enabled_dimensions,
      'options' => array_merge([
        'include_details' => TRUE,
        'format' => 'json',
      ], $options),
    ];
  }

  /**
   * Makes the actual API request to RAIL with timeout configuration.
   *
   * @param array $request_data
   *   The request data.
   * @param array $options
   *   Additional options including timeout configuration.
   *
   * @return array|null
   *   The parsed response data, or NULL on failure.
   */
  protected function makeApiRequest(array $request_data, array $options = []): ?array {
    $config = $this->configFactory->get('rail_ai_provider.settings');
    $api_key = $config->get('api_key');
    
    if (empty($api_key)) {
      throw new \Exception('RAIL API key not configured.');
    }

    // Use a default endpoint if not configured in guardrail
    // This will be overridden by guardrail-specific configuration
    $endpoint_url = 'https://api.rail.com/v1/evaluate/ethical';

    // Configure timeouts based on options or defaults
    $timeout = $options['timeout'] ?? self::DEFAULT_TIMEOUT;
    $connect_timeout = min(30, $timeout / 2); // Connect timeout should be shorter

    $request_options = [
      'json' => $request_data,
      'headers' => [
        'Authorization' => 'Bearer ' . $api_key,
        'Content-Type' => 'application/json',
        'Accept' => 'application/json',
        'User-Agent' => 'Drupal-RAIL-Provider/1.0',
      ],
      'timeout' => $timeout,
      'connect_timeout' => $connect_timeout,
      'verify' => TRUE, // Enforce SSL certificate verification
    ];

    try {
      $response = $this->httpClient->request('POST', $endpoint_url, $request_options);
      $body = $response->getBody()->getContents();
      
      if (empty($body)) {
        throw new \Exception('Empty response from RAIL API.');
      }

      $data = json_decode($body, TRUE);
      
      if (json_last_error() !== JSON_ERROR_NONE) {
        throw new \Exception('Invalid JSON response from RAIL API: ' . json_last_error_msg());
      }

      return $this->parseApiResponse($data);
    }
    catch (GuzzleException $e) {
      $this->logger->error('RAIL API request failed: @message', [
        '@message' => $e->getMessage(),
      ]);
      throw $e;
    }
  }

  /**
   * Makes an API request with custom endpoint and performance optimizations.
   *
   * @param string $endpoint_url
   *   The custom endpoint URL.
   * @param string $api_key
   *   The API key.
   * @param array $request_data
   *   The request data.
   * @param array $options
   *   Additional options including timeout configuration.
   *
   * @return array|null
   *   The parsed response data, or NULL on failure.
   */
  public function makeCustomApiRequest(string $endpoint_url, string $api_key, array $request_data, array $options = []): ?array {
    // Validate and sanitize inputs
    $validation_result = $this->validateApiRequestInputs($endpoint_url, $api_key, $request_data);
    if (!$validation_result['valid']) {
      $this->logger->error('RAIL API request validation failed: @errors', [
        '@errors' => implode(', ', $validation_result['errors'])
      ]);
      return NULL;
    }

    if (!$this->isApiAvailable()) {
      $this->logger->debug('RAIL API currently marked as unavailable, skipping custom request (fail-open).');
      return NULL;
    }

    if ($this->hasRecentAuthFailure()) {
      $this->logger->debug('RAIL API authentication recently failed, skipping custom request (fail-open).');
      return NULL;
    }

    // Check cache for repeated evaluations (performance optimization)
    $cache_key = $this->generateCustomRequestCacheKey($endpoint_url, $request_data);
    $cached_result = $this->getCachedEvaluation($cache_key);
    
    if ($cached_result !== NULL) {
      $this->logger->debug('RAIL custom API cache hit for request hash @hash', [
        '@hash' => substr($cache_key, -8),
      ]);
      $this->trackApiUsage('cache_hit');
      return $cached_result;
    }

    // Sanitize the request data
    $sanitized_data = $this->sanitizeRequestData($request_data);

    // Configure timeouts based on options or defaults
    $timeout = $options['timeout'] ?? self::DEFAULT_TIMEOUT;
    $connect_timeout = min(30, $timeout / 2); // Connect timeout should be shorter

    $request_options = [
      'json' => $sanitized_data,
      'headers' => [
        'Authorization' => 'Bearer ' . $api_key,
        'Content-Type' => 'application/json',
        'Accept' => 'application/json',
        'User-Agent' => 'Drupal-RAIL-Provider/1.0',
      ],
      'timeout' => $timeout,
      'connect_timeout' => $connect_timeout,
      'verify' => TRUE, // Enforce SSL certificate verification
    ];

    try {
      $start_time = microtime(TRUE);
      
      $response = $this->httpClient->request('POST', $endpoint_url, $request_options);
      $body = $response->getBody()->getContents();
      
      $response_time = microtime(TRUE) - $start_time;
      
      if (empty($body)) {
        throw new \Exception('Empty response from RAIL API.');
      }

      $data = json_decode($body, TRUE);
      
      if (json_last_error() !== JSON_ERROR_NONE) {
        throw new \Exception('Invalid JSON response from RAIL API: ' . json_last_error_msg());
      }

      // Validate API response before processing
      $response_validation = $this->validateApiResponse($data);
      if (!$response_validation['valid']) {
        throw new \Exception('Invalid API response structure: ' . implode(', ', $response_validation['errors']));
      }

      $parsed_response = $this->parseApiResponse($data);
      
      // Cache successful evaluations for performance
      if ($parsed_response) {
        $this->cacheEvaluation($cache_key, $parsed_response, $request_data['content'] ?? '');
      }

      $this->logger->info('Custom RAIL API request completed successfully in @time seconds.', [
        '@time' => number_format($response_time, 3),
      ]);
      
      $this->trackApiUsage('success', $response_time);
      return $parsed_response;
    }
    catch (GuzzleException $e) {
      $this->logger->error('Custom RAIL API request failed: @message', [
        '@message' => $e->getMessage(),
      ]);
      
      // Handle failures for fail-open approach
      $this->handleEvaluationFailure($e);
      $this->trackApiUsage('error');
      
      return NULL;
    }
  }

  /**
   * Validates API request inputs for security and correctness.
   *
   * @param string $endpoint_url
   *   The API endpoint URL.
   * @param string $api_key
   *   The API key.
   * @param array $request_data
   *   The request data.
   *
   * @return array
   *   Validation result with 'valid' boolean and 'errors' array.
   */
  protected function validateApiRequestInputs(string $endpoint_url, string $api_key, array $request_data): array {
    $errors = [];

    // Validate endpoint URL
    if (empty($endpoint_url)) {
      $errors[] = 'Endpoint URL is required';
    } else {
      // Ensure HTTPS-only communication
      if (!filter_var($endpoint_url, FILTER_VALIDATE_URL)) {
        $errors[] = 'Invalid endpoint URL format';
      } elseif (!preg_match('/^https:\/\//', $endpoint_url)) {
        $errors[] = 'Endpoint URL must use HTTPS protocol for security';
      } else {
        // Additional URL security checks
        $parsed_url = parse_url($endpoint_url);
        if (!$parsed_url || empty($parsed_url['host'])) {
          $errors[] = 'Invalid endpoint URL structure';
        } elseif (filter_var($parsed_url['host'], FILTER_VALIDATE_IP)) {
          // Allow IP addresses but log for security monitoring
          $this->logger->info('RAIL API request to IP address: @ip', ['@ip' => $parsed_url['host']]);
        }
      }
    }

    // Validate API key
    if (empty($api_key)) {
      $errors[] = 'API key is required';
    } else {
      // Basic API key format validation
      $api_key = trim($api_key);
      if (strlen($api_key) < 10) {
        $errors[] = 'API key appears to be too short';
      } elseif (strlen($api_key) > 500) {
        $errors[] = 'API key appears to be too long';
      } elseif (!preg_match('/^[a-zA-Z0-9\-_\.]+$/', $api_key)) {
        $errors[] = 'API key contains invalid characters';
      }
    }

    // Validate request data structure
    if (!is_array($request_data)) {
      $errors[] = 'Request data must be an array';
    } else {
      // Validate required fields
      if (!isset($request_data['content'])) {
        $errors[] = 'Content field is required in request data';
      } elseif (!is_string($request_data['content'])) {
        $errors[] = 'Content must be a string';
      } elseif (strlen($request_data['content']) > 100000) {
        $errors[] = 'Content exceeds maximum length (100KB)';
      }

      if (!isset($request_data['dimensions'])) {
        $errors[] = 'Dimensions field is required in request data';
      } elseif (!is_array($request_data['dimensions'])) {
        $errors[] = 'Dimensions must be an array';
      } elseif (empty($request_data['dimensions'])) {
        $errors[] = 'At least one dimension must be specified';
      } else {
        // Validate dimension names
        $valid_dimensions = ['fairness', 'safety', 'reliability', 'transparency', 'privacy', 'accountability', 'inclusivity', 'user_impact'];
        foreach ($request_data['dimensions'] as $dimension) {
          if (!is_string($dimension) || !in_array($dimension, $valid_dimensions)) {
            $errors[] = "Invalid dimension: {$dimension}";
          }
        }
      }

      // Validate options if present
      if (isset($request_data['options']) && !is_array($request_data['options'])) {
        $errors[] = 'Options must be an array if provided';
      }
    }

    return [
      'valid' => empty($errors),
      'errors' => $errors,
    ];
  }

  /**
   * Sanitizes request data to prevent injection attacks.
   *
   * @param array $request_data
   *   The request data to sanitize.
   *
   * @return array
   *   The sanitized request data.
   */
  protected function sanitizeRequestData(array $request_data): array {
    $sanitized = [];

    // Sanitize content
    if (isset($request_data['content'])) {
      $content = $request_data['content'];
      
      // Remove null bytes and other dangerous characters
      $content = str_replace(["\0", "\x0B"], '', $content);
      
      // Normalize line endings
      $content = str_replace(["\r\n", "\r"], "\n", $content);
      
      // Limit content length for security
      if (strlen($content) > 100000) {
        $content = substr($content, 0, 100000);
        $this->logger->warning('RAIL API request content truncated to 100KB for security');
      }
      
      $sanitized['content'] = $content;
    }

    // Sanitize dimensions
    if (isset($request_data['dimensions']) && is_array($request_data['dimensions'])) {
      $valid_dimensions = ['fairness', 'safety', 'reliability', 'transparency', 'privacy', 'accountability', 'inclusivity', 'user_impact'];
      $sanitized['dimensions'] = array_intersect($request_data['dimensions'], $valid_dimensions);
    }

    // Sanitize options
    if (isset($request_data['options']) && is_array($request_data['options'])) {
      $sanitized_options = [];
      
      // Only allow specific option keys
      $allowed_options = ['include_details', 'format'];
      foreach ($allowed_options as $option_key) {
        if (isset($request_data['options'][$option_key])) {
          $value = $request_data['options'][$option_key];
          
          if ($option_key === 'include_details') {
            $sanitized_options[$option_key] = (bool) $value;
          } elseif ($option_key === 'format') {
            $sanitized_options[$option_key] = in_array($value, ['json', 'xml']) ? $value : 'json';
          }
        }
      }
      
      $sanitized['options'] = $sanitized_options;
    }

    return $sanitized;
  }

  /**
   * Validates API response structure before processing.
   *
   * @param array $response_data
   *   The response data from the API.
   *
   * @return array
   *   Validation result with 'valid' boolean and 'errors' array.
   */
  protected function validateApiResponse(array $response_data): array {
    $errors = [];

    // Check for required top-level structure
    if (!isset($response_data['result'])) {
      $errors[] = 'Missing required "result" field in API response';
      return ['valid' => false, 'errors' => $errors];
    }

    $result = $response_data['result'];
    if (!is_array($result)) {
      $errors[] = 'Result field must be an array';
      return ['valid' => false, 'errors' => $errors];
    }

    // Validate dimension scores structure
    if (!isset($result['dimension_scores'])) {
      $errors[] = 'Missing required "dimension_scores" field in result';
    } elseif (!is_array($result['dimension_scores'])) {
      $errors[] = 'Dimension scores must be an array';
    } else {
      // Validate individual dimension scores
      foreach ($result['dimension_scores'] as $dimension => $score_data) {
        if (!is_string($dimension)) {
          $errors[] = 'Dimension names must be strings';
          continue;
        }
        
        if (!is_array($score_data)) {
          $errors[] = "Score data for dimension '{$dimension}' must be an array";
          continue;
        }
        
        if (!isset($score_data['score'])) {
          $errors[] = "Missing score for dimension '{$dimension}'";
          continue;
        }
        
        $score = $score_data['score'];
        if (!is_numeric($score)) {
          $errors[] = "Score for dimension '{$dimension}' must be numeric";
        } elseif ($score < 0 || $score > 10) {
          $errors[] = "Score for dimension '{$dimension}' must be between 0 and 10";
        }
      }
    }

    // Validate overall score if present
    if (isset($result['rail_score'])) {
      if (!is_array($result['rail_score'])) {
        $errors[] = 'Rail score must be an array';
      } elseif (isset($result['rail_score']['score'])) {
        $overall_score = $result['rail_score']['score'];
        if (!is_numeric($overall_score)) {
          $errors[] = 'Overall rail score must be numeric';
        } elseif ($overall_score < 0 || $overall_score > 10) {
          $errors[] = 'Overall rail score must be between 0 and 10';
        }
      }
    }

    return [
      'valid' => empty($errors),
      'errors' => $errors,
    ];
  }

  /**
   * Parses and validates the API response.
   *
   * @param array $response_data
   *   The raw response data from API.
   *
   * @return array
   *   The parsed and validated response.
   *
   * @throws \Exception
   *   If response structure is invalid.
   */
  protected function parseApiResponse(array $response_data): array {
    // Validate response structure
    if (!isset($response_data['result'])) {
      throw new \Exception('Invalid RAIL API response: missing result field.');
    }

    $result = $response_data['result'];
    
    // Extract overall score
    $overall_score = $result['rail_score']['score'] ?? 0;
    
    // Extract dimension scores
    $dimension_scores = [];
    if (isset($result['dimension_scores']) && is_array($result['dimension_scores'])) {
      foreach ($result['dimension_scores'] as $dimension => $score_data) {
        if (is_array($score_data) && isset($score_data['score'])) {
          $dimension_scores[$dimension] = [
            'score' => (float) $score_data['score'],
            'details' => $score_data['details'] ?? '',
          ];
        }
      }
    }

    if (empty($dimension_scores)) {
      throw new \Exception('Invalid RAIL API response: no dimension scores found.');
    }

    return [
      'overall_score' => (float) $overall_score,
      'dimension_scores' => $dimension_scores,
      'metadata' => $response_data['metadata'] ?? [],
    ];
  }

  /**
   * Tests the API connection with given credentials.
   *
   * @param string $endpoint_url
   *   The API endpoint URL.
   * @param string $api_key
   *   The API key.
   *
   * @return array
   *   Array with 'success' boolean and 'message' string.
   */
  public function testConnection(string $endpoint_url, string $api_key): array {
    if (empty($endpoint_url) || empty($api_key)) {
      return [
        'success' => FALSE,
        'message' => 'Endpoint URL and API key are required.',
      ];
    }

    // Test with minimal content
    $test_data = $this->buildRequestData('Test content for connection verification.', ['safety']);

    try {
      $result = $this->makeCustomApiRequest($endpoint_url, $api_key, $test_data);
      
      if ($result) {
        return [
          'success' => TRUE,
          'message' => 'Connection successful. RAIL API is responding correctly.',
        ];
      }
      else {
        return [
          'success' => FALSE,
          'message' => 'Connection failed. Please check your endpoint URL and API key.',
        ];
      }
    }
    catch (\Exception $e) {
      return [
        'success' => FALSE,
        'message' => 'Connection test failed: ' . $e->getMessage(),
      ];
    }
  }

  /**
   * Gets API usage statistics with comprehensive tracking.
   *
   * @return array
   *   Array of usage statistics including performance metrics.
   */
  public function getUsageStatistics(): array {
    $cache_item = $this->cache->get(self::USAGE_STATS_CACHE_KEY);
    
    if ($cache_item === FALSE) {
      // Initialize default statistics
      return [
        'total_calls' => 0,
        'successful_calls' => 0,
        'failed_calls' => 0,
        'error_calls' => 0,
        'cache_hits' => 0,
        'average_response_time' => 0,
        'min_response_time' => 0,
        'max_response_time' => 0,
        'last_reset' => time(),
      ];
    }
    
    return $cache_item->data;
  }

  /**
   * Tracks API usage for monitoring and optimization.
   *
   * @param string $type
   *   The type of API call ('success', 'failure', 'error', 'cache_hit').
   * @param float $response_time
   *   The response time in seconds (optional).
   */
  protected function trackApiUsage(string $type, float $response_time = 0): void {
    $stats = $this->getUsageStatistics();
    
    // Update counters
    $stats['total_calls']++;
    
    switch ($type) {
      case 'success':
        $stats['successful_calls']++;
        $this->updateResponseTimeStats($stats, $response_time);
        break;
        
      case 'failure':
        $stats['failed_calls']++;
        $this->updateResponseTimeStats($stats, $response_time);
        break;
        
      case 'error':
        $stats['error_calls']++;
        break;
        
      case 'cache_hit':
        $stats['cache_hits']++;
        break;
    }
    
    // Cache the updated statistics (expire after 1 hour)
    $this->cache->set(self::USAGE_STATS_CACHE_KEY, $stats, time() + 3600);
    
    // Log performance warnings if needed
    if ($type === 'success' && $response_time > 10) {
      $this->logger->warning('RAIL API slow response detected: @time seconds', [
        '@time' => number_format($response_time, 3),
      ]);
    }
  }

  /**
   * Updates response time statistics.
   *
   * @param array &$stats
   *   The statistics array to update (passed by reference).
   * @param float $response_time
   *   The response time in seconds.
   */
  protected function updateResponseTimeStats(array &$stats, float $response_time): void {
    if ($response_time <= 0) {
      return;
    }
    
    // Update average response time
    $total_timed_calls = $stats['successful_calls'] + $stats['failed_calls'];
    if ($total_timed_calls === 1) {
      $stats['average_response_time'] = $response_time;
      $stats['min_response_time'] = $response_time;
      $stats['max_response_time'] = $response_time;
    } else {
      // Calculate running average
      $previous_total = ($total_timed_calls - 1) * $stats['average_response_time'];
      $stats['average_response_time'] = ($previous_total + $response_time) / $total_timed_calls;
      
      // Update min/max
      $stats['min_response_time'] = min($stats['min_response_time'], $response_time);
      $stats['max_response_time'] = max($stats['max_response_time'], $response_time);
    }
  }

  /**
   * Generates a cache key for evaluation results.
   *
   * @param string $content
   *   The content being evaluated.
   * @param array $enabled_dimensions
   *   The enabled dimensions.
   * @param array $options
   *   Additional options.
   *
   * @return string
   *   The cache key.
   */
  protected function generateEvaluationCacheKey(string $content, array $enabled_dimensions, array $options = []): string {
    // Create a hash of the content and configuration for caching
    $cache_data = [
      'content_hash' => hash('sha256', $content),
      'dimensions' => sort($enabled_dimensions) ? $enabled_dimensions : $enabled_dimensions,
      'options' => $options,
    ];
    
    return self::EVALUATION_CACHE_PREFIX . hash('sha256', serialize($cache_data));
  }

  /**
   * Generates a cache key for custom API requests.
   *
   * @param string $endpoint_url
   *   The API endpoint URL.
   * @param array $request_data
   *   The request data.
   *
   * @return string
   *   The cache key.
   */
  protected function generateCustomRequestCacheKey(string $endpoint_url, array $request_data): string {
    $cache_data = [
      'endpoint' => $endpoint_url,
      'content_hash' => hash('sha256', $request_data['content'] ?? ''),
      'dimensions' => $request_data['dimensions'] ?? [],
      'options' => $request_data['options'] ?? [],
    ];
    
    return self::EVALUATION_CACHE_PREFIX . 'custom_' . hash('sha256', serialize($cache_data));
  }

  /**
   * Gets cached evaluation result if available.
   *
   * @param string $cache_key
   *   The cache key.
   *
   * @return array|null
   *   The cached result or NULL if not found.
   */
  protected function getCachedEvaluation(string $cache_key): ?array {
    $cache_item = $this->cache->get($cache_key);
    
    if ($cache_item !== FALSE && is_array($cache_item->data)) {
      return $cache_item->data;
    }
    
    return NULL;
  }

  /**
   * Caches an evaluation result for performance optimization.
   *
   * @param string $cache_key
   *   The cache key.
   * @param array $result
   *   The evaluation result to cache.
   * @param string $content
   *   The original content (for size checking).
   */
  protected function cacheEvaluation(string $cache_key, array $result, string $content): void {
    // Only cache results for reasonably sized content to avoid memory issues
    if (strlen($content) > self::MAX_CACHEABLE_CONTENT_LENGTH) {
      return;
    }
    
    // Add cache metadata
    $cached_result = $result;
    $cached_result['_cache_metadata'] = [
      'cached_at' => time(),
      'content_length' => strlen($content),
    ];
    
    $this->cache->set(
      $cache_key,
      $cached_result,
      time() + self::EVALUATION_CACHE_TIMEOUT
    );
  }

  /**
   * Resets API usage statistics.
   */
  public function resetUsageStatistics(): void {
    $this->cache->delete(self::USAGE_STATS_CACHE_KEY);
    $this->logger->info('RAIL API usage statistics have been reset.');
  }

  /**
   * Clears evaluation cache.
   */
  public function clearEvaluationCache(): void {
    // Clear all evaluation cache entries
    $this->cache->deleteMultiple([]);
    $this->logger->info('RAIL evaluation cache has been cleared.');
  }

}