<?php

declare(strict_types=1);

namespace Drupal\rail_ai_provider;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Cache\CacheBackendInterface;

/**
 * Service for monitoring and logging RAIL guardrail performance and violations.
 * 
 * This service implements comprehensive evaluation logging, dimension-specific
 * violation logging, integration with Drupal's logging system, and performance
 * monitoring capabilities as required by task 10.2.
 * 
 * Requirements: 7.1, 7.2, 7.4
 */
class RailGuardrailMonitoringService {

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected LoggerChannelFactoryInterface $loggerFactory;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected StateInterface $state;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected CacheBackendInterface $cache;

  /**
   * The logger channel for RAIL guardrail monitoring.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * Constructs a RailGuardrailMonitoringService object.
   *
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   */
  public function __construct(
    LoggerChannelFactoryInterface $logger_factory,
    StateInterface $state,
    CacheBackendInterface $cache
  ) {
    $this->loggerFactory = $logger_factory;
    $this->state = $state;
    $this->cache = $cache;
    $this->logger = $logger_factory->get('rail_guardrail_monitoring');
  }

  /**
   * Logs comprehensive evaluation details.
   *
   * Implements comprehensive evaluation logging as required by requirement 7.1.
   * Logs evaluation results with appropriate detail levels for monitoring and debugging.
   *
   * @param array $evaluation_data
   *   The evaluation data containing:
   *   - guardrail_id: The guardrail instance ID
   *   - content_length: Length of evaluated content
   *   - enabled_dimensions: Array of enabled dimensions
   *   - evaluation_context: 'input' or 'output'
   *   - weighted_score: The calculated weighted score
   *   - overall_threshold: The threshold used
   *   - result_type: 'pass', 'stop', 'rewrite'
   *   - processing_time: Time taken for evaluation in milliseconds
   *   - api_response_time: Time taken for API call in milliseconds
   *   - dimension_scores: Individual dimension scores
   *   - violations: Array of violations if any
   */
  public function logEvaluationResult(array $evaluation_data): void {
    $context = [
      'guardrail_id' => $evaluation_data['guardrail_id'] ?? 'unknown',
      'content_length' => $evaluation_data['content_length'] ?? 0,
      'enabled_dimensions' => $evaluation_data['enabled_dimensions'] ?? [],
      'evaluation_context' => $evaluation_data['evaluation_context'] ?? 'unknown',
      'weighted_score' => $evaluation_data['weighted_score'] ?? 0.0,
      'overall_threshold' => $evaluation_data['overall_threshold'] ?? 0.0,
      'result_type' => $evaluation_data['result_type'] ?? 'unknown',
      'processing_time' => $evaluation_data['processing_time'] ?? 0,
      'api_response_time' => $evaluation_data['api_response_time'] ?? 0,
      'timestamp' => time(),
    ];

    // Log basic evaluation result
    $this->logger->info('RAIL guardrail evaluation completed: @result_type (@context, score: @weighted_score/@threshold, time: @processing_time ms)', [
      '@result_type' => $context['result_type'],
      '@context' => $context['evaluation_context'],
      '@weighted_score' => number_format($context['weighted_score'], 2),
      '@threshold' => number_format($context['overall_threshold'], 2),
      '@processing_time' => $context['processing_time'],
    ]);

    // Log detailed evaluation data for debugging
    $this->logger->debug('RAIL guardrail evaluation details', $context);

    // Update performance metrics
    $this->updatePerformanceMetrics($evaluation_data);

    // Log dimension-specific details if available
    if (!empty($evaluation_data['dimension_scores'])) {
      $this->logDimensionScores($evaluation_data['dimension_scores'], $context);
    }

    // Log violations if any
    if (!empty($evaluation_data['violations'])) {
      $this->logViolationDetails($evaluation_data['violations'], $context);
    }
  }

  /**
   * Logs dimension-specific violation details.
   *
   * Implements dimension-specific violation logging as required by requirement 7.2.
   * Logs specific information about which dimensions failed and their scores.
   *
   * @param array $violations
   *   Array of violation information.
   * @param array $context
   *   The evaluation context.
   */
  public function logViolationDetails(array $violations, array $context): void {
    foreach ($violations as $violation) {
      $dimension = $violation['dimension'] ?? 'unknown';
      $score = $violation['score'] ?? 0.0;
      $threshold = $violation['threshold'] ?? 0.0;
      $weight = $violation['weight'] ?? 1.0;
      $severity = $violation['severity'] ?? 'medium';

      $violation_context = array_merge($context, [
        'dimension' => $dimension,
        'dimension_score' => $score,
        'dimension_threshold' => $threshold,
        'dimension_weight' => $weight,
        'violation_severity' => $severity,
        'score_gap' => $threshold - $score,
      ]);

      // Log violation with appropriate severity level
      if ($severity === 'critical' || in_array($dimension, ['safety', 'privacy'])) {
        $this->logger->error('RAIL guardrail CRITICAL violation: @dimension (@score/@threshold, weight: @weight)', [
          '@dimension' => $dimension,
          '@score' => number_format($score, 2),
          '@threshold' => number_format($threshold, 2),
          '@weight' => number_format($weight, 2),
        ]);
      } elseif ($severity === 'high') {
        $this->logger->warning('RAIL guardrail HIGH violation: @dimension (@score/@threshold, weight: @weight)', [
          '@dimension' => $dimension,
          '@score' => number_format($score, 2),
          '@threshold' => number_format($threshold, 2),
          '@weight' => number_format($weight, 2),
        ]);
      } else {
        $this->logger->notice('RAIL guardrail violation: @dimension (@score/@threshold, weight: @weight)', [
          '@dimension' => $dimension,
          '@score' => number_format($score, 2),
          '@threshold' => number_format($threshold, 2),
          '@weight' => number_format($weight, 2),
        ]);
      }

      // Log detailed violation context for debugging
      $this->logger->debug('RAIL guardrail violation details', $violation_context);

      // Update violation statistics
      $this->updateViolationStatistics($dimension, $severity, $violation_context);
    }
  }

  /**
   * Logs individual dimension scores for analysis.
   *
   * @param array $dimension_scores
   *   The dimension scores from RAIL API.
   * @param array $context
   *   The evaluation context.
   */
  protected function logDimensionScores(array $dimension_scores, array $context): void {
    foreach ($dimension_scores as $dimension => $score_data) {
      $score = is_array($score_data) ? ($score_data['score'] ?? 0.0) : $score_data;
      
      $dimension_context = array_merge($context, [
        'dimension' => $dimension,
        'dimension_score' => $score,
      ]);

      $this->logger->debug('RAIL dimension score: @dimension = @score', [
        '@dimension' => $dimension,
        '@score' => number_format($score, 2),
      ]);
    }
  }

  /**
   * Updates performance metrics for monitoring.
   *
   * Implements performance monitoring capabilities as required by requirement 7.4.
   * Tracks API usage statistics, response times, and system performance.
   *
   * @param array $evaluation_data
   *   The evaluation data.
   */
  protected function updatePerformanceMetrics(array $evaluation_data): void {
    $metrics_key = 'rail_guardrail_performance_metrics';
    $current_metrics = $this->state->get($metrics_key, [
      'total_evaluations' => 0,
      'total_processing_time' => 0,
      'total_api_time' => 0,
      'evaluations_by_context' => ['input' => 0, 'output' => 0],
      'evaluations_by_result' => ['pass' => 0, 'stop' => 0, 'rewrite' => 0],
      'average_processing_time' => 0,
      'average_api_time' => 0,
      'last_updated' => 0,
    ]);

    // Update counters
    $current_metrics['total_evaluations']++;
    $current_metrics['total_processing_time'] += $evaluation_data['processing_time'] ?? 0;
    $current_metrics['total_api_time'] += $evaluation_data['api_response_time'] ?? 0;
    
    $context = $evaluation_data['evaluation_context'] ?? 'unknown';
    if (isset($current_metrics['evaluations_by_context'][$context])) {
      $current_metrics['evaluations_by_context'][$context]++;
    }
    
    $result_type = $evaluation_data['result_type'] ?? 'unknown';
    if (isset($current_metrics['evaluations_by_result'][$result_type])) {
      $current_metrics['evaluations_by_result'][$result_type]++;
    }

    // Calculate averages
    if ($current_metrics['total_evaluations'] > 0) {
      $current_metrics['average_processing_time'] = $current_metrics['total_processing_time'] / $current_metrics['total_evaluations'];
      $current_metrics['average_api_time'] = $current_metrics['total_api_time'] / $current_metrics['total_evaluations'];
    }

    $current_metrics['last_updated'] = time();

    // Save updated metrics
    $this->state->set($metrics_key, $current_metrics);

    // Log performance alerts if needed
    $this->checkPerformanceAlerts($current_metrics, $evaluation_data);
  }

  /**
   * Updates violation statistics for monitoring trends.
   *
   * @param string $dimension
   *   The dimension that violated.
   * @param string $severity
   *   The violation severity.
   * @param array $context
   *   The violation context.
   */
  protected function updateViolationStatistics(string $dimension, string $severity, array $context): void {
    $stats_key = 'rail_guardrail_violation_stats';
    $current_stats = $this->state->get($stats_key, [
      'total_violations' => 0,
      'violations_by_dimension' => [],
      'violations_by_severity' => ['critical' => 0, 'high' => 0, 'medium' => 0, 'low' => 0],
      'last_violation' => 0,
    ]);

    // Update counters
    $current_stats['total_violations']++;
    
    if (!isset($current_stats['violations_by_dimension'][$dimension])) {
      $current_stats['violations_by_dimension'][$dimension] = 0;
    }
    $current_stats['violations_by_dimension'][$dimension]++;
    
    if (isset($current_stats['violations_by_severity'][$severity])) {
      $current_stats['violations_by_severity'][$severity]++;
    }
    
    $current_stats['last_violation'] = time();

    // Save updated statistics
    $this->state->set($stats_key, $current_stats);
  }

  /**
   * Checks for performance alerts and logs warnings if thresholds are exceeded.
   *
   * @param array $metrics
   *   Current performance metrics.
   * @param array $evaluation_data
   *   The current evaluation data.
   */
  protected function checkPerformanceAlerts(array $metrics, array $evaluation_data): void {
    // Alert if processing time is too high
    $processing_time = $evaluation_data['processing_time'] ?? 0;
    if ($processing_time > 5000) { // 5 seconds
      $this->logger->warning('RAIL guardrail performance alert: High processing time (@time ms)', [
        '@time' => $processing_time,
      ]);
    }

    // Alert if API response time is too high
    $api_time = $evaluation_data['api_response_time'] ?? 0;
    if ($api_time > 10000) { // 10 seconds
      $this->logger->warning('RAIL guardrail performance alert: High API response time (@time ms)', [
        '@time' => $api_time,
      ]);
    }

    // Alert if average processing time is increasing
    if ($metrics['total_evaluations'] > 10 && $metrics['average_processing_time'] > 2000) { // 2 seconds average
      $this->logger->notice('RAIL guardrail performance notice: Average processing time is @avg ms over @count evaluations', [
        '@avg' => number_format($metrics['average_processing_time'], 0),
        '@count' => $metrics['total_evaluations'],
      ]);
    }
  }

  /**
   * Gets current performance metrics.
   *
   * @return array
   *   The current performance metrics.
   */
  public function getPerformanceMetrics(): array {
    return $this->state->get('rail_guardrail_performance_metrics', []);
  }

  /**
   * Gets current violation statistics.
   *
   * @return array
   *   The current violation statistics.
   */
  public function getViolationStatistics(): array {
    return $this->state->get('rail_guardrail_violation_stats', []);
  }

  /**
   * Resets performance metrics and violation statistics.
   */
  public function resetStatistics(): void {
    $this->state->delete('rail_guardrail_performance_metrics');
    $this->state->delete('rail_guardrail_violation_stats');
    $this->logger->info('RAIL guardrail statistics reset');
  }

  /**
   * Logs API usage for billing and monitoring purposes.
   *
   * @param array $api_usage_data
   *   API usage data containing:
   *   - endpoint: The API endpoint used
   *   - request_size: Size of the request in bytes
   *   - response_size: Size of the response in bytes
   *   - dimensions_requested: Number of dimensions evaluated
   *   - success: Whether the API call was successful
   */
  public function logApiUsage(array $api_usage_data): void {
    $usage_context = [
      'endpoint' => $api_usage_data['endpoint'] ?? 'unknown',
      'request_size' => $api_usage_data['request_size'] ?? 0,
      'response_size' => $api_usage_data['response_size'] ?? 0,
      'dimensions_requested' => $api_usage_data['dimensions_requested'] ?? 0,
      'success' => $api_usage_data['success'] ?? false,
      'timestamp' => time(),
    ];

    $this->logger->info('RAIL API usage: @endpoint (@dimensions dimensions, @request_size bytes request, success: @success)', [
      '@endpoint' => $usage_context['endpoint'],
      '@dimensions' => $usage_context['dimensions_requested'],
      '@request_size' => $usage_context['request_size'],
      '@success' => $usage_context['success'] ? 'yes' : 'no',
    ]);

    // Update API usage statistics
    $this->updateApiUsageStatistics($usage_context);
  }

  /**
   * Updates API usage statistics.
   *
   * @param array $usage_context
   *   The API usage context.
   */
  protected function updateApiUsageStatistics(array $usage_context): void {
    $stats_key = 'rail_api_usage_stats';
    $current_stats = $this->state->get($stats_key, [
      'total_requests' => 0,
      'successful_requests' => 0,
      'failed_requests' => 0,
      'total_request_bytes' => 0,
      'total_response_bytes' => 0,
      'total_dimensions_evaluated' => 0,
      'last_request' => 0,
    ]);

    // Update counters
    $current_stats['total_requests']++;
    
    if ($usage_context['success']) {
      $current_stats['successful_requests']++;
    } else {
      $current_stats['failed_requests']++;
    }
    
    $current_stats['total_request_bytes'] += $usage_context['request_size'];
    $current_stats['total_response_bytes'] += $usage_context['response_size'];
    $current_stats['total_dimensions_evaluated'] += $usage_context['dimensions_requested'];
    $current_stats['last_request'] = time();

    // Save updated statistics
    $this->state->set($stats_key, $current_stats);
  }

  /**
   * Gets API usage statistics.
   *
   * @return array
   *   The API usage statistics.
   */
  public function getApiUsageStatistics(): array {
    return $this->state->get('rail_api_usage_stats', []);
  }

  /**
   * Logs system integration events.
   *
   * Integrates with Drupal's logging system as required by requirement 7.4.
   *
   * @param string $event_type
   *   The type of integration event.
   * @param string $message
   *   The event message.
   * @param array $context
   *   Additional context data.
   */
  public function logIntegrationEvent(string $event_type, string $message, array $context = []): void {
    $full_context = array_merge($context, [
      'event_type' => $event_type,
      'timestamp' => time(),
    ]);

    switch ($event_type) {
      case 'guardrail_registered':
      case 'guardrail_enabled':
      case 'configuration_updated':
        $this->logger->info($message, $full_context);
        break;
        
      case 'guardrail_disabled':
      case 'configuration_error':
        $this->logger->warning($message, $full_context);
        break;
        
      case 'system_error':
      case 'integration_failure':
        $this->logger->error($message, $full_context);
        break;
        
      default:
        $this->logger->debug($message, $full_context);
        break;
    }
  }

}