<?php

namespace Drupal\rail_score\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\rail_score\RailScoreClient;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure RAIL Score settings.
 */
class RailScoreConfigForm extends ConfigFormBase {

  /**
   * The RAIL Score client.
   *
   * @var \Drupal\rail_score\RailScoreClient
   */
  protected $railScoreClient;

  /**
   * Constructs a RailScoreConfigForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface|null $typed_config_manager
   *   The typed config manager (Drupal 11+).
   * @param \Drupal\rail_score\RailScoreClient|null $rail_score_client
   *   The RAIL Score client.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    ?TypedConfigManagerInterface $typed_config_manager = NULL,
    ?RailScoreClient $rail_score_client = NULL
  ) {
    // Support both Drupal 9/10 (1 param) and Drupal 11 (2 params).
    if ($typed_config_manager) {
      parent::__construct($config_factory, $typed_config_manager);
    }
    else {
      parent::__construct($config_factory);
    }
    $this->railScoreClient = $rail_score_client;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    // Check if config.typed service exists (Drupal 11+).
    $typed_config = $container->has('config.typed') ? $container->get('config.typed') : NULL;

    return new static(
      $container->get('config.factory'),
      $typed_config,
      $container->get('rail_score.client')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['rail_score.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'rail_score_config_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('rail_score.settings');

    $form['api_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('API Settings'),
      '#description' => $this->t('Configure your RAIL Score API connection.'),
    ];

    // API Key with show/hide toggle
    $form['api_settings']['api_key_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['api-key-wrapper']],
    ];

    $form['api_settings']['api_key_wrapper']['api_key'] = [
      '#type' => 'password',
      '#title' => $this->t('RAIL Score API Key'),
      '#default_value' => $config->get('api_key'),
      '#description' => $this->t('Get your API key from <a href="@url" target="_blank">responsibleailabs.ai</a>', [
        '@url' => 'https://responsibleailabs.ai',
      ]),
      '#required' => TRUE,
      '#maxlength' => 255,
      '#attributes' => [
        'id' => 'rail-score-api-key',
        'autocomplete' => 'off',
      ],
    ];

    $form['api_settings']['api_key_wrapper']['show_api_key'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show API key'),
      '#default_value' => FALSE,
      '#attributes' => [
        'id' => 'show-api-key-toggle',
      ],
    ];

    // Add inline JavaScript to toggle password visibility
    $form['api_settings']['api_key_wrapper']['#attached']['library'][] = 'rail_score/admin';

    $form['api_settings']['base_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Base URL'),
      '#default_value' => $config->get('base_url') ?: 'https://api.responsibleailabs.ai',
      '#description' => $this->t('The RAIL Score API base URL. Leave default unless using a custom endpoint.'),
      '#required' => TRUE,
      '#maxlength' => 255,
    ];

    $form['api_settings']['test_connection'] = [
      '#type' => 'button',
      '#value' => $this->t('Test Connection'),
      '#ajax' => [
        'callback' => '::testConnectionCallback',
        'wrapper' => 'connection-test-result',
      ],
    ];

    $form['api_settings']['connection_result'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'connection-test-result'],
    ];

    // Endpoint selection.
    $form['api_settings']['endpoint_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Endpoint Configuration'),
      '#open' => FALSE,
    ];

    $form['api_settings']['endpoint_settings']['score_endpoint'] = [
      '#type' => 'radios',
      '#title' => $this->t('Score Endpoint'),
      '#options' => [
        'basic' => $this->t('Basic - Fast, concise scoring'),
        'detailed' => $this->t('Detailed - Comprehensive analysis with explanations'),
      ],
      '#default_value' => $config->get('score_endpoint') ?: 'basic',
      '#description' => $this->t('Choose between basic or detailed scoring endpoint.'),
    ];

    $form['api_settings']['endpoint_settings']['reprompt_endpoint'] = [
      '#type' => 'radios',
      '#title' => $this->t('Reprompt Endpoint'),
      '#options' => [
        'basic' => $this->t('Basic - Quick improvement suggestions'),
        'detailed' => $this->t('Detailed - In-depth improvement guidance'),
      ],
      '#default_value' => $config->get('reprompt_endpoint') ?: 'detailed',
      '#description' => $this->t('Choose between basic or detailed reprompt endpoint. <strong>🚧 Coming Soon</strong>'),
      '#disabled' => TRUE,
    ];

    // Coming Soon endpoints
    $form['api_settings']['endpoint_settings']['coming_soon'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('🚧 Coming Soon'),
      '#description' => $this->t('These endpoints will be available in future releases.'),
      '#attributes' => ['class' => ['coming-soon-endpoints']],
    ];

    $form['api_settings']['endpoint_settings']['coming_soon']['info'] = [
      '#type' => 'item',
      '#markup' => '<ul>
        <li><strong>Compliance Endpoints</strong> - GDPR, CCPA, and regulatory compliance checking</li>
        <li><strong>Content Generation</strong> - AI-powered content improvement and regeneration</li>
      </ul>',
    ];

    $form['evaluation_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Evaluation Settings'),
      '#description' => $this->t('Configure how content is evaluated.'),
    ];

    $form['evaluation_settings']['auto_evaluate'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Automatically evaluate content on save'),
      '#default_value' => $config->get('auto_evaluate') ?? TRUE,
      '#description' => $this->t('Automatically evaluate content when it is saved. Disable to manually trigger evaluations.'),
    ];

    $form['evaluation_settings']['threshold'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum Score Threshold'),
      '#default_value' => $config->get('threshold') ?? 7.0,
      '#min' => 0,
      '#max' => 10,
      '#step' => 0.1,
      '#description' => $this->t('Content with scores below this threshold will be flagged for review.'),
      '#required' => TRUE,
    ];

    $form['evaluation_settings']['auto_unpublish'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Auto-unpublish low-scoring content'),
      '#default_value' => $config->get('auto_unpublish') ?? FALSE,
      '#description' => $this->t('Automatically unpublish content that falls below the threshold. <strong>Use with caution.</strong>'),
    ];

    $form['evaluation_settings']['dimensions'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Dimensions to Evaluate'),
      '#options' => [
        'safety' => $this->t('Safety'),
        'privacy' => $this->t('Privacy'),
        'fairness' => $this->t('Fairness'),
        'transparency' => $this->t('Transparency'),
        'accountability' => $this->t('Accountability'),
        'reliability' => $this->t('Reliability'),
        'inclusivity' => $this->t('Inclusivity'),
        'user_impact' => $this->t('User Impact'),
      ],
      '#default_value' => $config->get('dimensions') ?: [],
      '#description' => $this->t('Select which dimensions to evaluate. Leave empty to evaluate all dimensions.'),
    ];

    // Dimension weights configuration.
    $form['evaluation_settings']['dimension_weights'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Dimension Weights (Optional)'),
      '#description' => $this->t('Assign custom weights to each dimension as percentages. Leave empty for equal weighting. <strong>Weights should sum to 100%</strong>.'),
      '#collapsible' => TRUE,
      '#collapsed' => TRUE,
    ];

    $dimension_weights = $config->get('dimension_weights') ?: [];
    $dimensions_list = [
      'safety', 'privacy', 'fairness', 'transparency',
      'accountability', 'reliability', 'inclusivity', 'user_impact',
    ];

    foreach ($dimensions_list as $dimension) {
      // Convert stored fractional value (0-1) to percentage (0-100) for display.
      $default_value = isset($dimension_weights[$dimension]) ? ($dimension_weights[$dimension] * 100) : '';

      $form['evaluation_settings']['dimension_weights'][$dimension . '_weight'] = [
        '#type' => 'number',
        '#title' => $this->t('@dimension Weight (%)', ['@dimension' => ucfirst(str_replace('_', ' ', $dimension))]),
        '#default_value' => $default_value,
        '#min' => 0,
        '#max' => 100,
        '#step' => 0.1,
        '#size' => 10,
        '#placeholder' => '12.5',
        '#field_suffix' => '%',
      ];
    }

    // Advanced settings - Dimension-level thresholds.
    $form['evaluation_settings']['advanced'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced Settings'),
      '#description' => $this->t('Configure dimension-specific score thresholds and confidence requirements.'),
      '#open' => FALSE,
    ];

    $form['evaluation_settings']['advanced']['dimension_thresholds'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Dimension-Level Score Thresholds'),
      '#description' => $this->t('Set minimum score thresholds for specific dimensions. Leave empty to use the global threshold.'),
      '#collapsible' => TRUE,
      '#collapsed' => TRUE,
    ];

    $dimension_thresholds = $config->get('dimension_thresholds') ?: [];
    foreach ($dimensions_list as $dimension) {
      $default_value = isset($dimension_thresholds[$dimension]) ? $dimension_thresholds[$dimension] : '';

      $form['evaluation_settings']['advanced']['dimension_thresholds'][$dimension . '_threshold'] = [
        '#type' => 'number',
        '#title' => $this->t('@dimension Minimum Score', ['@dimension' => ucfirst(str_replace('_', ' ', $dimension))]),
        '#default_value' => $default_value,
        '#min' => 0,
        '#max' => 10,
        '#step' => 0.1,
        '#size' => 10,
        '#placeholder' => '7.0',
        '#description' => $this->t('Minimum score for @dimension dimension.', ['@dimension' => $dimension]),
      ];
    }

    $form['evaluation_settings']['advanced']['dimension_confidence'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Dimension-Level Confidence Requirements'),
      '#description' => $this->t('Set minimum confidence requirements for selected dimensions (0-1). Leave empty to accept all confidence levels.'),
      '#collapsible' => TRUE,
      '#collapsed' => TRUE,
    ];

    $dimension_confidence = $config->get('dimension_confidence') ?: [];
    foreach ($dimensions_list as $dimension) {
      $default_value = isset($dimension_confidence[$dimension]) ? $dimension_confidence[$dimension] : '';

      $form['evaluation_settings']['advanced']['dimension_confidence'][$dimension . '_confidence'] = [
        '#type' => 'number',
        '#title' => $this->t('@dimension Minimum Confidence', ['@dimension' => ucfirst(str_replace('_', ' ', $dimension))]),
        '#default_value' => $default_value,
        '#min' => 0,
        '#max' => 1,
        '#step' => 0.01,
        '#size' => 10,
        '#placeholder' => '0.7',
        '#description' => $this->t('Minimum confidence level for @dimension (e.g., 0.8 = 80%).', ['@dimension' => $dimension]),
      ];
    }

    $form['content_types'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Content Types'),
      '#description' => $this->t('Select which content types should be automatically evaluated.'),
    ];

    // Load available content types.
    $content_types = \Drupal::entityTypeManager()
      ->getStorage('node_type')
      ->loadMultiple();

    $options = [];
    foreach ($content_types as $type) {
      $options[$type->id()] = $type->label();
    }

    $form['content_types']['enabled_content_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Enabled Content Types'),
      '#options' => $options,
      '#default_value' => $config->get('enabled_content_types') ?: [],
      '#description' => $this->t('Select which content types should be evaluated with RAIL Score.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * AJAX callback to test API connection.
   */
  public function testConnectionCallback(array &$form, FormStateInterface $form_state) {
    $element = $form['api_settings']['connection_result'];

    // Get values from the form (not saved config).
    $api_key = $form_state->getValue('api_key');
    $base_url = $form_state->getValue('base_url');

    // Test connection and get available endpoints.
    $result = $this->railScoreClient->testAllEndpoints($api_key, $base_url);

    if ($result['success']) {
      $markup = '<div class="messages messages--status">';
      $markup .= '<strong>' . $this->t('✓ Connection Successful!') . '</strong><br><br>';
      $markup .= '<strong>' . $this->t('Available Endpoints:') . '</strong>';
      $markup .= '<ul style="margin: 10px 0; padding-left: 20px;">';

      foreach ($result['endpoints'] as $endpoint => $status) {
        $icon = $status ? '✓' : '✗';
        $class = $status ? 'color: green;' : 'color: red;';
        $markup .= '<li style="' . $class . '">' . $icon . ' ' . $endpoint . '</li>';
      }

      $markup .= '</ul></div>';
      $element['#markup'] = $markup;
    }
    else {
      $element['#markup'] = '<div class="messages messages--error">' .
        '<strong>' . $this->t('✗ Connection Failed') . '</strong><br>' .
        $this->t('Error: @error', ['@error' => $result['error']]) .
        '</div>';
    }

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    $api_key = $form_state->getValue('api_key');
    if (empty($api_key) || strlen($api_key) < 10) {
      $form_state->setErrorByName('api_key', $this->t('API key must be at least 10 characters.'));
    }

    $base_url = $form_state->getValue('base_url');
    if (!filter_var($base_url, FILTER_VALIDATE_URL)) {
      $form_state->setErrorByName('base_url', $this->t('Base URL must be a valid URL.'));
    }

    $threshold = $form_state->getValue('threshold');
    if ($threshold < 0 || $threshold > 10) {
      $form_state->setErrorByName('threshold', $this->t('Threshold must be between 0 and 10.'));
    }

    // Validate dimension weights sum to 100% (if any are set).
    $dimensions_list = [
      'safety', 'privacy', 'fairness', 'transparency',
      'accountability', 'reliability', 'inclusivity', 'user_impact',
    ];

    $total_weight = 0;
    $has_weights = FALSE;

    foreach ($dimensions_list as $dimension) {
      $weight_key = $dimension . '_weight';
      $weight_value = $form_state->getValue($weight_key);

      if (!empty($weight_value)) {
        $has_weights = TRUE;
        $total_weight += (float) $weight_value;
      }
    }

    // Only validate if at least one weight is set.
    if ($has_weights) {
      // Allow small rounding errors (within 0.1%).
      if (abs($total_weight - 100) > 0.1) {
        $form_state->setErrorByName('dimension_weights',
          $this->t('Dimension weights must sum to 100%. Current total: @total%', [
            '@total' => number_format($total_weight, 1),
          ])
        );
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Collect dimension weights and convert percentages to fractions.
    $dimension_weights = [];
    $dimension_thresholds = [];
    $dimension_confidence = [];

    $dimensions_list = [
      'safety', 'privacy', 'fairness', 'transparency',
      'accountability', 'reliability', 'inclusivity', 'user_impact',
    ];

    foreach ($dimensions_list as $dimension) {
      // Dimension weights
      $weight_key = $dimension . '_weight';
      if (!empty($form_state->getValue($weight_key))) {
        // Convert percentage (0-100) to fraction (0-1) for API.
        $dimension_weights[$dimension] = (float) $form_state->getValue($weight_key) / 100;
      }

      // Dimension thresholds
      $threshold_key = $dimension . '_threshold';
      if (!empty($form_state->getValue($threshold_key))) {
        $dimension_thresholds[$dimension] = (float) $form_state->getValue($threshold_key);
      }

      // Dimension confidence
      $confidence_key = $dimension . '_confidence';
      if (!empty($form_state->getValue($confidence_key))) {
        $dimension_confidence[$dimension] = (float) $form_state->getValue($confidence_key);
      }
    }

    $this->config('rail_score.settings')
      ->set('api_key', $form_state->getValue('api_key'))
      ->set('base_url', $form_state->getValue('base_url'))
      ->set('score_endpoint', $form_state->getValue('score_endpoint'))
      ->set('reprompt_endpoint', $form_state->getValue('reprompt_endpoint'))
      ->set('auto_evaluate', $form_state->getValue('auto_evaluate'))
      ->set('threshold', (float) $form_state->getValue('threshold'))
      ->set('auto_unpublish', $form_state->getValue('auto_unpublish'))
      ->set('dimensions', array_filter($form_state->getValue('dimensions')))
      ->set('dimension_weights', $dimension_weights)
      ->set('dimension_thresholds', $dimension_thresholds)
      ->set('dimension_confidence', $dimension_confidence)
      ->set('enabled_content_types', array_filter($form_state->getValue('enabled_content_types')))
      ->save();

    parent::submitForm($form, $form_state);

    $this->messenger()->addStatus($this->t('RAIL Score settings have been saved.'));
  }

}
