<?php

namespace Drupal\rail_score;

use Drupal\Core\Config\ConfigFactoryInterface;
use GuzzleHttp\ClientInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use GuzzleHttp\Exception\GuzzleException;

/**
 * RAIL Score API client service.
 *
 * Provides methods for evaluating content using the RAIL Score API.
 * Handles authentication, error handling, and response processing.
 *
 * @see https://responsibleailabs.ai/docs
 */
class RailScoreClient {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The logger service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * Constructs a RailScoreClient object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    ClientInterface $http_client,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->configFactory = $config_factory;
    $this->httpClient = $http_client;
    $this->logger = $logger_factory->get('rail_score');
  }

  /**
   * Evaluate content with RAIL Score API.
   *
   * @param string $content
   *   The content to evaluate.
   * @param array $options
   *   Additional evaluation options:
   *   - dimensions: Array of dimensions to evaluate.
   *   - weights: Array of dimension weights.
   *
   * @return array|false
   *   The evaluation result array, or FALSE on failure.
   */
  public function evaluate(string $content, array $options = []) {
    $config = $this->configFactory->get('rail_score.settings');
    $api_key = $config->get('api_key');
    $base_url = $config->get('base_url') ?: 'https://api.responsibleailabs.ai';

    if (empty($api_key)) {
      $this->logger->error('[RAIL Score] API key not configured');
      return FALSE;
    }

    // Prepare dimensions if configured.
    $dimensions = $config->get('dimensions') ?: [];
    if (!empty($dimensions)) {
      $options['dimensions'] = array_values(array_filter($dimensions));
    }

    // Prepare dimension weights if configured.
    $dimension_weights = $config->get('dimension_weights') ?: [];
    if (!empty($dimension_weights)) {
      $options['weights'] = $dimension_weights;
    }

    // Add confidence threshold if configured.
    $confidence_threshold = $config->get('confidence_threshold');
    if (!empty($confidence_threshold)) {
      $options['confidence_threshold'] = (float) $confidence_threshold;
    }

    // Get configured endpoint (basic or detailed).
    $endpoint_type = $config->get('score_endpoint') ?: 'basic';
    $endpoint = '/railscore/v1/score/' . $endpoint_type;

    $this->logger->info('[RAIL Score] Evaluating content (@length characters) using @endpoint', [
      '@length' => strlen($content),
      '@endpoint' => $endpoint,
    ]);

    try {
      $response = $this->httpClient->request('POST', $base_url . $endpoint, [
        'headers' => [
          'Content-Type' => 'application/json',
          'Authorization' => 'Bearer ' . $api_key,
        ],
        'json' => array_merge(['content' => $content], $options),
        'timeout' => 60,
      ]);

      $data = json_decode($response->getBody()->getContents(), TRUE);

      // Log the full response structure for debugging
      $this->logger->debug('[RAIL Score] Full API response: @response', [
        '@response' => json_encode($data, JSON_PRETTY_PRINT),
      ]);

      if (isset($data['result']['rail_score']['score'])) {
        $score = $data['result']['rail_score']['score'];
        $this->logger->notice('[RAIL Score] ✓ Evaluation complete: score @score/10', [
          '@score' => number_format($score, 1),
        ]);

        // Check for dimensions in response (API returns dimension_scores not dimensions)
        if (isset($data['result']['dimension_scores'])) {
          $this->logger->debug('[RAIL Score] Dimension scores found: @dims', [
            '@dims' => implode(', ', array_keys($data['result']['dimension_scores'])),
          ]);
        } elseif (isset($data['result']['dimensions'])) {
          $this->logger->debug('[RAIL Score] Dimensions found: @dims', [
            '@dims' => implode(', ', array_keys($data['result']['dimensions'])),
          ]);
        } else {
          $this->logger->warning('[RAIL Score] No dimension_scores or dimensions found in API response');
        }
      }

      return $data;
    }
    catch (GuzzleException $e) {
      $this->logger->error('[RAIL Score] ✗ API error: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Evaluate specific dimension with RAIL Score API.
   *
   * @param string $content
   *   The content to evaluate.
   * @param string $dimension
   *   The dimension to evaluate (safety, privacy, fairness, transparency,
   *   accountability, reliability, inclusivity, user_impact).
   * @param bool $detailed
   *   Use detailed endpoint for comprehensive analysis.
   *
   * @return array|false
   *   The evaluation result array, or FALSE on failure.
   */
  public function evaluateDimension(string $content, string $dimension, bool $detailed = FALSE) {
    $config = $this->configFactory->get('rail_score.settings');
    $api_key = $config->get('api_key');
    $base_url = $config->get('base_url') ?: 'https://api.responsibleailabs.ai';

    if (empty($api_key)) {
      $this->logger->error('[RAIL Score] API key not configured');
      return FALSE;
    }

    $valid_dimensions = [
      'safety', 'privacy', 'fairness', 'transparency',
      'accountability', 'reliability', 'inclusivity', 'user_impact',
    ];

    if (!in_array($dimension, $valid_dimensions)) {
      $this->logger->error('[RAIL Score] Invalid dimension: @dimension', [
        '@dimension' => $dimension,
      ]);
      return FALSE;
    }

    $endpoint_type = $detailed ? 'detailed' : 'basic';
    $endpoint = '/railscore/v1/score/' . $endpoint_type . '/dimension';

    $this->logger->info('[RAIL Score] Evaluating @dimension dimension using @endpoint', [
      '@dimension' => $dimension,
      '@endpoint' => $endpoint,
    ]);

    try {
      $response = $this->httpClient->request('POST', $base_url . $endpoint, [
        'headers' => [
          'Content-Type' => 'application/json',
          'Authorization' => 'Bearer ' . $api_key,
        ],
        'json' => [
          'content' => $content,
          'dimension' => $dimension,
        ],
        'timeout' => 60,
      ]);

      $data = json_decode($response->getBody()->getContents(), TRUE);

      $this->logger->notice('[RAIL Score] ✓ Dimension evaluation complete: @dimension', [
        '@dimension' => $dimension,
      ]);

      return $data;
    }
    catch (GuzzleException $e) {
      $this->logger->error('[RAIL Score] ✗ Dimension evaluation error: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Check GDPR compliance.
   *
   * NOTE: This endpoint is marked as "Coming Soon" in RAIL Score API v1.0.0.
   * It may not be available yet. The endpoint path has been kept for future compatibility.
   *
   * @param string $content
   *   The content to check.
   * @param array $context
   *   Additional context information:
   *   - data_type: Type of data being processed.
   *   - processing_purpose: Purpose of data processing.
   *
   * @return array|false
   *   The compliance result array, or FALSE on failure.
   */
  public function checkGdprCompliance(string $content, array $context = []) {
    $config = $this->configFactory->get('rail_score.settings');
    $api_key = $config->get('api_key');
    $base_url = $config->get('base_url') ?: 'https://api.responsibleailabs.ai';

    if (empty($api_key)) {
      $this->logger->error('[RAIL Score] API key not configured');
      return FALSE;
    }

    $this->logger->info('[RAIL Score] Checking GDPR compliance');

    try {
      // NOTE: This endpoint may not be available in API v1.0.0 (marked as "Coming Soon").
      $response = $this->httpClient->request('POST', $base_url . '/v1/compliance/gdpr', [
        'headers' => [
          'Content-Type' => 'application/json',
          'Authorization' => 'Bearer ' . $api_key,
        ],
        'json' => [
          'content' => $content,
          'context' => $context,
        ],
        'timeout' => 60,
      ]);

      $data = json_decode($response->getBody()->getContents(), TRUE);

      $this->logger->notice('[RAIL Score] ✓ GDPR compliance check complete');

      return $data;
    }
    catch (GuzzleException $e) {
      $this->logger->error('[RAIL Score] ✗ GDPR check error: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Get usage statistics from RAIL Score API.
   *
   * NOTE: This endpoint may not be documented in RAIL Score API v1.0.0.
   * Verify availability with the latest API documentation.
   *
   * @return array|false
   *   Usage statistics array, or FALSE on failure.
   */
  public function getUsageStats() {
    $config = $this->configFactory->get('rail_score.settings');
    $api_key = $config->get('api_key');
    $base_url = $config->get('base_url') ?: 'https://api.responsibleailabs.ai';

    if (empty($api_key)) {
      $this->logger->error('[RAIL Score] API key not configured');
      return FALSE;
    }

    try {
      // NOTE: Verify this endpoint is available in your API version.
      $response = $this->httpClient->request('GET', $base_url . '/v1/usage', [
        'headers' => [
          'Authorization' => 'Bearer ' . $api_key,
        ],
        'query' => ['limit' => 50],
        'timeout' => 30,
      ]);

      $data = json_decode($response->getBody()->getContents(), TRUE);

      return $data;
    }
    catch (GuzzleException $e) {
      $this->logger->error('[RAIL Score] ✗ Usage stats error: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Test API connection.
   *
   * @param string|null $api_key
   *   Optional API key to test. If not provided, uses saved config.
   * @param string|null $base_url
   *   Optional base URL to test. If not provided, uses saved config.
   *
   * @return bool
   *   TRUE if connection is successful, FALSE otherwise.
   */
  public function testConnection($api_key = NULL, $base_url = NULL) {
    $config = $this->configFactory->get('rail_score.settings');

    // Use provided values or fall back to saved config.
    if ($api_key === NULL) {
      $api_key = $config->get('api_key');
    }
    if ($base_url === NULL) {
      $base_url = $config->get('base_url') ?: 'https://api.responsibleailabs.ai';
    }

    if (empty($api_key)) {
      $this->logger->error('[RAIL Score] ✗ Connection test failed: API key is empty');
      return FALSE;
    }

    try {
      // First, check if API is online (no auth required).
      $response = $this->httpClient->request('GET', $base_url . '/', [
        'timeout' => 10,
        'http_errors' => FALSE,
      ]);

      if ($response->getStatusCode() !== 200) {
        // Try health check endpoint as fallback.
        $response = $this->httpClient->request('GET', $base_url . '/health', [
          'timeout' => 10,
          'http_errors' => FALSE,
        ]);

        if ($response->getStatusCode() !== 200) {
          $this->logger->error('[RAIL Score] ✗ API is not accessible at @url', [
            '@url' => $base_url,
          ]);
          return FALSE;
        }
      }

      // Verify the API key using the /verify endpoint.
      $response = $this->httpClient->request('POST', $base_url . '/verify', [
        'headers' => [
          'Authorization' => 'Bearer ' . $api_key,
        ],
        'timeout' => 10,
        'http_errors' => FALSE,
      ]);

      $status_code = $response->getStatusCode();
      $body = json_decode($response->getBody()->getContents(), TRUE);

      if ($status_code === 200 && isset($body['status']) && $body['status'] === 'verified') {
        $this->logger->notice('[RAIL Score] ✓ Connection test successful - API key verified');
        return TRUE;
      }
      elseif ($status_code === 401 || $status_code === 403 || (isset($body['detail']) && stripos($body['detail'], 'invalid') !== FALSE)) {
        $this->logger->error('[RAIL Score] ✗ API key verification failed: Invalid or inactive API key');
        return FALSE;
      }
      else {
        $this->logger->error('[RAIL Score] ✗ Unexpected response from API: @status @body', [
          '@status' => $status_code,
          '@body' => json_encode($body),
        ]);
        return FALSE;
      }
    }
    catch (GuzzleException $e) {
      $this->logger->error('[RAIL Score] ✗ Connection test failed: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Test all API endpoints and return their availability.
   *
   * @param string|null $api_key
   *   Optional API key to test.
   * @param string|null $base_url
   *   Optional base URL to test.
   *
   * @return array
   *   Array with 'success', 'error', and 'endpoints' keys.
   */
  public function testAllEndpoints($api_key = NULL, $base_url = NULL) {
    $config = $this->configFactory->get('rail_score.settings');

    if ($api_key === NULL) {
      $api_key = $config->get('api_key');
    }
    if ($base_url === NULL) {
      $base_url = $config->get('base_url') ?: 'https://api.responsibleailabs.ai';
    }

    if (empty($api_key)) {
      return [
        'success' => FALSE,
        'error' => 'API key is empty',
        'endpoints' => [],
      ];
    }

    $endpoints_to_test = [
      '/health' => 'GET',
      '/verify' => 'POST',
      '/railscore/v1/score/basic' => 'POST',
      '/railscore/v1/score/detailed' => 'POST',
      '/railscore/v1/score/basic/dimension' => 'POST',
      '/railscore/v1/score/detailed/dimension' => 'POST',
      '/railscore/v1/reprompt/basic' => 'POST',
      '/railscore/v1/reprompt/detailed' => 'POST',
    ];

    $results = [];
    $api_accessible = FALSE;

    try {
      // First check if API is online.
      $response = $this->httpClient->request('GET', $base_url . '/health', [
        'timeout' => 10,
        'http_errors' => FALSE,
      ]);

      if ($response->getStatusCode() === 200) {
        $api_accessible = TRUE;
        $results['/health'] = TRUE;
      }
      else {
        return [
          'success' => FALSE,
          'error' => 'API is not accessible at ' . $base_url,
          'endpoints' => [],
        ];
      }

      // Test API key verification.
      $response = $this->httpClient->request('POST', $base_url . '/verify', [
        'headers' => ['Authorization' => 'Bearer ' . $api_key],
        'timeout' => 10,
        'http_errors' => FALSE,
      ]);

      $status = $response->getStatusCode();
      if ($status === 200) {
        $results['/verify'] = TRUE;
      }
      elseif ($status === 401) {
        return [
          'success' => FALSE,
          'error' => 'Invalid API key',
          'endpoints' => $results,
        ];
      }
      else {
        $results['/verify'] = FALSE;
      }

      // Test score endpoints with minimal payload.
      $test_payload = [
        'content' => 'Test content',
      ];

      foreach (['/railscore/v1/score/basic', '/railscore/v1/score/detailed'] as $endpoint) {
        try {
          $response = $this->httpClient->request('POST', $base_url . $endpoint, [
            'headers' => [
              'Content-Type' => 'application/json',
              'Authorization' => 'Bearer ' . $api_key,
            ],
            'json' => $test_payload,
            'timeout' => 15,
            'http_errors' => FALSE,
          ]);
          $results[$endpoint] = ($response->getStatusCode() === 200);
        }
        catch (\Exception $e) {
          $results[$endpoint] = FALSE;
        }
      }

      // Test dimension-specific endpoints.
      $dimension_payload = [
        'content' => 'Test content',
        'dimension' => 'safety',
      ];

      foreach (['/railscore/v1/score/basic/dimension', '/railscore/v1/score/detailed/dimension'] as $endpoint) {
        try {
          $response = $this->httpClient->request('POST', $base_url . $endpoint, [
            'headers' => [
              'Content-Type' => 'application/json',
              'Authorization' => 'Bearer ' . $api_key,
            ],
            'json' => $dimension_payload,
            'timeout' => 15,
            'http_errors' => FALSE,
          ]);
          $results[$endpoint] = ($response->getStatusCode() === 200);
        }
        catch (\Exception $e) {
          $results[$endpoint] = FALSE;
        }
      }

      // Test reprompt endpoints.
      $reprompt_payload = [
        'content' => 'Test content',
        'current_score' => 5.0,
        'accepted_score' => 7.0,
      ];

      foreach (['/railscore/v1/reprompt/basic', '/railscore/v1/reprompt/detailed'] as $endpoint) {
        try {
          $response = $this->httpClient->request('POST', $base_url . $endpoint, [
            'headers' => [
              'Content-Type' => 'application/json',
              'Authorization' => 'Bearer ' . $api_key,
            ],
            'json' => $reprompt_payload,
            'timeout' => 15,
            'http_errors' => FALSE,
          ]);
          $results[$endpoint] = ($response->getStatusCode() === 200);
        }
        catch (\Exception $e) {
          $results[$endpoint] = FALSE;
        }
      }

      return [
        'success' => TRUE,
        'error' => NULL,
        'endpoints' => $results,
      ];
    }
    catch (GuzzleException $e) {
      return [
        'success' => FALSE,
        'error' => $e->getMessage(),
        'endpoints' => $results,
      ];
    }
  }

  /**
   * Get improvement suggestions for content using RAIL Score reprompt.
   *
   * @param string $content
   *   The content to get suggestions for.
   * @param float $current_score
   *   The current RAIL Score of the content.
   * @param array $options
   *   Additional options:
   *   - threshold: Score threshold (default: from config).
   *   - detailed: Use detailed endpoint (default: TRUE).
   *
   * @return array|false
   *   The reprompt result with suggestions, or FALSE on failure.
   */
  public function getImprovementSuggestions(string $content, float $current_score, array $options = []) {
    $config = $this->configFactory->get('rail_score.settings');
    $api_key = $config->get('api_key');
    $base_url = $config->get('base_url') ?: 'https://api.responsibleailabs.ai';

    if (empty($api_key)) {
      $this->logger->error('[RAIL Score] API key not configured');
      return FALSE;
    }

    // Determine threshold and accepted score.
    $threshold = $options['threshold'] ?? (float) $config->get('threshold') ?? 7.0;
    $accepted_score = $options['accepted_score'] ?? (float) $config->get('accepted_score') ?? $threshold;

    // Get configured reprompt endpoint (basic or detailed).
    $endpoint_type = $config->get('reprompt_endpoint') ?: 'detailed';
    $endpoint = '/railscore/v1/reprompt/' . $endpoint_type;

    $this->logger->info('[RAIL Score] Getting improvement suggestions using @endpoint (score: @score, accepted: @accepted)', [
      '@endpoint' => $endpoint,
      '@score' => $current_score,
      '@accepted' => $accepted_score,
    ]);

    // Prepare request payload.
    // TODO: Update reprompt endpoint payload in next patch
    $payload = [
      'content' => $content,
      'current_score' => $current_score,
      'accepted_score' => $accepted_score,
    ];

    // Add dimension weights if configured.
    $dimension_weights = $config->get('dimension_weights') ?: [];
    if (!empty($dimension_weights)) {
      $payload['weights'] = $dimension_weights;
    }

    // Add confidence threshold if configured.
    $confidence_threshold = $config->get('confidence_threshold');
    if (!empty($confidence_threshold)) {
      $payload['confidence_threshold'] = (float) $confidence_threshold;
    }

    try {
      $response = $this->httpClient->request('POST', $base_url . $endpoint, [
        'headers' => [
          'Content-Type' => 'application/json',
          'Authorization' => 'Bearer ' . $api_key,
        ],
        'json' => $payload,
        'timeout' => 60,
      ]);

      $data = json_decode($response->getBody()->getContents(), TRUE);

      $this->logger->notice('[RAIL Score] ✓ Improvement suggestions retrieved');

      return $data;
    }
    catch (GuzzleException $e) {
      $this->logger->error('[RAIL Score] ✗ Reprompt error: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

}
