<?php

namespace Drupal\random_coupon_generator\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Wotz\UniqueCodes\UniqueCodes;
use Symfony\Component\HttpFoundation\StreamedResponse;

/**
 * Form to generate random coupon codes.
 */
class GeneratorForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'random_coupon_generator_generator_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = \Drupal::config('random_coupon_generator.settings');

    // Add an introduction text to the generator form.
    $form['help_intro'] = [
      '#type' => 'markup',
      '#markup' => $this->t('<p>Generate unique, random-looking codes. These codes can be used for vouchers, coupons, ... You can generate thousands of codes without having to check if a code has already been generated in the past. The codes are generated using a mathematical algorithm based on prime numbers. This ensures that all codes are unique and non-sequential.</p>'),
    ];

    // Allow default settings override.
    $form['settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Generator settings'),
    ];
    $form['settings']['markup'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('<p>These default settings can be changed in the <a href="@settings">module settings</a>.</p>', [
        '@settings' => Url::fromRoute('random_coupon_generator.settings')->toString(),
      ]),
    ];
    $form['settings']['length'] = [
      '#type' => 'number',
      '#title' => $this->t('Default coupon length'),
      '#description' => $this->t('Number of characters for generated coupons.'),
      '#default_value' => $config->get('length'),
      '#min' => 4,
      '#max' => 64,
      '#required' => TRUE,
    ];
    $form['settings']['charset'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Character set'),
      '#description' => $this->t('Characters to use when generating coupons. Remove ambiguous characters if desired.<br><strong>Example:</strong> "ABCDEFGHJKLMNPQRSTUVWXYZ23456789".'),
      '#default_value' => $config->get('charset'),
      '#required' => TRUE,
    ];
    $form['settings']['prefix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Prefix'),
      '#description' => $this->t('The prefix of each unique code.<br><strong>Example:</strong> "SALE-" will generate codes like "SALE-ABCD1234".'),
      '#default_value' => $config->get('prefix'),
    ];
    $form['settings']['suffix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Suffix'),
      '#description' => $this->t('The suffix of each unique code.<br><strong>Example:</strong> "-SALE" will generate codes like "ABCD1234-SALE".'),
      '#default_value' => $config->get('suffix'),
    ];
    // Disabled for now, doesn't work properly at the moment.
    /*$form['settings']['delimiter'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Delimiter'),
      '#description' => $this->t('The code can be split in different pieces and glued together using the specified delimiter. Provide the delimiter character followed by the number of characters between each delimiter.<br>Leave empty to disable delimiters.<br><strong>Example:</strong> "-,2" will generate codes like "AB-CD-12-34".'),
      '#default_value' => $config->get('delimiter'),
    ];*/
    $form['settings']['unique'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Generate unique codes'),
      '#description' => $this->t('Normally the codes generated use the same numbers as their base (1, 2, 3, ... , max number). If you enable this option, the codes will be generated based on the current timestamp which creates unique codes.'),
      '#default_value' => $config->get('unique'),
    ];

    // Main form elements.
    $form['amount'] = [
      '#type' => 'number',
      '#title' => $this->t('Number of codes to generate'),
      '#description' => $this->t('Number of characters for generated coupons.'),
      '#min' => 1,
      '#max' => 50000,
      '#required' => TRUE,
    ];
    $form['output'] = [
      '#type' => 'radios',
      '#title' => $this->t('Output'),
      '#description' => $this->t('Choose whether to download the generated coupons as a CSV file or show them on the page.'),
      '#options' => [
        'csv' => $this->t('Export to CSV'),
        'page' => $this->t('Output on page'),
      ],
      '#default_value' => 'csv',
      '#required' => TRUE,
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Generate coupons'),
    ];

    // Show results when available.
    if ($form_state->get('generated') && !empty($form_state->getValue('coupons'))) {
      $form['results'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Generated coupons'),
        '#description' => $this->t('Output of the generated coupon codes. Refresh the page before generating new codes.'),
        '#default_value' => implode("\n", $form_state->getValue('coupons') ?: []),
        '#attributes' => ['readonly' => 'readonly'],
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // Validation can be added here if needed.
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Get Obfuscating and Max prime from config.
    $config = \Drupal::config('random_coupon_generator.settings');
    $obfuscating_prime = (int) $config->get('obfuscating_prime');
    $max_prime = (int) $config->get('max_prime');

    // Get form values.
    $length = (int) $form_state->getValue('length');
    $charset = $form_state->getValue('charset');
    $prefix = $form_state->getValue('prefix');
    $suffix = $form_state->getValue('suffix');
    // $delimiter = $form_state->getValue('delimiter');
    $amount = (int) $form_state->getValue('amount');
    $unique = (bool) $form_state->getValue('unique');
    $output = $form_state->getValue('output');

    $amount_start = 1;
    $amount_end = $amount;
    if ($unique) {
      $amount_start = time();
      $amount_end = $amount_start + $amount - 1;
    }

    // Generate the coupons.
    $coupons = (new UniqueCodes())
      ->setObfuscatingPrime($obfuscating_prime)
      ->setMaxPrime($max_prime)
      ->setLength($length)
      ->setCharacters($charset)
      ->setPrefix($prefix)
      ->setSuffix($suffix)
      // ->setDelimiter($delimiter)
      ->generate($amount_start, $amount_end, TRUE);

    $this->messenger()->addStatus($this->t('Generated @count coupons.', ['@count' => $amount]));

    // If the user requested CSV export, return a StreamedResponse with the
    // CSV contents. Otherwise, rebuild the form to show results on the page.
    if ($output === 'csv') {
      $filename = 'random_coupons_' . date('Ymd-Hi') . '.csv';
      $response = new StreamedResponse(function () use ($coupons) {
        $out = fopen('php://output', 'w');
        // Optionally write a header row. Here we only write one column.
        foreach ($coupons as $code) {
          fputcsv($out, [$code]);
        }
        fclose($out);
      });
      $response->headers->set('Content-Type', 'text/csv; charset=UTF-8');
      $response->headers->set('Content-Disposition', 'attachment; filename="' . $filename . '"');

      // Set the response so Drupal returns the CSV immediately.
      $form_state->setResponse($response);
      return;
    }
    else {
      $form_state->set('generated', TRUE);
      $form_state->setValue('coupons', $coupons);
      $form_state->setRebuild();
    }

  }

}
