<?php

namespace Drupal\random_coupon_generator\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure settings for Random Coupon Generator.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'random_coupon_generator_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'random_coupon_generator.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('random_coupon_generator.settings');

    $form['obfuscating_prime'] = [
      '#type' => 'number',
      '#title' => $this->t('Obfuscating prime number'),
      '#description' => $this->t('A prime number used to obfuscate a number between 1 and the max prime number. This prime number must be bigger than your Max Prime number. Primes can be easily generated using an online tool like <a href="https://bigprimes.org">Big Primes</a>.<br><strong>Example:</strong> 9006077.'),
      '#default_value' => $config->get('obfuscating_prime'),
      '#min' => 1,
      '#required' => TRUE,
    ];
    $form['max_prime'] = [
      '#type' => 'number',
      '#title' => $this->t('Max prime number'),
      '#description' => $this->t('The max prime determines the maximum amount of unique codes you can generate. If you provide 101, then you can generate codes from 1 to 100. This prime number must be smaller than the prime number you provide to the Obfuscating Prime field.<br><strong>Example:</strong> 7230323.'),
      '#default_value' => $config->get('max_prime'),
      '#min' => 2,
      '#required' => TRUE,
    ];
    $form['length'] = [
      '#type' => 'number',
      '#title' => $this->t('Default coupon length'),
      '#description' => $this->t('Number of characters for generated coupons.'),
      '#default_value' => $config->get('length'),
      '#min' => 4,
      '#max' => 64,
      '#required' => TRUE,
    ];
    $form['charset'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Character set'),
      '#description' => $this->t('Characters to use when generating coupons. Remove ambiguous characters if desired.<br><strong>Example:</strong> "ABCDEFGHJKLMNPQRSTUVWXYZ23456789".'),
      '#default_value' => $config->get('charset'),
      '#required' => TRUE,
    ];
    $form['prefix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Prefix'),
      '#description' => $this->t('The prefix of each unique code.<br><strong>Example:</strong> "SALE-" will generate codes like "SALE-ABCD1234".'),
      '#default_value' => $config->get('prefix'),
    ];
    $form['suffix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Suffix'),
      '#description' => $this->t('The suffix of each unique code.<br><strong>Example:</strong> "-SALE" will generate codes like "ABCD1234-SALE".'),
      '#default_value' => $config->get('suffix'),
    ];
    // Disabled for now, doesn't work properly at the moment.
    /*$form['delimiter'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Delimiter'),
      '#description' => $this->t('The code can be split in different pieces and glued together using the specified delimiter. Provide the delimiter character followed by the number of characters between each delimiter.<br>Leave empty to disable delimiters.<br><strong>Example:</strong> "-,2" will generate codes like "AB-CD-12-34".'),
      '#default_value' => $config->get('delimiter'),
    ];*/
    $form['unique'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Generate unique codes'),
      '#description' => $this->t('Normally the codes generated use the same numbers as their base (1, 2, 3, ... , max number). If you enable this option, the codes will be generated based on the current timestamp which creates unique codes.'),
      '#default_value' => $config->get('unique'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $obfuscating_prime = (int) $form_state->getValue('obfuscating_prime');
    $max_prime = (int) $form_state->getValue('max_prime');
    if ($obfuscating_prime <= $max_prime) {
      $form_state->setErrorByName('obfuscating_prime', $this->t('The obfuscating prime number must be bigger than the max prime number.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('random_coupon_generator.settings')
      ->set('obfuscating_prime', (int) $form_state->getValue('obfuscating_prime'))
      ->set('max_prime', (int) $form_state->getValue('max_prime'))
      ->set('length', (int) $form_state->getValue('length'))
      ->set('charset', $form_state->getValue('charset'))
      ->set('prefix', $form_state->getValue('prefix'))
      ->set('suffix', $form_state->getValue('suffix'))
      ->set('delimiter', $form_state->getValue('delimiter'))
      ->set('unique', $form_state->getValue('unique'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
