<?php

declare(strict_types=1);

namespace Drupal\rankcrew\Plugin\rest\resource;

use Drupal\rest\Attribute\RestResource;
use Drupal\rest\Plugin\ResourceBase;
use Drupal\rest\ResourceResponse;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Psr\Log\LoggerInterface;

/**
 * Represents rankcrew categories as resources.
 *
 * @RestResource(
 *   id = "rankcrew_categories",
 *   label = @Translation("Rankcrew Categories"),
 *   uri_paths = {
 *     "canonical" = "/api/rankcrew/categories"
 *   }
 * )
 */
final class RankcrewCategoriesResource extends ResourceBase
{

    public function __construct(
        array $configuration,
        $plugin_id,
        $plugin_definition,
        array $serializer_formats,
        LoggerInterface $logger
    ) {
        parent::__construct($configuration, $plugin_id, $plugin_definition, $serializer_formats, $logger);
    }

    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self
    {
        return new self(
            $configuration,
            $plugin_id,
            $plugin_definition,
            $container->getParameter('serializer.formats'),
            $container->get('logger.factory')->get('rest')
        );
    }

    /**
     * GET /api/rankcrew/categories?vocabulary_id=tags
     *
     * Returns a list of taxonomy terms, optionally filtered by vocabulary.
     */
    public function get()
    {
        try {
            // Get vocabulary ID from query parameter
            $vocabulary_id = \Drupal::request()->query->get('vocabulary_id');

            $query = \Drupal::entityQuery('taxonomy_term')
                ->accessCheck(TRUE);

            // Filter by vocabulary if provided
            if ($vocabulary_id) {
                $query->condition('vid', $vocabulary_id);
            }

            $tids = $query->execute();
            $terms = \Drupal::entityTypeManager()->getStorage('taxonomy_term')->loadMultiple($tids);

            $data = [];
            foreach ($terms as $term) {
                $data[] = [
                    'id' => (string) $term->id(),
                    'name' => $term->label(),
                    'slug' => $term->id(), // Using ID as slug for consistency/simplicity
                    'count' => 0, // Placeholder
                    'vocabulary' => $term->bundle(),
                ];
            }

            // Return as array
            return new ResourceResponse($data, 200);
        } catch (\Exception $e) {
            $this->logger->error('Error fetching categories: @message', ['@message' => $e->getMessage()]);
            return new ResourceResponse(['error' => 'Failed to fetch categories'], 500);
        }
    }

}
