<?php

declare(strict_types=1);

namespace Drupal\rankcrew\Plugin\rest\resource;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\File\FileSystemInterface;
use Drupal\file\Entity\File;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\rest\Attribute\RestResource;
use Drupal\rest\Plugin\ResourceBase;
use Drupal\rest\ResourceResponse;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Psr\Log\LoggerInterface;
use Drupal\Core\KeyValueStore\KeyValueFactoryInterface;
use Drupal\Core\KeyValueStore\KeyValueStoreInterface;

/**
 * Represents rankcrew records as resources, with base64 images.
 *
 * @RestResource(
 *   id = "rankcrew_rankcrew",
 *   label = @Translation("Rankcrew"),
 *   uri_paths = {
 *     "create" = "/api/rankcrew"
 *   }
 * )
 */
final class RankcrewResource extends ResourceBase
{

  /**
   * The key-value storage.
   */
  private readonly KeyValueStoreInterface $storage;

  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    array $serializer_formats,
    LoggerInterface $logger,
    KeyValueFactoryInterface $keyValueFactory
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $serializer_formats, $logger);
    $this->storage = $keyValueFactory->get('rankcrew_rankcrew');
  }

  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self
  {
    return new self(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->getParameter('serializer.formats'),
      $container->get('logger.factory')->get('rest'),
      $container->get('keyvalue')
    );
  }

  /**
   * POST /api/rankcrew
   *
   * Expects JSON with content_type + data, e.g:
   * {
   *   "content_type": "article",
   *   "data": {
   *     "en": {
   *       "title": "English Title",
   *       "body": "<p>Text</p>",
   *       "image": {
   *         "image_base64": "...",
   *         "mime_type": "image/jpeg",
   *         "field_image_name": "field_image"
   *       }
   *     },
   *     "fr": {...}
   *   }
   * }
   */
  public function post(Request $request)
  {
    $payload = json_decode($request->getContent(), TRUE);
    if (!is_array($payload) || empty($payload)) {
      return new JsonResponse(['error' => 'Invalid or empty JSON payload.'], 400);
    }

    if (empty($payload['content_type']) || empty($payload['data'])) {
      return new JsonResponse(['error' => 'Must specify content_type and data.'], 400);
    }

    $content_type = $payload['content_type'];
    $node_types = NodeType::loadMultiple();
    if (!isset($node_types[$content_type])) {
      return new JsonResponse([
        'error' => "Content type '$content_type' does not exist or is not available."
      ], 400);
    }

    $all_lang_data = $payload['data'];
    $category_id = $payload['category_id'] ?? null;
    $field_category_name = $payload['field_category_name'] ?? 'field_tags';
    $languages = array_keys($all_lang_data);

    // Debug: Log all received languages and their validity
    $manager = \Drupal::languageManager();
    $debug_info = [];
    foreach ($languages as $code) {
      $is_valid = $manager->getLanguage($code) ? 'YES' : 'NO';
      $debug_info[] = "$code (Valid: $is_valid)";
    }
    $this->logger->notice('Rankcrew Payload Languages: @info', ['@info' => implode(', ', $debug_info)]);

    // 1) Create base node from the first language
    $base_langcode = array_shift($languages);
    $base_data = $all_lang_data[$base_langcode];

    $this->logger->notice('Rankcrew Base Language: @lang', ['@lang' => $base_langcode]);

    try {
      $node = Node::create([
        'type' => $content_type,
        'langcode' => $base_langcode,
      ]);

      // Fill base fields
      $this->applyFields($node, $base_data, $category_id, $field_category_name);
      $node->save();
      $this->logger->notice('Rankcrew Base Node Created: NID @nid', ['@nid' => $node->id()]);

      // 2) Additional translations
      foreach ($languages as $langcode) {
        $this->logger->notice('Rankcrew Processing Translation: @lang', ['@lang' => $langcode]);

        if ($node->isTranslatable() && $manager->getLanguage($langcode)) {
          $lang_data = $all_lang_data[$langcode] ?? [];
          $translation = $node->addTranslation($langcode);
          $this->applyFields($translation, $lang_data, $category_id, $field_category_name);
          $translation->save();
          $this->logger->notice('Rankcrew Translation Added: @lang', ['@lang' => $langcode]);
        } else {
          $this->logger->warning('Rankcrew Translation Skipped: @lang. Translatable: @trans, Valid: @valid', [
            '@lang' => $langcode,
            '@trans' => $node->isTranslatable() ? 'YES' : 'NO',
            '@valid' => $manager->getLanguage($langcode) ? 'YES' : 'NO'
          ]);
        }
      }

      $response_data = [
        'nid' => $node->id(),
        'uuid' => $node->uuid(),
      ];
      return new ResourceResponse($response_data, 200);

    } catch (\Exception $e) {
      $this->logger->error('Rankcrew Error: @message', ['@message' => $e->getMessage()]);
      return new JsonResponse(['error' => 'Internal Server Error: ' . $e->getMessage()], 500);
    }
  }

  /**
   * Apply fields (title, body, image, category) to a node/translation.
   */
  private function applyFields(Node $node, array $data, ?string $category_id = null, string $field_category_name = 'field_tags'): void
  {
    // Title
    if (!empty($data['title'])) {
      $node->set('title', $data['title']);
    }

    // Body
    if (!empty($data['body'])) {
      $node->set('body', [
        'value' => $data['body'],
        'format' => 'full_html',
      ]);
    }

    // Category/Tags
    if ($category_id) {
      $tid = (int) $category_id;
      if ($tid > 0) {
        if ($field_category_name && $node->hasField($field_category_name)) {
          $node->set($field_category_name, ['target_id' => $tid]);
        } elseif ($field_category_name !== 'field_tags' && $node->hasField('field_tags')) {
          $node->set('field_tags', ['target_id' => $tid]);
        } elseif ($node->hasField('field_category')) {
          $node->set('field_category', ['target_id' => $tid]);
        }
      }
    }

    // Image
    if (!empty($data['image'])) {
      $image = $data['image'];
      if (!empty($image['image_base64']) && !empty($image['mime_type']) && !empty($image['field_image_name'])) {
        $file = $this->createFileFromBase64($image['image_base64'], $image['mime_type']);
        if ($file) {
          $field_name = $image['field_image_name'];
          // Attach to an image field on the node
          $node->set($field_name, [
            'target_id' => $file->id(),
            'alt' => $node->label(),
            'title' => $node->label(),
          ]);
        }
      }
    }
  }

  /**
   * Creates a file from base64 data using the File Repository service.
   *
   * @param string $base64
   *   Base64-encoded file data.
   * @param string $mime_type
   *   The mime type, e.g. "image/jpeg".
   *
   * @return \Drupal\file\FileInterface|false
   *   A managed File entity on success, or false on failure.
   */
  private function createFileFromBase64(string $base64, string $mime_type)
  {
    $decoded = base64_decode($base64);
    if (!$decoded) {
      return false;
    }

    // guess extension from mime
    $extension_map = [
      'image/jpeg' => 'jpg',
      'image/png' => 'png',
      'image/gif' => 'gif',
      'image/webp' => 'webp',
      'image/bmp' => 'bmp',
    ];
    $ext = $extension_map[$mime_type] ?? 'jpg';
    $filename = 'rankcrew_' . uniqid() . '.' . $ext;
    $fileUri = 'public://' . $filename;

    // Write data with the file.repository service
    $file_repository = \Drupal::service('file.repository');
    $file = $file_repository->writeData(
      $decoded,
      $fileUri,
      FileSystemInterface::EXISTS_RENAME
    );
    if (!$file) {
      return false;
    }

    // Force the mime type
    $file->setMimeType($mime_type);
    // Make it permanent
    $file->setPermanent();
    $file->save();

    return $file;
  }

}
