<?php

declare(strict_types=1);

namespace Drupal\rankcrew\Plugin\rest\resource;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\rest\Attribute\RestResource;
use Drupal\rest\Plugin\ResourceBase;
use Drupal\rest\ResourceResponse;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Psr\Log\LoggerInterface;

/**
 * Represents rankcrew vocabularies as resources.
 *
 * @RestResource(
 *   id = "rankcrew_vocabularies",
 *   label = @Translation("Rankcrew Vocabularies"),
 *   uri_paths = {
 *     "canonical" = "/api/rankcrew/vocabularies"
 *   }
 * )
 */
final class RankcrewVocabulariesResource extends ResourceBase
{
    /**
     * The entity type manager.
     *
     * @var \Drupal\Core\Entity\EntityTypeManagerInterface
     */
    protected $entityTypeManager;

    public function __construct(
        array $configuration,
        $plugin_id,
        $plugin_definition,
        array $serializer_formats,
        LoggerInterface $logger,
        EntityTypeManagerInterface $entity_type_manager
    ) {
        parent::__construct($configuration, $plugin_id, $plugin_definition, $serializer_formats, $logger);
        $this->entityTypeManager = $entity_type_manager;
    }

    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self
    {
        return new self(
            $configuration,
            $plugin_id,
            $plugin_definition,
            $container->getParameter('serializer.formats'),
            $container->get('logger.factory')->get('rest'),
            $container->get('entity_type.manager')
        );
    }

    /**
     * GET /api/rankcrew/vocabularies
     *
     * Returns a list of vocabularies.
     */
    public function get()
    {
        try {
            $vocabularies = $this->entityTypeManager
                ->getStorage('taxonomy_vocabulary')
                ->loadMultiple();

            $data = [];
            foreach ($vocabularies as $vocabulary) {
                $data[] = [
                    'id' => $vocabulary->id(),
                    'name' => $vocabulary->label(),
                    'machine_name' => $vocabulary->id(),
                    'description' => $vocabulary->getDescription(),
                ];
            }

            // Return as array
            return new ResourceResponse($data, 200);
        } catch (\Exception $e) {
            $this->logger->error('Error fetching vocabularies: @message', ['@message' => $e->getMessage()]);
            return new ResourceResponse(['error' => 'Failed to fetch vocabularies'], 500);
        }
    }

}
