<?php

namespace Drupal\ratatouille\Commands;

use Drupal\ratatouille\Service\RecipeFileGenerator;
use Drupal\ratatouille\Service\RecipeConfigurationHandler;
use Drupal\ratatouille\Service\RecipeContentHandler;
use Drupal\ratatouille\Service\RecipeDependencyResolver;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drush\Commands\DrushCommands;
use Symfony\Component\Console\Question\Question;
use Drupal\ratatouille\Exception\InvalidRecipeNameException;
use Drupal\ratatouille\Exception\RecipeGenerationException;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * A Drush commandfile to generate Drupal recipes.
 */
class RecipeGeneratorCommands extends DrushCommands {

  use StringTranslationTrait;

  /**
   * The recipe file generator.
   *
   * @var \Drupal\ratatouille\Service\RecipeFileGenerator
   */
  protected $fileGenerator;


  /**
   * The recipe configuration handler.
   *
   * @var \Drupal\ratatouille\Service\RecipeConfigurationHandler
   */
  protected $configHandler;


  /**
   * The recipe configuration handler.
   *
   * @var \Drupal\ratatouille\Service\RecipeContentHandler
   */
  protected $contentHandler;

  /**
   * The recipe dependency resolver.
   *
   * @var \Drupal\ratatouille\Service\RecipeDependencyResolver
   */
  protected $dependencyResolver;


  /**
   * An array of conflicting configurations.
   *
   * @var array
   */
  protected $conflictingConfigs;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * RecipeGeneratorCommands constructor.
   */
  public function __construct(RecipeFileGenerator $file_generator, RecipeConfigurationHandler $config_handler, RecipeContentHandler $content_handler, RecipeDependencyResolver $dependency_resolver, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct();
    $this->fileGenerator = $file_generator;
    $this->configHandler = $config_handler;
    $this->contentHandler = $content_handler;
    $this->dependencyResolver = $dependency_resolver;
    $this->entityTypeManager = $entity_type_manager;
    $this->conflictingConfigs = \Drupal::service('settings')->get('recipe_conflicting_configs', []);
  }

  /**
   * Generates a new recipe from existing configuration and content.
   *
   * @command ratatouille:cook
   * @aliases rtc
   * @option name The machine name for the recipe. Required for non-interactive mode.
   * @option label The human-readable label for the recipe.
   * @option description A short description for the recipe.
   * @option include-config A comma-separated list of configuration names or patterns to include.
   * @option include-content A comma-separated list of content entities to include, in entity_type:id format (e.g., node:1,user:2).
   * @option add-modules A comma-separated list of module names to add to the install list manually.
   * @option maintain-file-structure A boolean flag to maintain the original file structure. Defaults to false.
   * @usage rtc
   *   Generate a recipe using the interactive menu.
   * @usage rtc --name=my_recipe --label="My Recipe" --include-config="system.site,node.type.*" --include-content="node:1" --add-modules="views,token"
   *   Generate a recipe non-interactively.
   */
  public function generate($options = [
    'name' => NULL,
    'label' => NULL,
    'description' => NULL,
    'include-config' => NULL,
    'include-content' => NULL,
    'add-modules' => NULL,
    'maintain-file-structure' => FALSE,
  ]) {
    $this->io()->title($this->t('Drupal Recipe Generator'));

    // Non-interactive mode.
    if (!empty($options['name'])) {
      $name = $options['name'];
      $label = $options['label'] ?? $name;
      $description = $options['description'] ?? 'Generated recipe.';
      $maintain_file_structure = $options['maintain-file-structure'];
      $selected_config = [];
      $selected_content = [];
      $manual_modules = [];
      $composer_dependencies = []; // Not yet implemented via options.

      // Validate required options for non-interactive mode.
      if (empty($options['include-config']) && empty($options['include-content'])) {
        throw new \InvalidArgumentException('For non-interactive mode, you must provide at least an --include-config or --include-content option.');
      }

      // Parse --include-config option.
      if (!empty($options['include-config'])) {
        $config_patterns = array_map('trim', explode(',', $options['include-config']));
        $expanded_config = [];
        $all_system_configs = $this->configHandler->listAllConfigurations();

        foreach ($config_patterns as $pattern) {
          if (str_contains($pattern, '*')) {
            $matches = array_filter($all_system_configs, fn($config_name) => fnmatch($pattern, $config_name));
            if (empty($matches)) {
              $this->io()->warning($this->t('The pattern "@pattern" did not match any configurations.', ['@pattern' => $pattern]));
            }
            else {
              $expanded_config = array_merge($expanded_config, $matches);
            }
          }
          else {
            $expanded_config[] = $pattern;
          }
        }
        $selected_config = array_unique($expanded_config);
      }

      // Parse --include-content option.
      if (!empty($options['include-content'])) {
        if (!$this->contentHandler->isDefaultContentEnabled()) {
          $this->io()->error($this->t('The "default_content" module is required for the --include-content option. Please enable it first.'));
          return;
        }
        $content_items = array_map('trim', explode(',', $options['include-content']));
        foreach ($content_items as $item) {
          $parts = explode(':', $item);
          if (count($parts) !== 2) {
            $this->io()->warning($this->t('Invalid content format: "@item". Should be entity_type:id. Skipping.', ['@item' => $item]));
            continue;
          }
          list($entity_type_id, $entity_id) = $parts;

          try {
            $storage = $this->entityTypeManager->getStorage($entity_type_id);
            $entity = $storage->load($entity_id);
            if ($entity) {
              $selected_content[$entity_type_id][] = $entity->uuid();
            }
            else {
              $this->io()->warning($this->t('Could not load entity of type "@type" with ID "@id".', ['@type' => $entity_type_id, '@id' => $entity_id]));
            }
          }
          catch (\Exception $e) {
            $this->io()->warning($this->t('Error loading entity type "@type": @message', ['@type' => $entity_type_id, '@message' => $e->getMessage()]));
          }
        }
      }

      // Parse --add-modules option.
      if (!empty($options['add-modules'])) {
        $manual_modules = array_map('trim', explode(',', $options['add-modules']));
      }

      // Run the generation.
      try {
        $this->doGenerate($name, $label, $description, $selected_config, $selected_content, $manual_modules, $composer_dependencies, $maintain_file_structure);
      }
      catch (RecipeGenerationException $e) {
        $this->io()->error($e->getMessage());
      }
      return;
    }

    // Interactive mode.
    $name = $this->askMachineName();
    $label = $this->io()->ask($this->t('Recipe label'), $name);
    $description = $this->io()->ask($this->t('Recipe description'), 'Generated recipe.');
    $maintain_file_structure = $this->io()->confirm($this->t('Maintain original file structure for copied files? (e.g., public/2025/image.png instead of image.png)'));

    $selected_config = [];
    $selected_content = [];
    $manual_modules = [];
    $composer_dependencies = []; // Simple collection for now.

    while (TRUE) {
      $this->showSummary($selected_config, $selected_content, $manual_modules);

      $options = [
        'Add Configuration',
        'Add Content',
        'Add Module Dependencies',
        'Finish and Generate Recipe',
        'Cancel',
      ];
      $choice_key = $this->io()->choice('What would you like to do?', $options);
      $choice = $options[$choice_key];

      switch ($choice) {
        case 'Add Configuration':
          $this->promptForConfiguration($selected_config);
          break;

        case 'Add Content':
          $this->promptForContent($selected_content);
          break;

        case 'Add Module Dependencies':
          $this->promptForManualModules($manual_modules);
          break;

        case 'Finish and Generate Recipe':
          try {
            $this->doGenerate($name, $label, $description, $selected_config, $selected_content, $manual_modules, $composer_dependencies, $maintain_file_structure);
          }
          catch (RecipeGenerationException $e) {
            $this->io()->error($e->getMessage());
          }
          return;

        case 'Cancel':
          $this->io()->warning($this->t('Operation cancelled.'));
          return;
      }
    }
  }

  /**
   * Executes the final recipe generation.
   */
  protected function doGenerate($name, $label, $description, $selected_config, $selected_content, $manual_modules, $composer_dependencies, bool $maintain_file_structure) {
    $this->io()->section($this->t('Generating recipe "@name"...', ['@name' => $name]));

    $recipe_path = $this->fileGenerator->createRecipeDirectory($name);
    if (!$recipe_path) {
      return; // Should already be confirmed by user.
    }

    list($module_deps, $config_deps, $content_deps, $file_deps) = $this->dependencyResolver->resolve($selected_config, $selected_content);

    // Add manually specified modules.
    $final_module_deps = array_unique(array_merge($module_deps, array_map(fn($m) => 'drupal:' . $m, $manual_modules)));

    $handled_configs = $this->handleConflictingConfigurations($config_deps);

    $configs_to_write = $this->configHandler->getConfigurationForItems($handled_configs['import']);
    $this->fileGenerator->writeConfigurationFiles($recipe_path, $configs_to_write);
    $this->io()->writeln(sprintf('Exported %d configuration items.', count($configs_to_write)));

    $content_files = $this->processContent($recipe_path, $content_deps, $name, $maintain_file_structure);
    $this->io()->writeln(sprintf('Exported %d content entities.', count($content_files)));

    // Copy files.
    $copied_files = $this->fileGenerator->copyFiles($recipe_path, $file_deps, $maintain_file_structure);
    $this->io()->writeln(sprintf('Copied %d files.', count($copied_files)));

    $recipe_data = $this->buildRecipeData($label, $description, $final_module_deps, $handled_configs, $content_files);
    $this->fileGenerator->writeRecipeYaml($recipe_path, $recipe_data);

    $this->fileGenerator->writeComposerJson($recipe_path, $name, $composer_dependencies);

    $this->io()->success($this->t('Recipe "@name" generated successfully at @path', ['@name' => $name, '@path' => $recipe_path]));
  }

  private function buildRecipeData(string $label, string $description, array $module_deps, array $handled_configs, array $content_files): array {
    $recipe_data = [
        'name' => $label,
        'description' => $description,
        'type' => 'Site', // Or some other default
        'install' => array_values(array_unique(array_map(fn($dep) => str_replace('drupal:', '', $dep), $module_deps))),
        'config' => [
            'import' => [],
            'actions' => [],
        ],
    ];

    if (!empty($content_files)) {
      $recipe_data['content'] = $content_files;
    }

    // Populate config:import.
    foreach ($handled_configs['import'] as $config_name) {
        $provider = $this->configHandler->getProvider($config_name);
        $key = $provider ?? 'core';

        if (!isset($recipe_data['config']['import'][$key])) {
            $recipe_data['config']['import'][$key] = [];
        }
        $recipe_data['config']['import'][$key][] = $config_name;
    }

    // Populate config:actions with simple_config_update.
    foreach ($handled_configs['actions'] as $config_name => $config_data) {
      // Remove UUID from action configs as it's not needed and can cause issues.
      if (isset($config_data['uuid'])) {
        unset($config_data['uuid']);
      }
      // Remove _core hash as it's not needed for actions.
      if (isset($config_data['_core'])) {
        unset($config_data['_core']);
      }
      $recipe_data['config']['actions'][$config_name]['simple_config_update'] = $config_data;
    }

    // Handle user roles actions (existing logic, but now applied to imported configs).
    foreach ($handled_configs['import'] as $config_name) {
      if (str_starts_with($config_name, 'user.role.')) {
          $config_data = $this->configHandler->getConfigurationForItems([$config_name])[$config_name] ?? [];
          if (!empty($config_data['permissions'])) {
              $recipe_data['config']['actions'][$config_name]['grantPermissions'] = $config_data['permissions'];
          }
      }
    }

    // Sort for consistency.
    ksort($recipe_data['config']['import']);
    foreach ($recipe_data['config']['import'] as &$items) {
        sort($items);
    }

    if (empty($recipe_data['config']['actions'])) {
        unset($recipe_data['config']['actions']);
    }

    return $recipe_data;
  }

  private function promptForConfiguration(&$selected_config) {
    $config_types = $this->configHandler->listAvailableConfigurationTypes();
    $config_types_options = array_combine($config_types, $config_types);
    $config_types_options['all'] = $this->t('All types');
    $selected_types = $this->io()->choice($this->t('Select configuration types to export'), $config_types_options, NULL, TRUE);

    $options_to_select = [];
    if (in_array('all', $selected_types)) {
      $options_to_select = $this->configHandler->listConfigurationItemsByTypes($config_types);
    } else if (!empty($selected_types)) {
      $options_to_select = $this->configHandler->listConfigurationItemsByTypes($selected_types);
    }

    if (!empty($options_to_select)) {
      $selection_options = array_combine($options_to_select, $options_to_select);
      $selection_options['all'] = $this->t('All items');
      $selection_options['pattern'] = $this->t('Select by pattern (regex)');
      $selection_options['back'] = $this->t('Go back');

      $new_selection_choice = $this->io()->choice($this->t('Select configs to add'), $selection_options, NULL, FALSE);

      if ($new_selection_choice === 'all') {
        $new_selection = $options_to_select;
      } elseif ($new_selection_choice === 'pattern') {
        $pattern = $this->io()->ask($this->t('Enter regex pattern to filter config names'));
        $new_selection = array_filter($options_to_select, fn($item) => preg_match('/' . $pattern . '/', $item));
      } elseif ($new_selection_choice === 'back') {
        return; // Go back to the previous menu.
      } else {
        $new_selection = [$new_selection_choice];
      }

      // Resolve dependencies for the newly selected configurations.
      list($resolved_config_deps, $resolved_module_deps) = $this->dependencyResolver->resolveConfigDependencies($new_selection);

      // Merge resolved dependencies with existing selected configurations.
      $selected_config = array_unique(array_merge($selected_config, $resolved_config_deps));

      // Note: Module dependencies are not stored in $selected_config, but they are resolved
      // and will be picked up by the main resolve() call in doGenerate().
      // If we wanted to show them in the summary, we'd need to store them here.
    }
  }

  private function promptForContent(&$selected_content) {
    if (!$this->contentHandler->isDefaultContentEnabled()) {
      $this->io()->error($this->t('The "default_content" module is required. Please enable it first.'));
      return;
    }

    $type_options = $this->contentHandler->listExportableEntityTypes();
    if (empty($type_options)) {
      $this->io()->warning($this->t('No exportable content entity types found.'));
      return;
    }

        $choice_options = [];
    foreach ($type_options as $id => $label) {
      $choice_options[$id] = $label;
    }
    $choice_options['all'] = (string) $this->t('All types');

    // Ensure display labels are unique for io()->choice().
    $display_labels_seen = [];
    $final_choice_options = [];
    foreach ($choice_options as $id => $label) {
      $display_label = (string) $label; // Ensure it's a string.
      if (isset($display_labels_seen[$display_label])) {
        // Append a unique identifier if the display label is duplicated.
        $display_label .= ' (' . $id . ')';
      }
      $final_choice_options[$id] = $display_label;
      $display_labels_seen[$display_label] = TRUE;
    }

    $selected_entity_type_ids = $this->io()->choice($this->t('Select entity types to export'), $final_choice_options, NULL, TRUE);

    $entity_types_to_process = [];
    if (in_array('all', $selected_entity_type_ids)) {
      $entity_types_to_process = array_keys($type_options);
    } else {
      $entity_types_to_process = $selected_entity_type_ids;
    }

    foreach ($entity_types_to_process as $type_id) {
      $label = $type_options[$type_id] ?? $type_id;
      $autocomplete_options = $this->contentHandler->getEntityAutocompleteOptions($type_id);

      if (empty($autocomplete_options)) {
        $this->io()->note($this->t('No content found for type "@type".', ['@type' => $label]));
        continue;
      }

      $selection_method_options = [
        'autocomplete' => $this->t('Search by name (autocomplete)'),
        'list' => $this->t('List all items (use with caution for large numbers)'),
        'pattern' => $this->t('Select by pattern (regex)'),
        'skip' => $this->t('Skip this type'),
      ];

      $method_choice = $this->io()->choice($this->t('How would you like to select content for type "@type"?', ['@type' => $label]), $selection_method_options, 'autocomplete');

      switch ($method_choice) {
        case 'autocomplete':
          while (TRUE) {
            $question = new Question($this->t('Search for content of type "@type" to add (leave empty to finish)', ['@type' => $label]));
            $question->setAutocompleterValues(array_values($autocomplete_options));
            $answer = $this->io()->askQuestion($question);

            if (empty($answer)) {
              break;
            }

            $uuid = array_search($answer, $autocomplete_options);
            if ($uuid) {
              $selected_content[$type_id][] = $uuid;
              $selected_content[$type_id] = array_unique($selected_content[$type_id]);
              $this->io()->success(sprintf('Added "%%s".', $answer));
            } else {
              $this->io()->warning(sprintf('Could not find content "%%s".', $answer));
            }
          }
          break;

        case 'list':
          if (count($autocomplete_options) > 100) {
            if (!$this->io()->confirm($this->t('Listing %%count items might be slow. Are you sure?', ['%%count' => count($autocomplete_options)]))) {
              break; // Go back to method choice.
            }
          }
          $list_options = array_combine(array_keys($autocomplete_options), array_values($autocomplete_options));
          $list_options['all'] = $this->t('All items of this type'); // Add 'all' option
          $selected_uuids_from_list_choice = $this->io()->choice($this->t('Select content for type "@type"', ['@type' => $label]), $list_options, NULL, TRUE);

          if (in_array('all', $selected_uuids_from_list_choice)) {
            foreach ($autocomplete_options as $uuid => $item_label) {
              $selected_content[$type_id][] = $uuid;
            }
            $selected_content[$type_id] = array_unique($selected_content[$type_id]);
            $this->io()->success(sprintf('Added all content for type "%%s".', $label));
          } else {
            foreach ($selected_uuids_from_list_choice as $uuid) {
              $selected_content[$type_id][] = $uuid;
            }
            $selected_content[$type_id] = array_unique($selected_content[$type_id]);
            $this->io()->success(sprintf('Added %%d items for type "%%s".', count($selected_uuids_from_list_choice), $label));
          }
          break;

        case 'pattern':
          $pattern = $this->io()->ask($this->t('Enter regex pattern to filter content labels'));
          $filtered_uuids = array_filter(array_keys($autocomplete_options), fn($uuid) => preg_match('/' . $pattern . '/', $autocomplete_options[$uuid]));
          foreach ($filtered_uuids as $uuid) {
            $selected_content[$type_id][] = $uuid;
          }
          $selected_content[$type_id] = array_unique($selected_content[$type_id]);
          $this->io()->success(sprintf('Added content matching pattern for type "%%s".', $label));
          break;

        case 'skip':
          $this->io()->note($this->t('Skipped content selection for type "@type".', ['@type' => $label]));
          break;
      }
    }
  }

  private function processContent(string $recipe_path, array $content_deps, string $recipe_name, bool $maintain_file_structure): array {
    $exported_file_paths = [];

    foreach ($content_deps as $entity_type_id => $uuids) {
      foreach ($uuids as $uuid) {
        $exported_yaml = $this->contentHandler->exportEntity($entity_type_id, $uuid, $recipe_name, $maintain_file_structure);
        if ($exported_yaml) {
          $file_path = $this->fileGenerator->writeContentFile($recipe_path, $entity_type_id, $uuid, $exported_yaml);
          if ($file_path) {
            $exported_file_paths[] = $file_path;
          }
        }
      }
    }

    return $exported_file_paths;
  }

  private function askMachineName(): string {
    return $this->io()->ask($this->t('Recipe machine name'), NULL, function ($name) {
      if (empty($name) || !preg_match('/^[a-z0-9_]+$/', $name)) {
        throw new InvalidRecipeNameException('Machine name must be lowercase and contain only letters, numbers, and underscores.');
      }
      $path = DRUPAL_ROOT . '/recipes/custom/' . $name;
      if (is_dir($path) && !$this->io()->confirm($this->t('Recipe directory "@path" already exists. Overwrite?', ['@path' => $path]))) {
        throw new InvalidRecipeNameException('Aborted by user.');
      }
      return $name;
    });
  }

  private function showSummary($selected_config, $selected_content, $manual_modules) {
    $this->io()->section('Current Recipe');
    $this->io()->writeln(sprintf('Configuration items: %d', count($selected_config)));
    $content_count = array_reduce($selected_content, fn($c, $i) => $c + count($i), 0);
    $this->io()->writeln(sprintf('Content items: %d', $content_count));
    $this->io()->writeln(sprintf('Manual modules: %d', count($manual_modules)));
    $this->io()->newLine();
  }

  private function promptForManualModules(array &$manual_modules) {
    $new_modules_str = $this->io()->ask($this->t('Enter a comma-separated list of modules to add'));
    if (!empty($new_modules_str)) {
      $new_modules = array_map('trim', explode(',', $new_modules_str));
      $manual_modules = array_unique(array_merge($manual_modules, $new_modules));
      $this->io()->success($this->t('Added @count modules.', ['@count' => count($new_modules)]));
    }
  }

  private function handleConflictingConfigurations(array $config_deps): array {
    $processed_configs = [
      'import' => [],
      'actions' => [],
      'omit' => [],
    ];

    // Map action codes from YAML to internal strings and translated labels.
    $action_map = [
      1 => 'import',
      2 => 'action',
      3 => 'omit',
    ];
    $action_labels = [
      'import' => $this->t('Add to config:import (default)'),
      'action' => $this->t('Add to config:actions (simple_config_update)'),
      'omit' => $this->t('Omit from recipe'),
    ];

    $this->io()->section($this->t('Reviewing Conflicting Configurations'));

    foreach ($config_deps as $config_name) {
      $matched_action = NULL;
      $matched_pattern = NULL;

      // Find the first matching pattern in the conflicting configs file.
      if (is_array($this->conflictingConfigs)) {
        foreach ($this->conflictingConfigs as $pattern => $action_code) {
          if (fnmatch($pattern, $config_name)) {
            $matched_action = $action_code;
            $matched_pattern = $pattern;
            break;
          }
        }
      }

      // If no pattern matched, default to importing.
      if ($matched_action === NULL) {
        $processed_configs['import'][] = $config_name;
        continue;
      }

      $choice = NULL;
      // If the action code is a valid automatic action (1, 2, or 3).
      if (isset($action_map[$matched_action])) {
        $choice = $action_map[$matched_action];
        $this->io()->writeln($this->t('Configuration <info>@config_name</info> matches pattern "<info>@pattern</info>". Applying pre-set action: <info>@action</info>.', [
          '@config_name' => $config_name,
          '@pattern' => $matched_pattern,
          '@action' => $choice,
        ]));
      }
      // Otherwise, the action is 0 (or invalid), so we ask the user.
      else {
        $this->io()->writeln($this->t('Conflicting configuration detected: <info>@config_name</info>', ['@config_name' => $config_name]));
        $choice = $this->io()->choice(
          $this->t('How would you like to handle this configuration?'),
          $action_labels,
          'import'
        );
      }

      // Process the choice (either automatic or from user input).
      switch ($choice) {
        case 'import':
          $processed_configs['import'][] = $config_name;
          break;
        case 'action':
          $processed_configs['actions'][$config_name] = $this->configHandler->getConfigurationForItems([$config_name])[$config_name];
          break;
        case 'omit':
          $processed_configs['omit'][] = $config_name;
          break;
      }
    }
    return $processed_configs;
  }

  

}
