<?php

namespace Drupal\ratatouille\Service;

use Drupal\Core\Config\ConfigManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Handles fetching and processing of configuration.
 */
class RecipeConfigurationHandler {

  /**
   * The config manager service.
   *
   * @var \Drupal\Core\Config\ConfigManagerInterface
   */
  protected $configManager;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * RecipeConfigurationHandler constructor.
   */
  public function __construct(ConfigManagerInterface $config_manager, EntityTypeManagerInterface $entity_type_manager) {
    $this->configManager = $config_manager;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Gets a list of all available configuration types (prefixes).
   *
   * @return string[]
   *   A sorted array of configuration type prefixes.
   */
  public function listAvailableConfigurationTypes(): array {
    $config_list = $this->configManager->getConfigFactory()->listAll();
    $config_types = array_unique(array_map(fn($cn) => explode('.', $cn)[0], $config_list));
    sort($config_types);
    return $config_types;
  }

  /**
   * Gets a list of configuration items for the given types.
   *
   * @param string[] $types
   *   An array of configuration type prefixes.
   *
   * @return string[]
   *   An array of configuration item names.
   */
  public function listConfigurationItemsByTypes(array $types): array {
    $config_list = $this->configManager->getConfigFactory()->listAll();
    $options = [];
    foreach ($types as $type) {
        $options += array_filter($config_list, fn($cn) => str_starts_with($cn, $type . '.'));
    }
    return $options;
  }

  /**
   * Gets the raw configuration data for a list of items.
   *
   * @param string[] $selected_config
   *   The list of configuration item names.
   *
   * @return array
   *   An associative array of config data, keyed by config name.
   */
  public function getConfigurationForItems(array $selected_config): array {
    $all_configs_to_write = [];
    foreach ($selected_config as $config_name) {
      $config = $this->configManager->getConfigFactory()->get($config_name);
      $all_configs_to_write[$config_name] = $config->get();
    }
    return $all_configs_to_write;
  }

  public function getProvider(string $config_name): ?string {
    $config_data = NULL;

    // Handle core.base_field_override.* by reading its internal entity_type.
    if (str_starts_with($config_name, 'core.base_field_override.')) {
      $config_data = $this->configManager->getConfigFactory()->get($config_name)->get();
      if (isset($config_data['entity_type'])) {
        try {
          $entity_type = $this->entityTypeManager->getDefinition($config_data['entity_type']);
          return $entity_type->getProvider();
        }
        catch (\Exception $e) {
          // Fallback.
        }
      }
    }

    // Handle entity form/view displays by reading their internal targetEntityType.
    if (str_starts_with($config_name, 'core.entity_form_display.') || str_starts_with($config_name, 'core.entity_view_display.')) {
      $config_data = $this->configManager->getConfigFactory()->get($config_name)->get();
      if (isset($config_data['targetEntityType'])) {
        try {
          $entity_type = $this->entityTypeManager->getDefinition($config_data['targetEntityType']);
          return $entity_type->getProvider();
        }
        catch (\Exception $e) {
          // Fallback.
        }
      }
    }

    // Handle entity form/view modes by parsing the entity type from the name.
    if (str_starts_with($config_name, 'core.entity_view_mode.') || str_starts_with($config_name, 'core.entity_form_mode.')) {
      $parts = explode('.', $config_name);
      if (isset($parts[2])) {
        try {
          $entity_type = $this->entityTypeManager->getDefinition($parts[2]);
          return $entity_type->getProvider();
        }
        catch (\Exception $e) {
          // Fallback.
        }
      }
    }

    // Handle field and field storage configurations.
    if (str_starts_with($config_name, 'field.field.') || str_starts_with($config_name, 'field.storage.')) {
      $config_data = $this->configManager->getConfigFactory()->get($config_name)->get();
      if (isset($config_data['entity_type'])) {
        try {
          $entity_type = $this->entityTypeManager->getDefinition($config_data['entity_type']);
          return $entity_type->getProvider();
        }
        catch (\Exception $e) {
          // Fallback.
        }
      }
    }

    // Try to find a provider based on the config entity type.
    $entity_type_id = $this->configManager->getEntityTypeIdByName($config_name);
    if ($entity_type_id) {
      try {
        $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
        return $entity_type->getProvider();
      }
      catch (\Exception $e) {
        // Ignore and fallback.
      }
    }

    // Final fallback for simple config (e.g., system.site).
    $parts = explode('.', $config_name);
    if (count($parts) > 1) {
      return $parts[0];
    }
    return NULL;
  }

  /**
   * Gets a list of all configuration items available in the system.
   *
   * @return string[]
   *   An array of all configuration item names.
   */
  public function listAllConfigurations(): array {
    return $this->configManager->getConfigFactory()->listAll();
  }

}
