<?php

namespace Drupal\ratatouille\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Collects configuration dependencies related to content entities.
 */
class ContentConfigDependencyCollector {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a new ContentConfigDependencyCollector object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Adds bundle-specific configuration to the dependency list.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   * @param array $config_deps
   *   The array of collected configuration dependencies, passed by reference.
   */
  public function addBundleConfig(string $entity_type_id, string $bundle_id, array &$config_deps) {
    $bundle_config_map = [
      'node' => 'node.type.',
      'block_content' => 'block_content.type.',
      'taxonomy_term' => 'taxonomy.vocabulary.',
      // Add other entity types here if needed.
    ];

    if (isset($bundle_config_map[$entity_type_id])) {
      $config_deps[] = $bundle_config_map[$entity_type_id] . $bundle_id;
    }
  }

  /**
   * Adds field and field storage configuration for a bundle.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   * @param array $config_deps
   *   The array of collected configuration dependencies, passed by reference.
   */
  public function addFieldConfig(string $entity_type_id, string $bundle_id, array &$config_deps) {
    // Add form display config.
    $form_display_config_name = 'core.entity_form_display.' . $entity_type_id . '.' . $bundle_id . '.default';
    if ($this->entityTypeManager->getStorage('entity_form_display')->load($entity_type_id . '.' . $bundle_id . '.default')) {
      $config_deps[] = $form_display_config_name;
    }

    // Add view display config.
    $view_display_config_name = 'core.entity_view_display.' . $entity_type_id . '.' . $bundle_id . '.default';
    if ($this->entityTypeManager->getStorage('entity_view_display')->load($entity_type_id . '.' . $bundle_id . '.default')) {
      $config_deps[] = $view_display_config_name;
    }

    // Add field storage and field config for each component.
    $field_definitions = $this->entityTypeManager->getStorage('entity_form_display')->load($entity_type_id . '.' . $bundle_id . '.default');
    if (!$field_definitions) {
      return;
    }

    foreach ($field_definitions->getComponents() as $field_name => $options) {
      $field_storage = $this->entityTypeManager->getStorage('field_storage_config')->load($entity_type_id . '.' . $field_name);
      if ($field_storage) {
        $config_deps[] = $field_storage->getConfigDependencyName();
      }

      $field = $this->entityTypeManager->getStorage('field_config')->load($entity_type_id . '.' . $bundle_id . '.' . $field_name);
      if ($field) {
        $config_deps[] = $field->getConfigDependencyName();
      }
    }
  }

}
